﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/util/util_Optional.h>
#include <nn/util/util_TinyMt.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>

#include <nn/os/os_Mutex.h>
#include <nn/os/os_Tick.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "nim_DebugUtil.h"

namespace nn { namespace nim { namespace srv {
namespace
{

class NonRecursiveMutex : public os::Mutex
{
public:
    NonRecursiveMutex() NN_NOEXCEPT : os::Mutex(false) {}
};

const int UrlLength = 256;
char      s_ErrorUrl[UrlLength];
int       s_ErrorUrlLength;
util::TinyMt s_Mt;
uint32_t  s_ErrorRate = 0; // 0 - 10000 を指定。一万分率。
int32_t   s_ErrorResult = 0;

bool      s_SimulationConfigurationLoaded = false;
bool      s_SimulationEnabled = false;
NonRecursiveMutex s_SimulationLock;

// あらかじめ s_SimulationLock を取得しておくこと
void LoadErrorSimulationConfiguration() NN_NOEXCEPT
{
    if (!s_SimulationConfigurationLoaded)
    {
        s_SimulationConfigurationLoaded = true;
        s_SimulationEnabled = false;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // fwdbg からの読み込み
        bool enabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "nim.errorsimulate", "enable_error_simulate");
        if (size != sizeof(enabled) || !enabled)
        {
            return;
        }
        settings::fwdbg::GetSettingsItemValue(&s_ErrorUrl, sizeof(s_ErrorUrl), "nim.errorsimulate", "error_url_prefix");
        s_ErrorUrl[UrlLength - 1] = '\0';
        s_ErrorUrlLength = util::Strnlen(s_ErrorUrl, UrlLength);

        size = settings::fwdbg::GetSettingsItemValue(&s_ErrorRate, sizeof(s_ErrorRate), "nim.errorsimulate", "error_rate");
        if (size != sizeof(s_ErrorRate))
        {
            return;
        }
        size = settings::fwdbg::GetSettingsItemValue(&s_ErrorResult, sizeof(s_ErrorResult), "nim.errorsimulate", "error_result");
        if (size != sizeof(s_ErrorResult))
        {
            return;
        }

        s_Mt.Initialize(static_cast<Bit32>(os::GetSystemTick().GetInt64Value()));
        s_SimulationEnabled = true;
#endif
    }
}

}

    Result GetEciDeviceCertificateDebug(EciDeviceCertificate* outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(fs::MountSdCardForDebug("sdcard"));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount("sdcard"); };

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, "sdcard:/eci_device.cert", fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_RESULT_DO(fs::ReadFile(file, 0, outValue, sizeof(*outValue)));

        NN_RESULT_SUCCESS;
    }

    bool IsErrorSimulationEnabled() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(s_SimulationLock);
        LoadErrorSimulationConfiguration();
        return s_SimulationEnabled;
    }

    Result DoErrorSimulation(const char* url) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(s_SimulationLock);
        LoadErrorSimulationConfiguration();
        if (s_SimulationEnabled)
        {
            // 有効なのでシミュレートを行う
            // URL の前方一致と、発生率を判定してシミュレートするかどうかを決める
            if (util::Strncmp(url, s_ErrorUrl, s_ErrorUrlLength) == 0 && s_Mt.GenerateRandomN(10000) < s_ErrorRate)
            {
                NN_RESULT_THROW(result::detail::ConstructResult(s_ErrorResult));
            }
        }
        NN_RESULT_SUCCESS;
    }
    void ClearErrorSimulationCache() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(s_SimulationLock);
        s_SimulationConfigurationLoaded = false;
    }

    Result GetLocalCommunicationErrorResultForDebug() NN_NOEXCEPT
    {
        util::optional<Result> s_Result;
        if (!s_Result)
        {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            uint32_t innerValue;
            auto size = settings::fwdbg::GetSettingsItemValue(&innerValue, sizeof(innerValue), "nim.errorsimulate", "error_localcommunication_result");
            if (size != sizeof(innerValue))
            {
                // エラー無状態とし、成功を返す
                s_Result.emplace();
                *s_Result = ResultSuccess();
            }
            else
            {
                s_Result = result::detail::ConstructResult(innerValue);
            }
#else
            s_Result.emplace();
            *s_Result = ResultSuccess();
#endif
        }

        return *s_Result;
    }
}}}
