﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/nim/srv/detail/nim_BufferAllocator.h>


namespace nn { namespace nim { namespace srv {
    namespace detail {
        //
        // BufferInfo
        // Public
        //
        //
        BufferInfo::BufferInfo(size_t memorySize, BufferFlagSet flag, BufferFlagSet groupFlagSet) NN_NOEXCEPT
            : m_MemorySize(memorySize), m_Flag(flag), m_Group(groupFlagSet)
        {
            NN_ABORT_UNLESS(m_MemorySize > 0);
            NN_ABORT_UNLESS_EQUAL(0u, m_MemorySize % BufferAlign);
            NN_ABORT_UNLESS_EQUAL(1, m_Flag.CountPopulation());
            NN_ABORT_UNLESS_EQUAL(1, (m_Flag & m_Group).CountPopulation());
        }

        size_t BufferInfo::GetSize() const NN_NOEXCEPT
        {
            return m_MemorySize;
        }

        BufferFlagSet BufferInfo::GetFlag() const NN_NOEXCEPT
        {
            return m_Flag;
        }

        BufferFlagSet BufferInfo::GetGroup() const NN_NOEXCEPT
        {
            return m_Group;
        }


        //
        // BufferAllocator
        // Public
        //
        BufferAllocator::BufferAllocator(void* buffer, size_t bufferSize, const BufferInfo** pList, int numInfo) NN_NOEXCEPT
            : m_pInfoList(pList), m_NumInfo(numInfo), m_Mutex(false)
        {
            NN_ABORT_UNLESS_NOT_NULL(buffer);
            NN_ABORT_UNLESS_EQUAL(0u, bufferSize % BufferAlign);
            NN_ABORT_UNLESS(bufferSize > 0);
            NN_ABORT_UNLESS_NOT_NULL(pList);
            NN_ABORT_UNLESS(numInfo > 0);

            size_t maxRequiredSize = 0;
            for (int i = 0; i < m_NumInfo; i++)
            {
                maxRequiredSize = std::max(maxRequiredSize, CalculateGroupMemorySize(pList[i]->GetGroup()));
            }
            m_Handle = lmem::CreateExpHeap(buffer, bufferSize, lmem::CreationOption_ThreadSafe);
            size_t allocatableSize = lmem::GetExpHeapAllocatableSize(m_Handle, BufferAlign);
            NN_ABORT_UNLESS(allocatableSize >= maxRequiredSize);

            m_AllocatedFlagSet.Reset();
        }

        BufferAllocator::~BufferAllocator() NN_NOEXCEPT
        {
            lmem::DestroyExpHeap(m_Handle);
        }

        void* BufferAllocator::Allocate(size_t* outSize, BufferFlagSet flag) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(outSize);
            std::lock_guard<os::Mutex> lock(m_Mutex);
            auto pInfo = std::find_if(m_pInfoList, m_pInfoList + m_NumInfo, [&flag](const BufferInfo* iterator) NN_NOEXCEPT { return flag == iterator->GetFlag(); });
            NN_ABORT_UNLESS(pInfo != m_pInfoList + m_NumInfo);
            NN_ABORT_UNLESS_EQUAL(0, (m_AllocatedFlagSet & ~((*pInfo)->GetGroup())).CountPopulation());

            auto ptr = lmem::AllocateFromExpHeap(m_Handle, (*pInfo)->GetSize(), BufferAlign);
            NN_ABORT_UNLESS(ptr);
            *outSize = (*pInfo)->GetSize();

            m_AllocatedFlagSet |= (*pInfo)->GetFlag();

            return ptr;
        }

        void BufferAllocator::Free(void* buffer, BufferFlagSet flag) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> lock(m_Mutex);
            NN_ABORT_UNLESS_NOT_NULL(buffer);
            auto pInfo = std::find_if(m_pInfoList, m_pInfoList + m_NumInfo, [&flag](const BufferInfo* iterator) NN_NOEXCEPT { return flag == iterator->GetFlag(); });
            NN_ABORT_UNLESS(pInfo != m_pInfoList + m_NumInfo);
            NN_ABORT_UNLESS_EQUAL(1, ((*pInfo)->GetFlag() & m_AllocatedFlagSet).CountPopulation());

            lmem::FreeToExpHeap(m_Handle, buffer);
            m_AllocatedFlagSet &= ~((*pInfo)->GetFlag());
        }

        //
        // BufferAllocator
        // Private
        //
        size_t BufferAllocator::CalculateGroupMemorySize(BufferFlagSet group) const NN_NOEXCEPT
        {
            size_t size = 0;
            BufferFlagSet remainedFlagSet = group;
            for (int i = 0; i < m_NumInfo; i++)
            {
                if ((m_pInfoList[i]->GetFlag() & remainedFlagSet).CountPopulation() == 1)
                {
                    size += m_pInfoList[i]->GetSize();
                    remainedFlagSet &= ~m_pInfoList[i]->GetFlag();
                }
            }
            NN_ABORT_UNLESS_EQUAL(0, remainedFlagSet.CountPopulation());
            return size;
        }

    } // namespace detail
}}} // namespace nn::nim::srv
