﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/crypto.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "nim_Json.h"
#include "nim_StringUtil.h"
#include "nim_AsyncSystemUpdateMetaImpl.h"

namespace nn { namespace nim { namespace srv {

    namespace
    {
        static const int SystemUpdateMetaThreadCount = 1;
        static const size_t StackSize = 16 * 1024;

        os::ThreadType g_SystemUpdateMetaTaskThreadList[SystemUpdateMetaThreadCount];
        NN_OS_ALIGNAS_THREAD_STACK char g_SystemUpdateMetaTaskStack[StackSize * SystemUpdateMetaThreadCount];

        ThreadAllocator g_SystemUpdateMetaTaskThreadAllocator(g_SystemUpdateMetaTaskThreadList, SystemUpdateMetaThreadCount, NN_SYSTEM_THREAD_PRIORITY(nim, SystemUpdateMetaTask), g_SystemUpdateMetaTaskStack, sizeof(g_SystemUpdateMetaTaskStack), StackSize, NN_SYSTEM_THREAD_NAME(nim, SystemUpdateMetaTask));

        Result ParseSystemUpdateMetaKeyResponse(ncm::ContentMetaKey* outValue, char* readBuffer) NN_NOEXCEPT
        {
            nne::rapidjson::Document document;
            NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseSystemUpdateMetaParseError());

            auto metaArray = document.FindMember("system_update_metas");
            NN_RESULT_THROW_UNLESS(metaArray != document.MemberEnd() && metaArray->value.IsArray() && metaArray->value.Size() > 0, ResultUnexpectedResponseSystemUpdateMetaRootNotFound());
            NN_RESULT_THROW_UNLESS(metaArray->value.Size() > 0, ResultSystemUpdateMetaNotPublished());
            auto& metaObj = (metaArray->value)[0];

            auto titleIdObj = metaObj.FindMember("title_id");
            auto versionObj = metaObj.FindMember("title_version");
            NN_RESULT_THROW_UNLESS(titleIdObj != metaObj.MemberEnd() && titleIdObj->value.IsString(), ResultUnexpectedResponseSystemUpdateMetaTitleIdNotFound());
            NN_RESULT_THROW_UNLESS(versionObj != metaObj.MemberEnd() && versionObj->value.IsInt(), ResultUnexpectedResponseSystemUpdateMetaVersionNotFound());

            Bit64 contentMetaId = NN_NIM_STR_TO_ULL(titleIdObj->value.GetString(), nullptr, 16);
            uint32_t version = static_cast<uint32_t>(versionObj->value.GetInt());

            *outValue = ncm::ContentMetaKey::Make(contentMetaId, version, ncm::ContentMetaType::SystemUpdate);
            NN_RESULT_SUCCESS;
        }
    }

    AsyncSystemUpdateMetaImpl::AsyncSystemUpdateMetaImpl() NN_NOEXCEPT : AsyncValueImpl<AsyncSystemUpdateMetaImpl, ncm::ContentMetaKey>(this, &g_SystemUpdateMetaTaskThreadAllocator) {}

    Result AsyncSystemUpdateMetaImpl::Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
    {
        m_DeviceId = deviceContext->GetDeviceId();
        return m_Connection.Initialize(deviceContext);
    }

    Result AsyncSystemUpdateMetaImpl::ExecuteAndValue(ncm::ContentMetaKey* outValue) NN_NOEXCEPT
    {
        Url url;
        MakeSystemUpdateMetaQueryUrl(&url, m_DeviceId);

        const char* header("Accept:application/json");

        char readBuffer[1024];
        size_t readSize = 0;
        NN_RESULT_DO(m_Connection.Get(url,
            [&readSize, &readBuffer](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(readBuffer), ResultUnexpectedResponseSystemUpdateMetaTooLong());
                std::memcpy(&readBuffer[readSize], buffer, size);

                readSize += size;

                NN_RESULT_SUCCESS;
            },
            &header, 1));
        readBuffer[readSize] = '\0';
        NN_DETAIL_NIM_TRACE("[AsyncSystemUpdateMetaImpl] Response %s\n", readBuffer);

        NN_RESULT_DO(ParseSystemUpdateMetaKeyResponse(outValue, readBuffer));

        NN_RESULT_SUCCESS;
    }

    Result AsyncSystemUpdateMetaImpl::CancelImpl() NN_NOEXCEPT
    {
        m_Connection.Cancel();
        NN_RESULT_SUCCESS;
    }
}}}
