﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ec/system/ec_DeviceAuthenticationTypes.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TFormatString.h>

#include "nim_AsyncNotificationTokenImpl.h"
#include "nim_AsyncDeviceAuthenticationTokenImpl.h"
#include "nim_EShopUtil.h"
#include "nim_HttpUtil.h"
#include "nim_Json.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
    namespace {
        static const int RegisterNotificationTokenThreadCount = 1;
        static const size_t StackSize = 20 * 1024;

        os::ThreadType g_RegisterNotificationTokenTaskThreadList[RegisterNotificationTokenThreadCount];
        NN_OS_ALIGNAS_THREAD_STACK char g_RegisterNotificationTokenTaskStack[StackSize * RegisterNotificationTokenThreadCount];

        ThreadAllocator g_RegisterNotificationTokenTaskThreadAllocator(
            g_RegisterNotificationTokenTaskThreadList,
            RegisterNotificationTokenThreadCount,
            NN_SYSTEM_THREAD_PRIORITY(nim, RegisterNotificationTokenTask),
            g_RegisterNotificationTokenTaskStack,
            sizeof(g_RegisterNotificationTokenTaskStack),
            StackSize,
            NN_SYSTEM_THREAD_NAME(nim, RegisterNotificationTokenTask));

        Result RegisterNotificationToken(const npns::NotificationToken& notificationToken,
                                         const ec::system::DeviceAuthenticationToken& daToken,
                                         const DeviceContext& deviceContext,
                                         HttpConnection& httpConnection) NN_NOEXCEPT
        {
            Url url;
            auto deviceId = deviceContext.GetDeviceId();
            MakeNotificationTokenRegisterUrl(&url, deviceId);

            const int PostFiledBufferSize = 128 + sizeof(npns::NotificationToken);
            char postFieldBuffer[PostFiledBufferSize] = {};
            util::TSNPrintf(postFieldBuffer, sizeof(postFieldBuffer), "{\"notification_token\":\"%s\"}", notificationToken.data);

            const int AuthTokenBufferSize = 128 + sizeof(daToken.data);
            char authTokenBuffer[AuthTokenBufferSize];
            util::TSNPrintf(authTokenBuffer, sizeof(authTokenBuffer), "X-DeviceAuthorization: Bearer %s", daToken.data);

            const char* HeaderList[] = {
                "Accept: application/json",
                "Content-Type: application/json",
                authTokenBuffer,
            };

            char readBuffer[1024] = {};
            size_t readSize = 0;
            auto result = httpConnection.Post(url, postFieldBuffer,
                [&readBuffer, &readSize](const void* buffer, size_t bufferSize) -> Result
                {
                    NN_RESULT_THROW_UNLESS(readSize + bufferSize < sizeof(readBuffer), ResultUnexpectedResponseDeviceAuthenticationTokenTooLong());
                    std::memcpy(&readBuffer[readSize], buffer, bufferSize);
                    readSize += bufferSize;
                    NN_RESULT_SUCCESS;
                }, HeaderList, sizeof(HeaderList) / sizeof(HeaderList[0]));

            if (result <= ResultHttpStatus())
            {
                NN_RESULT_TRY(ToEShopResultResponseDestructively(readBuffer))
                    NN_RESULT_CATCH(ResultInvalidJson) { NN_RESULT_THROW(result); }
                NN_RESULT_END_TRY
            }
            NN_RESULT_DO(result);

            NN_RESULT_SUCCESS;
        }
    } // namespace

    AsyncRegisterNotificationTokenImpl::AsyncRegisterNotificationTokenImpl() NN_NOEXCEPT : AsyncResultImpl<AsyncRegisterNotificationTokenImpl>(this, &g_RegisterNotificationTokenTaskThreadAllocator), m_pDeviceContext(nullptr), m_NotificationToken({}) {}

    NN_STATIC_ASSERT(std::is_pod<npns::NotificationToken>::value);

    Result AsyncRegisterNotificationTokenImpl::Initialize(const npns::NotificationToken& notificationToken, DeviceContext* deviceContext) NN_NOEXCEPT
    {
        std::memcpy(&m_NotificationToken, &notificationToken, sizeof(m_NotificationToken));
        m_pDeviceContext = deviceContext;
        NN_RESULT_DO(m_HttpConnection.Initialize(m_pDeviceContext));
        NN_RESULT_SUCCESS;
    }

    Result AsyncRegisterNotificationTokenImpl::Execute() NN_NOEXCEPT
    {
        // デバイス認証トークンの取得
        ec::system::DeviceAuthenticationToken daToken = {};
        NN_RESULT_DO(GetShopDeviceAuthenticationToken(&daToken, &m_Cancelable));

        // 通知トークンの登録
        NN_RESULT_DO(RegisterNotificationToken(m_NotificationToken, daToken, *m_pDeviceContext, m_HttpConnection));
        NN_RESULT_SUCCESS;
    }

    Result AsyncRegisterNotificationTokenImpl::Cancel() NN_NOEXCEPT
    {
        m_HttpConnection.Cancel();
        m_Cancelable.Cancel();
        NN_RESULT_SUCCESS;
    }
}}}
