﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include "nim_AsyncDeviceAccountImpl.h"
#include "nim_EcSystemThreadAllocator.h"

namespace nn { namespace nim { namespace srv {

    template<typename TaskT>
    AsyncDeviceAccountResultImplBase<TaskT>::AsyncDeviceAccountResultImplBase(TaskT* self) NN_NOEXCEPT :
        AsyncResultImpl<TaskT>(self, &GetEcSystemThreadAllocator()) {}

    template<typename TaskT, typename T>
    AsyncDeviceAccountValueImplBase<TaskT, T>::AsyncDeviceAccountValueImplBase(TaskT* self) NN_NOEXCEPT :
        AsyncValueImpl<TaskT, T>(self, &GetEcSystemThreadAllocator()) {}


    AsyncRegisterDeviceAccountImpl::AsyncRegisterDeviceAccountImpl(DeviceAccountStore* store) NN_NOEXCEPT : AsyncDeviceAccountResultImplBase(this), m_Store(store) {}
    Result AsyncRegisterDeviceAccountImpl::Execute() NN_NOEXCEPT
    {
        DeviceAccountInfo info;
        EciAccessor accessor(m_DeviceContext, &m_Connection);

        NN_RESULT_DO(accessor.Register(&info.id, &info.token));
        NN_RESULT_DO(m_Store->Save(info));
        NN_RESULT_SUCCESS;
    }

    AsyncUnregisterDeviceAccountImpl::AsyncUnregisterDeviceAccountImpl(DeviceAccountStore* store) NN_NOEXCEPT : AsyncDeviceAccountResultImplBase(this), m_Store(store) {}
    Result AsyncUnregisterDeviceAccountImpl::Execute() NN_NOEXCEPT
    {
        EciAccessor accessor(m_DeviceContext, &m_Connection);

        NN_RESULT_DO(m_Store->Clear());
        NN_RESULT_DO(accessor.Unregister());
        NN_RESULT_SUCCESS;
    }

    AsyncDeviceAccountStatusImpl::AsyncDeviceAccountStatusImpl(DeviceAccountStore* store) NN_NOEXCEPT : AsyncDeviceAccountValueImplBase(this), m_Store(store) {}
    Result AsyncDeviceAccountStatusImpl::ExecuteAndValue(ec::system::DeviceAccountStatus* outValue) NN_NOEXCEPT
    {
        ec::system::DeviceAccountStatus status;
        EciAccessor accessor(m_DeviceContext, &m_Connection);

        NN_RESULT_DO(accessor.GetAccountStatus(&status, m_DeviceAccountInfo.token));

        *outValue = status;
        NN_RESULT_SUCCESS;
    }

    AsyncDeviceRegistrationInfoImpl::AsyncDeviceRegistrationInfoImpl() NN_NOEXCEPT : AsyncDeviceAccountValueImplBase(this) {}
    Result AsyncDeviceRegistrationInfoImpl::ExecuteAndValue(ec::system::DeviceRegistrationInfo* outValue) NN_NOEXCEPT
    {
        ec::system::DeviceAccountStatus status;
        DeviceAccountId accountId;
        DeviceAccountToken token;
        EciAccessor accessor(m_DeviceContext, &m_Connection);

        NN_RESULT_TRY(accessor.GetRegistrationInfo(&status, &accountId, &token))
            NN_RESULT_CATCH(ResultEciIasBadDeviceId) { NN_RESULT_THROW(ResultDeviceNotRegistered()); }
        NN_RESULT_END_TRY

        ec::system::DeviceRegistrationInfo regInfo = {};
        regInfo.accountStatus = status;
        util::Strlcpy(regInfo.accountInfo.id, accountId.data, static_cast<int>(sizeof(regInfo.accountInfo.id)));

        *outValue = regInfo;
        NN_RESULT_SUCCESS;
    }

    AsyncTransferDeviceAccountImpl::AsyncTransferDeviceAccountImpl(DeviceAccountStore* store) NN_NOEXCEPT : AsyncDeviceAccountResultImplBase(this), m_Store(store) {}
    Result AsyncTransferDeviceAccountImpl::Execute() NN_NOEXCEPT
    {
        DeviceAccountInfo info;
        EciAccessor accessor(m_DeviceContext, &m_Connection);

        NN_RESULT_DO(accessor.AccountTransfer(&info.id, &info.token));
        NN_RESULT_DO(m_Store->Save(info));
        NN_RESULT_SUCCESS;
    }

    AsyncSyncRegistrationImpl::AsyncSyncRegistrationImpl(DeviceAccountStore* store) NN_NOEXCEPT : AsyncDeviceAccountResultImplBase(this), m_Store(store) {}
    Result AsyncSyncRegistrationImpl::Execute() NN_NOEXCEPT
    {
        DeviceAccountInfo info;
        EciAccessor accessor(m_DeviceContext, &m_Connection);

        NN_RESULT_DO(accessor.SyncRegistration(&info.id, &info.token));
        NN_RESULT_DO(m_Store->Save(info));
        NN_RESULT_SUCCESS;
    }
}}}
