﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/util/nifm_IpConfiguration.win.h>

#include <nn/nifm/detail/core/profile/nifm_IpSetting.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_LockGuard.h>

#include <iphlpapi.h>

#include <algorithm>
#include <memory>
#include <mutex>

namespace nn
{
namespace nifm
{
namespace detail
{

IpConfiguration::IpConfiguration() NN_NOEXCEPT
    : m_IsInitialized(false),
      m_IsActive(false)
{
}

IpConfiguration::~IpConfiguration() NN_NOEXCEPT
{
    if (m_IsActive)
    {
        NotifyInterfaceDown();
    }
}

nn::Result IpConfiguration::Initialize(const MacAddress& macAddress) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!IsInitialized(), ResultInternalError());    // TODO: [TORIAEZU]

    m_MacAddress = macAddress;

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

bool IpConfiguration::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

nn::Result IpConfiguration::ApplyIpSetting(const IpSetting& ipSetting) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInitialized(), ResultInternalError());     // TODO: [TORIAEZU]
    NN_RESULT_THROW_UNLESS(!IsActive(), ResultInternalError());         // TODO: [TORIAEZU]

    NN_UNUSED(ipSetting);

    // do nothing

    m_IsActive = true;

    NN_RESULT_SUCCESS;
}

bool IpConfiguration::IsActive() const NN_NOEXCEPT
{
    return m_IsActive;
}

nn::Result IpConfiguration::NotifyInterfaceDown() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInitialized(), ResultInternalError());     // TODO: [TORIAEZU]
    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    // do nothing

    m_IsActive = false;

    NN_RESULT_SUCCESS;
}

nn::Result IpConfiguration::GetIpAddress(IpV4Address* pOutIpAddress) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInitialized(), ResultInternalError());     // TODO: [TORIAEZU]
    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    DWORD error;
    ULONG size = 0;

    error = ::GetAdaptersAddresses(AF_INET, GAA_FLAG_INCLUDE_PREFIX, NULL, NULL, &size);
    if (error != ERROR_BUFFER_OVERFLOW)
    {
        NN_DETAIL_NIFM_ERROR("GetAdaptersAddresses returned %d\n", error);
    }
    NN_RESULT_THROW_UNLESS(error == ERROR_BUFFER_OVERFLOW, ResultWindowsApiFailed());

    NN_DETAIL_NIFM_TRACE("GetAdaptersAddresses required size = %u\n", size);
    NN_RESULT_THROW_UNLESS(size <= static_cast<ULONG>(sizeof(m_BufferForGetAdaptersAddresses)), ResultOutOfMemory());

    NN_UTIL_LOCK_GUARD(m_Mutex);
    PIP_ADAPTER_ADDRESSES pIpAdaptersAddresses = reinterpret_cast<::PIP_ADAPTER_ADDRESSES>(m_BufferForGetAdaptersAddresses);

    error = ::GetAdaptersAddresses(AF_INET, GAA_FLAG_INCLUDE_PREFIX, NULL, pIpAdaptersAddresses, &size);
    if (error != NO_ERROR)
    {
        NN_DETAIL_NIFM_ERROR("GetAdaptersAddresses returned %d\n", error);
    }
    NN_RESULT_THROW_UNLESS(error == NO_ERROR, ResultWindowsApiFailed());

    for (PIP_ADAPTER_ADDRESSES p = pIpAdaptersAddresses; p != nullptr; p = p->Next)
    {
        // MAC アドレスの一致する NIC に登録された IP アドレスを返す
        if (p->PhysicalAddressLength == m_MacAddress.Size && std::memcmp(p->PhysicalAddress, m_MacAddress.data, m_MacAddress.Size) == 0)
        {
            NN_RESULT_THROW_UNLESS(p->OperStatus == IfOperStatusUp && p->FirstUnicastAddress != nullptr, ResultNotConnected()); // TODO

            ULONG addr = reinterpret_cast<LPSOCKADDR_IN>(p->FirstUnicastAddress->Address.lpSockaddr)->sin_addr.s_addr;
            pOutIpAddress->data[0] = (addr >> 0) & 0xff;
            pOutIpAddress->data[1] = (addr >> 8) & 0xff;
            pOutIpAddress->data[2] = (addr >> 16) & 0xff;
            pOutIpAddress->data[3] = (addr >> 24) & 0xff;

            NN_RESULT_SUCCESS;
        }
    }

    NN_DETAIL_NIFM_ERROR("NIC not found.\n");

    NN_RESULT_THROW(ResultResourceNotFound());  // TODO
}

nn::Result IpConfiguration::GetIpConfigInfo(IpAddressSetting* pOutIpAddressSetting, DnsSetting* pDnsSetting) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInitialized(), ResultInternalError());     // TODO: [TORIAEZU]
    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    DWORD error;
    ULONG size = 0;

    error = ::GetAdaptersAddresses(AF_INET, GAA_FLAG_INCLUDE_PREFIX | GAA_FLAG_INCLUDE_GATEWAYS, NULL, NULL, &size);
    if (error != ERROR_BUFFER_OVERFLOW)
    {
        NN_DETAIL_NIFM_ERROR("GetAdaptersAddresses returned %d\n", error);
    }
    NN_RESULT_THROW_UNLESS(error == ERROR_BUFFER_OVERFLOW, ResultWindowsApiFailed());

    NN_DETAIL_NIFM_TRACE("GetAdaptersAddresses required size = %u\n", size);
    NN_RESULT_THROW_UNLESS(size <= static_cast<ULONG>(sizeof(m_BufferForGetAdaptersAddresses)), ResultOutOfMemory());

    NN_UTIL_LOCK_GUARD(m_Mutex);
    PIP_ADAPTER_ADDRESSES pIpAdaptersAddresses = reinterpret_cast<::PIP_ADAPTER_ADDRESSES>(m_BufferForGetAdaptersAddresses);

    error = ::GetAdaptersAddresses(AF_INET, GAA_FLAG_INCLUDE_PREFIX | GAA_FLAG_INCLUDE_GATEWAYS, NULL, pIpAdaptersAddresses, &size);
    if (error != NO_ERROR)
    {
        NN_DETAIL_NIFM_ERROR("GetAdaptersAddresses returned %d\n", error);
    }
    NN_RESULT_THROW_UNLESS(error == NO_ERROR, ResultWindowsApiFailed());

    for (PIP_ADAPTER_ADDRESSES p = pIpAdaptersAddresses; p != nullptr; p = p->Next)
    {
        // MAC アドレスの一致する NIC に登録された IP アドレスを返す
        if (p->PhysicalAddressLength == m_MacAddress.Size && std::memcmp(p->PhysicalAddress, m_MacAddress.data, m_MacAddress.Size) == 0)
        {
            NN_RESULT_THROW_UNLESS(p->OperStatus == IfOperStatusUp && p->FirstUnicastAddress != nullptr, ResultNotConnected()); // TODO

            ULONG addr = 0;
            if (p->FirstUnicastAddress->Address.lpSockaddr != nullptr)
            {
                addr = reinterpret_cast<LPSOCKADDR_IN>(p->FirstUnicastAddress->Address.lpSockaddr)->sin_addr.s_addr;
            }
            pOutIpAddressSetting->ipAddress.data[0] = (addr >> 0) & 0xff;
            pOutIpAddressSetting->ipAddress.data[1] = (addr >> 8) & 0xff;
            pOutIpAddressSetting->ipAddress.data[2] = (addr >> 16) & 0xff;
            pOutIpAddressSetting->ipAddress.data[3] = (addr >> 24) & 0xff;

            if (ConvertLengthToIpv4Mask(p->FirstUnicastAddress->OnLinkPrefixLength, &addr) != NO_ERROR)
            {
                addr = 0;
            }
            pOutIpAddressSetting->subnetMask.data[0] = (addr >> 0) & 0xff;
            pOutIpAddressSetting->subnetMask.data[1] = (addr >> 8) & 0xff;
            pOutIpAddressSetting->subnetMask.data[2] = (addr >> 16) & 0xff;
            pOutIpAddressSetting->subnetMask.data[3] = (addr >> 24) & 0xff;

            addr = 0;
            if (p->FirstGatewayAddress != nullptr && p->FirstGatewayAddress->Address.lpSockaddr != nullptr)
            {
                addr = reinterpret_cast<LPSOCKADDR_IN>(p->FirstGatewayAddress->Address.lpSockaddr)->sin_addr.s_addr;
            }
            pOutIpAddressSetting->defaultGateway.data[0] = (addr >> 0) & 0xff;
            pOutIpAddressSetting->defaultGateway.data[1] = (addr >> 8) & 0xff;
            pOutIpAddressSetting->defaultGateway.data[2] = (addr >> 16) & 0xff;
            pOutIpAddressSetting->defaultGateway.data[3] = (addr >> 24) & 0xff;

            addr = 0;
            if (p->FirstDnsServerAddress != nullptr && p->FirstDnsServerAddress->Address.lpSockaddr != nullptr)
            {
                addr = reinterpret_cast<LPSOCKADDR_IN>(p->FirstDnsServerAddress->Address.lpSockaddr)->sin_addr.s_addr;
            }
            pDnsSetting->preferredDns.data[0] = (addr >> 0) & 0xff;
            pDnsSetting->preferredDns.data[1] = (addr >> 8) & 0xff;
            pDnsSetting->preferredDns.data[2] = (addr >> 16) & 0xff;
            pDnsSetting->preferredDns.data[3] = (addr >> 24) & 0xff;

            addr = 0;
            if (p->FirstDnsServerAddress != nullptr && p->FirstDnsServerAddress->Next != nullptr && p->FirstDnsServerAddress->Next->Address.lpSockaddr != nullptr)
            {
                addr = reinterpret_cast<LPSOCKADDR_IN>(p->FirstDnsServerAddress->Next->Address.lpSockaddr)->sin_addr.s_addr;
            }
            pDnsSetting->alternateDns.data[0] = (addr >> 0) & 0xff;
            pDnsSetting->alternateDns.data[1] = (addr >> 8) & 0xff;
            pDnsSetting->alternateDns.data[2] = (addr >> 16) & 0xff;
            pDnsSetting->alternateDns.data[3] = (addr >> 24) & 0xff;

            NN_RESULT_SUCCESS;
        }
    }

    NN_DETAIL_NIFM_ERROR("NIC not found.\n");

    NN_RESULT_THROW(ResultResourceNotFound());  // TODO
}

nn::os::SystemEvent* IpConfiguration::GetSystemEventPointer() NN_NOEXCEPT
{
    return nullptr;
}

nn::Result IpConfiguration::GetInterfaceStatus() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result IpConfiguration::GetDefaultGatewayMacAddress(MacAddress* pOutMacAddress) const NN_NOEXCEPT
{
    NN_UNUSED(pOutMacAddress);

    NN_RESULT_THROW(ResultNotImplemented());
}

}
}
}
