﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/service/nifm_ScanRequestImpl.h>

#include <nn/nifm/detail/core/nifm_ConnectionSelector.h>


namespace nn
{
namespace nifm
{
namespace detail
{

ScanRequestImpl::SystemEventHolder::SystemEventHolder() NN_NOEXCEPT
    : m_SystemEvent(nn::os::EventClearMode_AutoClear, true),
      m_IsToBeSignaled(false)
{
}

ScanRequestImpl::SystemEventHolder::~SystemEventHolder() NN_NOEXCEPT
{
}

void ScanRequestImpl::SystemEventHolder::Signal() NN_NOEXCEPT
{
    if (m_IsToBeSignaled)
    {
        m_SystemEvent.Signal();
        m_IsToBeSignaled = false;
    }
}

void ScanRequestImpl::SystemEventHolder::SetToBeSignaled(bool isToBeSignaled) NN_NOEXCEPT
{
    m_IsToBeSignaled = isToBeSignaled;
}

nn::os::SystemEvent& ScanRequestImpl::SystemEventHolder::Get() NN_NOEXCEPT
{
    return m_SystemEvent;
}

const nn::os::SystemEvent& ScanRequestImpl::SystemEventHolder::Get() const NN_NOEXCEPT
{
    return m_SystemEvent;
}


ScanRequestImpl::ScanRequestImpl(ConnectionSelector* pConnectionSelector, const Capability& capability) NN_NOEXCEPT
    : m_pConnectionSelector(pConnectionSelector),
      m_Capability(capability),
      m_ScanCounter(ConnectionSelector::InvalidScanCounter),
      m_SignalListItem(m_SystemEventHolder),
      m_ScanChannelNum(0),
      m_ScanChannels()
{
}

ScanRequestImpl::~ScanRequestImpl() NN_NOEXCEPT
{
    m_pConnectionSelector->RemoveScanCompleteListener(m_SignalListItem);
}

nn::Result ScanRequestImpl::Submit() NN_NOEXCEPT
{
    if (!m_pConnectionSelector->IsScanAllowed(m_ScanChannels, m_ScanChannelNum))
    {
        m_ScanCounter = ConnectionSelector::InvalidScanCounter;
        NN_RESULT_SUCCESS;
    }

    m_pConnectionSelector->RemoveScanCompleteListener(m_SignalListItem);
    m_SystemEventHolder.SetToBeSignaled(true);

    m_pConnectionSelector->TriggerScan(&m_ScanCounter, m_SignalListItem);

    NN_RESULT_SUCCESS;
}

nn::Result ScanRequestImpl::IsProcessing(nn::sf::Out<bool> outIsProcessing) const NN_NOEXCEPT
{
    outIsProcessing.Set(static_cast<IntRequestState>(IsProcessing()));

    NN_RESULT_SUCCESS;
}

nn::Result ScanRequestImpl::GetSystemEventReadableHandle(nn::sf::Out<nn::sf::NativeHandle> systemEventReadableHandle) NN_NOEXCEPT
{
    systemEventReadableHandle.Set(nn::sf::NativeHandle(GetSystemEventReadableHandle(), false));

    NN_RESULT_SUCCESS;
}

nn::Result ScanRequestImpl::GetResult() const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!IsProcessing(), ResultProcessing());    // TODO: ResultProcessing は ResultNotProperConnection に含まれている
    NN_RESULT_THROW_UNLESS(m_ScanCounter != ConnectionSelector::InvalidScanCounter, ResultProcessing());

    NN_RESULT_THROW(m_pConnectionSelector->GetScanResult());
}

bool ScanRequestImpl::IsProcessing() const NN_NOEXCEPT
{
    return m_ScanCounter == m_pConnectionSelector->GetScanCounter();
}

nn::os::NativeHandle ScanRequestImpl::GetSystemEventReadableHandle() NN_NOEXCEPT
{
    return m_SystemEventHolder.Get().GetReadableHandle();
}

nn::Result ScanRequestImpl::SetChannels(nn::sf::InArray<int16_t> scanChannelArray) NN_NOEXCEPT
{
    m_ScanChannelNum = static_cast<int>(std::min(scanChannelArray.GetLength(), NN_ARRAY_SIZE(m_ScanChannels)));

    for (size_t i = 0; i < scanChannelArray.GetLength(); ++i)
    {
        NN_RESULT_THROW_UNLESS(scanChannelArray[i] > 0, ResultInvalidArgument());
    }

    for (int i = 0; i < m_ScanChannelNum; ++i)
    {
        m_ScanChannels[i] = scanChannelArray[i];
    }

    NN_RESULT_SUCCESS;
}

}
}
}
