﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/nifm_RequestManager.h>

#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>

#include <nn/util/util_LockGuard.h>

namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{

class PriorityDescending
{
public:
    bool operator()(const UserRequest* pLeft, const UserRequest* pRight) const NN_NOEXCEPT
    {
        return pLeft->GetRequirement()._priority < pRight->GetRequirement()._priority;
    }
};

bool IsProfileIdAcceptable(const nn::util::Uuid& required, const nn::util::Uuid& actual)
{
    return required == nn::util::InvalidUuid || required == actual;
}

bool IsMoreRestricted(bool isGreedy1, bool isSharable1, bool isGreedy2, bool isSharable2)
{

    if (!isGreedy1 && isSharable1)
    {
        return true;
    }
    else if ((isGreedy1 && isSharable1) || (!isGreedy1 && !isSharable1))
    {
        if (isGreedy2 && !isSharable2)
        {
            return true;
        }
    }
    return false;
}

bool IsAbleToCoexist(bool isGreedy1, bool isSharable1, bool isGreedy2, bool isSharable2) NN_NOEXCEPT
{
    // !greedy && sharable はすべての要求と共存可能
    if ((!isGreedy1 && isSharable1) || (!isGreedy2 && isSharable2))
    {
        return true;
    }

    // greedy && sharable 同士は共存可能
    if ((isGreedy1 && isGreedy2) && (isSharable1 && isSharable2))
    {
        return true;
    }

    // !greedy 同士は共存可能
    if (!isGreedy1 && !isGreedy2)
    {
        return true;
    }

    return false;
}

bool IsAbleToCoexist(const Requirement& requirement, const TotalNetworkResourceInfo& totalNetworkResourceInfo) NN_NOEXCEPT
{
    const bool isGreedy1 = totalNetworkResourceInfo.isGreedy;
    const bool isSharable1 = totalNetworkResourceInfo.isSharable;
    const bool isGreedy2 = requirement._isGreedy;
    const bool isSharable2 = requirement._isSharable;

    return IsAbleToCoexist(isGreedy1, isSharable1, isGreedy2, isSharable2);
}

bool IsAbleToCoexist(const Requirement& requirement, const AggregatedRequestType& aggregateRequestType) NN_NOEXCEPT
{
    const bool isGreedy1 = aggregateRequestType.isGreedy;
    const bool isSharable1 = aggregateRequestType.isSharable;
    const bool isGreedy2 = requirement._isGreedy;
    const bool isSharable2 = requirement._isSharable;

    return IsAbleToCoexist(isGreedy1, isSharable1, isGreedy2, isSharable2);
}

bool IsUserRequestAcceptable(const UserRequest& userRequest, const TotalNetworkResourceInfo& totalNetworkResourceInfo) NN_NOEXCEPT
{
    const Requirement& requirement = userRequest.GetRequirement();

    // 使用されているネットワークリソースの種類が異なっていれば却下
    if (requirement._networkType != totalNetworkResourceInfo.networkType)
    {
        return false;
    }

    // 使用されている接続設定が異なっていれば却下
    if (!IsProfileIdAcceptable(requirement._profileId, totalNetworkResourceInfo.networkProfileId))
    {
        return false;
    }

    // 現在の接続と共存できない要求は却下
    if (userRequest.GetRevision() != totalNetworkResourceInfo.restrictedRevision &&
        !IsAbleToCoexist(requirement, totalNetworkResourceInfo))
    {
        return false;
    }

    // UserRequest に Requirement がセットされる際にチェックされているはずの範囲チェック
    NN_SDK_ASSERT_GREATER_EQUAL(requirement._connectionConfirmationOption, ConnectionConfirmationOption_Invalid);
    NN_SDK_ASSERT_LESS(requirement._connectionConfirmationOption, ConnectionConfirmationOption_Count);

    NN_SDK_ASSERT_LESS(totalNetworkResourceInfo.internetAvailability, InternetAvailability_Count);

    // 現在疎通確認されてなければ、疎通確認を要求している未受理の利用要求は却下
    // 疎通確認拒否のリクエストでも、接続が確認済みであることは問題ない
    if (userRequest.GetPresentConnectionConfirmationOption() >= ConnectionConfirmationOption_Required &&
        totalNetworkResourceInfo.internetAvailability <= InternetAvailability_NotConfirmed)
    {
        return false;
    }

    return true;
}

bool IsUserRequestCompatible(const UserRequest& userRequest, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    const Requirement& requirement = userRequest.GetRequirement();

    if (userRequest.GetRequestState() == RequestState_Free)
    {
        return false;
    }

    // 使用されているネットワークリソースの種類が異なっていれば却下
    if (requirement._networkType != aggregatedRequest.networkType)
    {
        return false;
    }

    // 使用されている接続設定が異なっていれば却下
    if (!IsProfileIdAcceptable(requirement._profileId, aggregatedRequest.profileId))
    {
        return false;
    }

    // 現在の要求と共存できない要求は却下
    if (userRequest.GetRevision() != aggregatedRequest.restrictedRevision &&
        !IsAbleToCoexist(requirement, aggregatedRequest))
    {
        return false;
    }

    // 疎通確認が禁止されているとき、疎通確認が必要な未受理の要求は却下
    if (userRequest.GetPresentConnectionConfirmationOption() >= ConnectionConfirmationOption_Required &&
        aggregatedRequest.connectionConfirmationOption == ConnectionConfirmationOption_Prohibited)
    {
        return false;
    }

    return true;
}

nn::Result ValidateRequest(const UserRequest& userRequest) NN_NOEXCEPT
{
    const Requirement& requirement = userRequest.GetRequirement();

    if (requirement._isSustainable)
    {
        // sustainable はインターネット接続要求
        if (requirement._networkType != NetworkType_Internet)
        {
            NN_RESULT_THROW(ResultInvalidRequirement());
        }
    }

    NN_RESULT_SUCCESS;
}

}

RequestManager::RequestManager() NN_NOEXCEPT
    : m_SubmitEvent(nn::os::EventClearMode_AutoClear),
      m_SubmitEventHandler(m_SubmitEvent),
      m_LastRevision(0),
      m_State(State::FullAwake),
      m_IsBackgroundRequestEnabled(true),
      m_IsAggregatedRequestStaged(false),
      m_AggregatedRequest()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_pInternalRequest); ++i)
    {
        m_pInternalRequest[i] = nullptr;
    }

    m_ExclusiveClientId.value = InvalidClientIdValue;
}

RequestManager::~RequestManager() NN_NOEXCEPT
{
    // TODO: m_UserRequestList に残ったリクエストを掃除する？
}

nn::Result RequestManager::CreateRequest(
    UserRequest** ppOutUserRequest,
    RequirementPreset requirementPreset,
    ISignalObject* pSignalObject,
    ClientId clientId,
    nn::Bit64 processId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(ppOutUserRequest);

    auto pUserRequest = m_UserRequestHeap.Create(this, requirementPreset, pSignalObject, clientId, processId);

    if (pUserRequest != nullptr)
    {
        {
            NN_UTIL_LOCK_GUARD(m_Mutex);

            m_RequestList.push_back(*pUserRequest);
        }

        *ppOutUserRequest = pUserRequest;
        NN_RESULT_SUCCESS;
    }
    else
    {
        *ppOutUserRequest = nullptr;
        NN_RESULT_THROW(nn::nifm::ResultOutOfResource());   // TODO
    }
}

nn::Result RequestManager::CreateInternalRequest(UserRequest** ppOutUserRequest, InternalRequestType requestType) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(ppOutUserRequest);
    NN_SDK_REQUIRES_LESS(requestType, InternalRequestType::Count);

    UserRequest** ppInternalRequest = nullptr;

    if (requestType == InternalRequestType::SustainConnection)
    {
        ppInternalRequest = &m_pInternalRequest[static_cast<int>(requestType)];

        if (ppInternalRequest != nullptr && *ppInternalRequest != nullptr)
        {
            Cancel(*ppInternalRequest);     // 以前の内部利用要求を取り下げ

            *ppOutUserRequest = *ppInternalRequest;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result result = ResultSuccess();
    switch (requestType)
    {
    case InternalRequestType::SustainConnection:
        {
            ClientId invalidClientId = { InvalidClientIdValue };
            result = CreateRequest(ppOutUserRequest, RequirementPreset_SustainConnection, nullptr, invalidClientId, 0);
        }
        break;
    default:
        break;
    }

    if (ppInternalRequest != nullptr && result.IsSuccess())
    {
        NN_SDK_ASSERT(*ppInternalRequest == nullptr);
        *ppInternalRequest = *ppOutUserRequest;
    }

    NN_RESULT_THROW(result);
}

nn::Result RequestManager::DestroyRequest(UserRequest* pInUserRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInUserRequest);

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        for( auto itr = m_RequestList.cbegin(); itr != m_RequestList.cend(); ++itr )
        {
            if( &(*itr) == pInUserRequest )
            {
                for (int i = 0; i < static_cast<int>(InternalRequestType::Count); ++i)
                {
                    if (m_pInternalRequest[i] == pInUserRequest)
                    {
                        m_pInternalRequest[i] = nullptr;
                    }
                }

                auto requestState = itr->GetRequestState();
                m_RequestList.erase(itr);
                if (requestState != RequestState_Free && requestState != RequestState_Invalid)
                {
                    NN_RESULT_DO(UpdateAggregatedRequest());
                    m_SubmitEvent.Signal();
                }
                break;
            }
        }
    }

    m_UserRequestHeap.Delete(pInUserRequest);

    NN_RESULT_SUCCESS;
}

void RequestManager::Sort(const UserRequest* pInUserRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInUserRequest);
    NN_SDK_ASSERT_EQUAL(pInUserRequest->GetRequestState(), RequestState_Free);
    NN_SDK_ASSERT(ValidateRequest(*pInUserRequest).IsSuccess());

    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    // 提出された要求を整列
    for (auto p = m_RequestList.crbegin(); p != m_RequestList.crend(); ++p)
    {
        auto requestState = p->GetRequestState();
        auto pInUserRequestIter = m_RequestList.iterator_to(static_cast<const UserRequest&>(*pInUserRequest));

        if ((requestState == RequestState_OnHold || requestState == RequestState_Accepted || requestState == RequestState_Blocking) &&
            pInUserRequestIter != std::next(p, 1).base() &&
            p->GetRequirement()._priority <= pInUserRequest->GetRequirement()._priority)
        {
            if (pInUserRequestIter == p.base())
            {
                // 元の位置
            }
            else
            {
                // 有効な要求が優先度順，同優先度なら提出順に並ぶ（requestState==Free な要求の順序は保証されない）
                m_RequestList.splice(p.base(), m_RequestList, pInUserRequestIter);
            }
            break;
        }
        else if (p == (--m_RequestList.crend()) && pInUserRequestIter != m_RequestList.begin())
        {
            // ここまで挿入する場所がなければ先頭に
            m_RequestList.splice(m_RequestList.begin(), m_RequestList, pInUserRequestIter);
            break;
        }
    }
}

nn::Result RequestManager::SubmitCore(const UserRequest* pInUserRequest, UserRequest** ppOutInternalRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInUserRequest);
    NN_SDK_ASSERT_NOT_NULL(ppOutInternalRequest);

    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    *ppOutInternalRequest = nullptr;

    switch (pInUserRequest->GetRequestState())
    {
    case RequestState_Free:
        break;

    case RequestState_OnHold:
    case RequestState_Blocking:
    case RequestState_Accepted:
        NN_RESULT_SUCCESS;

    default:
        NN_RESULT_THROW(ResultInternalError()); // TODO
    }

    // TODO:
    NN_RESULT_DO(ValidateRequest(*pInUserRequest));

    if (pInUserRequest->GetRequirement()._isSustainable)
    {
        UserRequest* pInternalRequest = nullptr;

        NN_RESULT_DO(CreateInternalRequest(&pInternalRequest, InternalRequestType::SustainConnection));

        NN_SDK_ASSERT(!pInternalRequest->GetRequirement()._isSustainable);
        NN_RESULT_DO(pInternalRequest->SetClientId(pInUserRequest->GetClientId()));
        NN_RESULT_DO(pInternalRequest->SetNetworkProfileId(pInUserRequest->GetRequirement()._profileId));
        NN_RESULT_DO(pInternalRequest->SetConnectionConfirmationOption(pInUserRequest->GetRequirement()._connectionConfirmationOption));

        Sort(pInternalRequest);

        *ppOutInternalRequest = pInternalRequest;
    }

    Sort(pInUserRequest);

    NN_RESULT_SUCCESS;
}

nn::Result RequestManager::Submit(UserRequest* pInUserRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInUserRequest);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    UserRequest* pInternalRequest = nullptr;
    auto result = SubmitCore(pInUserRequest, &pInternalRequest);

    if (result.IsSuccess())
    {
        const auto requestState = pInUserRequest->GetRequestState();

        pInUserRequest->Receive(++m_LastRevision);

        if (pInternalRequest != nullptr)
        {
            pInternalRequest->Receive(++m_LastRevision);
        }

        if (pInUserRequest->GetRequirement()._isInstant &&
            requestState == RequestState_Free)
        {
            NN_RESULT_DO(UpdateAggregatedRequest());
            m_SubmitEvent.Signal();
        }
    }
    else
    {
        pInUserRequest->Refuse(result);
    }

    NN_RESULT_SUCCESS;
}

nn::Result RequestManager::Cancel(UserRequest* pInUserRequest) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(pInUserRequest->GetRequestState() != RequestState_Free, ResultSuccess());

    for( auto itr = m_RequestList.cbegin(); itr != m_RequestList.cend(); ++itr )
    {
        if( &(*itr) == pInUserRequest )
        {
            pInUserRequest->Refuse(nn::nifm::ResultCanceled());
            NN_RESULT_DO(UpdateAggregatedRequest());
            m_SubmitEvent.Signal();
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

bool RequestManager::IsAnyInternetRequestAccepted(ClientId clientId) NN_NOEXCEPT
{
    // 内部的な持続要求などは InvalidClientIdValue をクライアント ID としているので先にはじく
    if (clientId.value == InvalidClientIdValue)
    {
        return false;
    }

    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& userRequest : m_RequestList)
    {
        if (userRequest.GetClientId() == clientId &&
            userRequest.GetRequirement()._networkType == NetworkType_Internet &&
            userRequest.GetRequestState() == RequestState_Accepted)
        {
            return true;
        }
    }

    return false;
}

bool RequestManager::IsAnyForegroundRequestAccepted() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& userRequest : m_RequestList)
    {
        if (userRequest.GetRequirement()._isForeground &&
            userRequest.GetRequestState() == RequestState_Accepted)
        {
            return true;
        }
    }

    return false;
}

bool RequestManager::IsAutomaticSwitchProhibited() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& userRequest : m_RequestList)
    {
        if (userRequest.GetRequirement()._isAutomaticSwitchProhibited)
        {
            switch (userRequest.GetRequestState())
            {
            case RequestState_OnHold:
            case RequestState_Accepted:
            case RequestState_Blocking:
                return true;

            case RequestState_Free:
            default:
                break;
            }
        }
    }

    return false;
}

nn::Result RequestManager::GetRequirement(Requirement *pOutRequirement, uint32_t revision) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutRequirement);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& userRequest : m_RequestList)
    {
        if (userRequest.GetRevision() == revision)
        {
            RequestState requestState = userRequest.GetRequestState();
            if( requestState != RequestState_Invalid)
            {
                *pOutRequirement = userRequest.GetRequirement();
                NN_RESULT_SUCCESS;
            }
        }
    }

    NN_RESULT_THROW(ResultRequestNotFound());
}

void RequestManager::GetAggregatedRequest(AggregatedRequestType* pOutAggregatedRequestType) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    *pOutAggregatedRequestType = m_AggregatedRequest;
}

void RequestManager::StageAggregatedRequest(AggregatedRequestType* pOutAggregatedRequestType) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_SDK_ASSERT(!m_IsAggregatedRequestStaged);

    m_IsAggregatedRequestStaged = true;

    CancelFlagHolder::GetSingleton().ClearCancelByRequestModified();

    *pOutAggregatedRequestType = m_AggregatedRequest;
}

void RequestManager::UnstageAggregatedRequest(AggregatedRequestType* pOutAggregatedRequestType) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_SDK_ASSERT(m_IsAggregatedRequestStaged);

    m_IsAggregatedRequestStaged = false;

    if (pOutAggregatedRequestType != nullptr)
    {
        *pOutAggregatedRequestType = m_AggregatedRequest;
    }
}

nn::Result RequestManager::AggregateSubmittedRequestsImpl(AggregatedRequestType* pOutAggregatedRequestType) NN_NOEXCEPT
{
    // TODO: とりあえず優先度のいちばん高いもの

    const UserRequest* pUserRequest = nullptr;

    NN_UTIL_LOCK_GUARD(m_Mutex);

    bool isAllRejectableRequest = true;

    for (const auto& userRequest : m_RequestList)
    {
        switch (userRequest.GetRequestState())
        {
        case RequestState_Accepted:
        case RequestState_Blocking:
            // 受理済みの要求は、スリープ状態に入ってもここではそのまま残し Renew() で却下する
            isAllRejectableRequest = isAllRejectableRequest && userRequest.GetRequirement()._isRejectable;
            if (pUserRequest == nullptr || userRequest.GetRequirement()._priority < pUserRequest->GetRequirement()._priority)
            {
                pUserRequest = &userRequest;
            }
            break;

        case RequestState_OnHold:
            // 未受理の要求は、スリープ中に維持するべきであったとしてもここでは保留し、間欠起動で起床状態になったときに処理する
            if (IsEvaluandRequest(userRequest))
            {
                if (pUserRequest == nullptr || userRequest.GetRequirement()._priority < pUserRequest->GetRequirement()._priority)
                {
                    pUserRequest = &userRequest;
                }
            }
            break;

        default:
            // do nothing
            break;
        }
    }

    *pOutAggregatedRequestType = {};
    if (pUserRequest == nullptr)
    {
        NN_DETAIL_NIFM_INFO("No user requests.\n");

        pOutAggregatedRequestType->priority = 255;
        pOutAggregatedRequestType->networkType = NetworkType_None;
        pOutAggregatedRequestType->profileId = nn::util::InvalidUuid;
        pOutAggregatedRequestType->isGreedy = false;
        pOutAggregatedRequestType->isSharable = true;
        pOutAggregatedRequestType->restrictedRevision = 0;

        pOutAggregatedRequestType->connectionConfirmationOption = ConnectionConfirmationOption_Invalid;
        pOutAggregatedRequestType->connectionConfirmationOptionSub = ConnectionConfirmationOption_Invalid;
        pOutAggregatedRequestType->revision = 0;

        pOutAggregatedRequestType->isAcceptedRejectable = true;
        pOutAggregatedRequestType->isForeground = false;
        pOutAggregatedRequestType->isInternalContextUpdatable = false;
    }
    else
    {
        const Requirement& requirement = pUserRequest->GetRequirement();

        pOutAggregatedRequestType->priority = requirement._priority;
        pOutAggregatedRequestType->networkType = requirement._networkType;
        pOutAggregatedRequestType->profileId = requirement._profileId;
        pOutAggregatedRequestType->isGreedy = requirement._isGreedy;
        pOutAggregatedRequestType->isSharable = requirement._isSharable;
        pOutAggregatedRequestType->restrictedRevision = pUserRequest->GetRevision();

        pOutAggregatedRequestType->connectionConfirmationOption = pUserRequest->GetPresentConnectionConfirmationOption();
        pOutAggregatedRequestType->connectionConfirmationOptionSub = pOutAggregatedRequestType->connectionConfirmationOption;

        pOutAggregatedRequestType->revision = pUserRequest->GetRevision();

        pOutAggregatedRequestType->isAcceptedRejectable = isAllRejectableRequest;
        pOutAggregatedRequestType->isForeground = requirement._isForeground;
        pOutAggregatedRequestType->isInternalContextUpdatable = false;

        for (const auto& userRequest2 : m_RequestList)
        {
            if (IsUserRequestCompatible(userRequest2, *pOutAggregatedRequestType))
            {
                if (userRequest2.GetRequestState() == RequestState_Accepted ||
                    userRequest2.GetRequestState() == RequestState_Blocking ||
                    IsEvaluandRequest(userRequest2))
                {
                    // 相乗り可能な要求の中で、疎通確認に関して最も強い指定を取り出す
                    if (requirement._networkType == NetworkType_Internet &&
                        pOutAggregatedRequestType->connectionConfirmationOption != ConnectionConfirmationOption_Prohibited)
                    {
                        ConnectionConfirmationOption option = userRequest2.GetPresentConnectionConfirmationOption();

                        if (pOutAggregatedRequestType->connectionConfirmationOptionSub < option)
                        {
                            pOutAggregatedRequestType->connectionConfirmationOptionSub = option;
                        }
                    }

                    // 相乗り可能な要求の中で、最も厳しい共存フラグを取り出す
                    const Requirement& requirement2 = userRequest2.GetRequirement();
                    if (IsMoreRestricted(pOutAggregatedRequestType->isGreedy, pOutAggregatedRequestType->isSharable, requirement2._isGreedy, requirement2._isSharable))
                    {
                        pOutAggregatedRequestType->isGreedy = requirement2._isGreedy;
                        pOutAggregatedRequestType->isSharable = requirement2._isSharable;
                        // TODO:
                        // 現在は（greedy, sharable の組み合わせとして）最大2種類の要求しか相乗りできないので、
                        // 最も厳しい要求を一つだけ覚えておけばよい
                        pOutAggregatedRequestType->restrictedRevision = userRequest2.GetRevision();
                    }

                    // 相乗り可能な要求の中で、一つでも FG 要求があれば FG
                    if (requirement2._isForeground)
                    {
                        pOutAggregatedRequestType->isForeground = true;
                    }
                    // 未評価の相乗り可能な要求の中で、一つでもテレメトリコンテキスト更新要求があれば更新
                    if (requirement2._isInternalContextUpdatable && userRequest2.GetRequestState() == RequestState_OnHold)
                    {
                        pOutAggregatedRequestType->isInternalContextUpdatable = true;
                    }
                }
            }
        }
    }

    pOutAggregatedRequestType->lastRevision = m_LastRevision;

    NN_RESULT_SUCCESS;
}   // NOLINT(impl/function_size) が必要

bool RequestManager::IsEvaluandRequest(const UserRequest& userRequest) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(userRequest.GetRequestState(), RequestState_OnHold);

    const Requirement requirement = userRequest.GetRequirement();

    if (userRequest.IsTemporaryExcluded())
    {
        return false;
    }
    else if (m_State == State::FullAwake &&
        (!m_ExclusiveClientId.IsValid() || m_ExclusiveClientId == userRequest.GetClientId()) &&
        (m_IsBackgroundRequestEnabled || requirement._isForeground || IsSpecifiedInternalRequest(userRequest, InternalRequestType::SustainConnection)))
    {
        return true;
    }
    // MinimumAwake 状態では、FG からの利用要求は評価されない
    else if (m_State == State::MinimumAwake &&
        (!m_ExclusiveClientId.IsValid() || m_ExclusiveClientId == userRequest.GetClientId()) &&
        ((m_IsBackgroundRequestEnabled && !requirement._isForeground) || IsSpecifiedInternalRequest(userRequest, InternalRequestType::SustainConnection)))
    {
        return true;
    }

    return false;
}

bool RequestManager::IsSpecifiedInternalRequest(const UserRequest& userRequest, InternalRequestType internetRequestType) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    UserRequest* pInternalRequest = m_pInternalRequest[static_cast<int>(internetRequestType)];

    return &userRequest == pInternalRequest;
}

bool RequestManager::IsAwake() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_State == State::FullAwake || m_State == State::MinimumAwake;
}

bool RequestManager::IsShuttingdown() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_State == State::ArrangingToShutdown || m_State == State::ReadyToShutdown;
}

void RequestManager::ResetAllTemporaryExcludedFlags() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto&& userReuquest : m_RequestList)
    {
        userReuquest.ResetTemporaryExcludedFlag();
    }
}

nn::Result RequestManager::UpdateAggregatedRequest() NN_NOEXCEPT
{
    ResetAllTemporaryExcludedFlags();

    AggregatedRequestType aggregatedRequest;
    NN_RESULT_DO(AggregateSubmittedRequestsImpl(&aggregatedRequest));

    // スキャンは ConnectionSelector::Renew() と関係なく発生するので m_IsAggregatedReqeuestStaged と関係なくキャンセルする
    if (aggregatedRequest.networkType != NetworkType_Internet && aggregatedRequest.networkType != NetworkType_None)
    {
        CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::InternetNoLongerRequested>();
    }

    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_IsAggregatedRequestStaged)
    {
        if (m_AggregatedRequest.profileId != aggregatedRequest.profileId && aggregatedRequest.profileId != nn::util::InvalidUuid)
        {
            CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::NetworkProfileSpecified>();
        }
        if (aggregatedRequest.connectionConfirmationOption == ConnectionConfirmationOption_Prohibited)
        {
            CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::ConnectionConfirmationNoLongerRequested>();
        }
    }

    m_AggregatedRequest = aggregatedRequest;

    NN_RESULT_SUCCESS;
}

bool RequestManager::IsStable() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& userRequest : m_RequestList)
    {
        if (RequestState_Blocking == userRequest.GetRequestState())
        {
            return false;
        }
    }

    return true;
}

// 現在提出されているユーザーリクエストに、ネットワークリソースの状態を伝えます。
// ネットワークリソースの確保に成功したリクエストは受領され、失敗したリクエストは却下されます。
nn::Result RequestManager::Renew(
    const TotalNetworkResourceInfo& totalNetworkResourceInfo,
    const AggregatedRequestType& aggregatedRequestType,
    const AdditionalInfo& additionalInfo,
    nn::Result result
) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("RequestManager::Renew()\n");

    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto&& userRequest : m_RequestList)
    {
        auto requestState = userRequest.GetRequestState();

        if (requestState == RequestState_Free)
        {
            continue;
        }

        const Requirement& requirement = userRequest.GetRequirement();

        if (m_ExclusiveClientId.IsValid() && m_ExclusiveClientId != userRequest.GetClientId())
        {
            userRequest.Reject(ResultLocked());
        }
        else if (!m_IsBackgroundRequestEnabled && !userRequest.GetRequirement()._isForeground &&
            !IsSpecifiedInternalRequest(userRequest, InternalRequestType::SustainConnection))
        {
            userRequest.Reject(ResultBackgroundRequestProhibited());
        }
        else if (!IsAwake() && requestState == RequestState_OnHold)
        {
            // スリープ準備に入ったら OnHold な利用要求は次の起床までずっと保留する
        }
        else if (IsShuttingdown())
        {
            // シャットダウン準備に入ったら OnHold な利用要求は保留のまま
        }
        else if (m_State == State::MinimumAwake && userRequest.GetRequirement()._isForeground && requestState == RequestState_OnHold)
        {
            // MinimumAwake では、OnHold な FG 利用要求を保留
        }
        else if (userRequest.GetRevision() > aggregatedRequestType.lastRevision)
        {
            // 接続更新試行時より後に提出されたリクエストは保留
        }
        else if (totalNetworkResourceInfo.networkResourceState == NetworkResourceState::Lost ||
                 totalNetworkResourceInfo.isDisconnectionBlocking)  // Lost にしたいのにできていない状態
        {
            // TODO: profileId の考慮
            if (requestState == RequestState_Accepted &&
                requirement._networkType == totalNetworkResourceInfo.networkType)
            {
                userRequest.Reject(totalNetworkResourceInfo.connectionResult, additionalInfo);
            }

            // Lost は途中状態なので、リソースが失われた受理済み要求の却下のみをおこない、
            // 審査中の利用要求に対しては何もしない
        }
        else if (IsUserRequestAcceptable(userRequest, totalNetworkResourceInfo))
        {
            // 受理
            userRequest.Accept();
        }
        else
        {
            if (result.IsSuccess())
            {
                if (IsUserRequestCompatible(userRequest, aggregatedRequestType))
                {
                    // 要求に相乗りしたが、獲得した接続に相乗りできなかったのは、
                    // もっとも優先度の高い要求より強い疎通確認を求めて、疎通確認に失敗したとき
                    if (userRequest.GetPresentConnectionConfirmationOption() >= ConnectionConfirmationOption_Required &&
                        totalNetworkResourceInfo.internetResult.IsFailure())
                    {
                        userRequest.Reject(totalNetworkResourceInfo.internetResult, additionalInfo);
                    }
                    else
                    {
                        // 想定外
                        userRequest.Reject(ResultRequestRejected());    // TODO
                    }
                }
                else
                {
                    userRequest.Reject(ResultLowPriority());   // TODO
                }
            }
            else
            {
                if (IsUserRequestCompatible(userRequest, aggregatedRequestType))
                {
                    // もっとも優先度の高い要求と、それに相乗り可能な要求には、 Renew の失敗をそのまま渡す
                    userRequest.Reject(result, additionalInfo);
                }
                else
                {
                    // もっとも優先度の高い要求に相乗り不可能な要求は、再評価の価値があるので保留
                    // TODO: 前回の接続試行時と統合リクエストの内容が変化するようにリクエストを却下する
                }
            }
        }
    }

    if (m_State == State::ArrangingToSleep)
    {
        // スリープ突入時に維持するべきでない要求がすべて取り下げられたら準備完了へ遷移
        bool isReadyToSleep = true;

        for (const auto& userRequest : m_RequestList)
        {
            const auto state = userRequest.GetRequestState();

            if ((state == RequestState_Accepted || state == RequestState_Blocking) &&
                !userRequest.GetRequirement()._isKeptInSleep)
            {
                isReadyToSleep = false;
                break;
            }
        }

        if (isReadyToSleep)
        {
            m_State = State::ReadyToSleep;
        }
    }
    else if (m_State == State::ArrangingToShutdown)
    {
        // 全要求が取り下げられたらシャットダウン準備完了へ遷移
        bool isReadyToShutdown = true;

        for (const auto& userRequest : m_RequestList)
        {
            const auto state = userRequest.GetRequestState();

            if (state == RequestState_Accepted || state == RequestState_Blocking)
            {
                isReadyToShutdown = false;
                break;
            }
        }

        if (isReadyToShutdown)
        {
            m_State = State::ReadyToShutdown;
        }
    }

    NN_RESULT_DO(AggregateSubmittedRequestsImpl(&m_AggregatedRequest));

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

void RequestManager::CancelSustainedRequests() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    UserRequest* pInternalRequest = m_pInternalRequest[static_cast<int>(InternalRequestType::SustainConnection)];

    if (pInternalRequest != nullptr)
    {
        Cancel(pInternalRequest);
    }
}

int RequestManager::GetSocketInfoToKeepInSleep(SocketInfo* pOutSocketInfoArray, int count) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    int outCount = 0;

    for (auto&& userRequest : m_RequestList)
    {
        if (userRequest.GetRequestState() == RequestState_Accepted)
        {
            auto socketDescripter = userRequest.GetSocketDescriptor();
            auto processId = userRequest.GetClientProcessId();

            if (socketDescripter >= 0 && processId > 0)
            {
                if (outCount < count)
                {
                    pOutSocketInfoArray[outCount].socketDescriptor = socketDescripter;
                    pOutSocketInfoArray[outCount].processId = processId;
                }

                ++outCount;
            }
        }
    }

    return outCount;
}

nn::Result RequestManager::PutToSleep(bool isPartial) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("PutToSleep\n");

    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(IsAwake(), ResultInternalError());   // TODO: [TORIAEZU]

    for (auto&& userRequest : m_RequestList)
    {
        auto requestState = userRequest.GetRequestState();

        // isPartial が真の場合、 _isKeptInSleep が真の要求を却下対象から除外
        if (requestState == RequestState_Accepted && !(isPartial && userRequest.GetRequirement()._isKeptInSleep))
        {
            userRequest.Reject(ResultDevicePutToSleep());
        }
    }

    m_State = State::ArrangingToSleep;

    NN_RESULT_DO(UpdateAggregatedRequest());
    m_SubmitEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result RequestManager::FullWakeUp() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_DETAIL_NIFM_INFO("RequestManager::FullWakeUp, %d\n", m_State);

    NN_RESULT_THROW_UNLESS(m_State == State::MinimumAwake || m_State == State::ReadyToSleep, ResultInternalError());    // TODO: [TORIAEZU]

    m_State = State::FullAwake;

    NN_RESULT_DO(UpdateAggregatedRequest());
    m_SubmitEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result RequestManager::MinimumWakeUp() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_DETAIL_NIFM_INFO("RequestManager::MinimumWakeUp, %d\n", m_State);

    NN_RESULT_THROW_UNLESS(m_State == State::ReadyToSleep || m_State == State::FullAwake, ResultInternalError());    // TODO: [TORIAEZU]

    for (auto&& userRequest : m_RequestList)
    {
        auto requestState = userRequest.GetRequestState();

        // MinimumAwake 状態では、FG からの利用要求を却下
        if (requestState == RequestState_Accepted && userRequest.GetRequirement()._isForeground)
        {
            userRequest.Reject(ResultDevicePutToSleep());
        }
    }

    m_State = State::MinimumAwake;

    NN_RESULT_DO(UpdateAggregatedRequest());
    m_SubmitEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result RequestManager::Shutdown() NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("RequestManager::Shutdown\n");

    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(IsAwake(), ResultInternalError());   // TODO: [TORIAEZU]

    for (auto&& userRequest : m_RequestList)
    {
        userRequest.Reject(ResultDeviceShutdown());
    }

    m_State = State::ArrangingToShutdown;

    NN_RESULT_DO(UpdateAggregatedRequest());
    m_SubmitEvent.Signal();

    NN_RESULT_SUCCESS;
}

bool RequestManager::IsReadyToSleep() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_State == State::ReadyToSleep;
}

bool RequestManager::IsReadyToShutdown() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_State == State::ReadyToShutdown;
}

bool RequestManager::IsFullAwaked() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_State == State::FullAwake;
}

nn::Result RequestManager::SetExclusiveClient(ClientId clientId) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_DETAIL_NIFM_INFO_V1("SetExclusiveClient(%d)\n", clientId.value);

    if (m_ExclusiveClientId != clientId)
    {
        m_ExclusiveClientId = clientId;
        NN_RESULT_DO(UpdateAggregatedRequest());
        m_SubmitEvent.Signal();
    }

    NN_RESULT_SUCCESS;
}

nn::Result RequestManager::SetBackgroundRequestEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_DETAIL_NIFM_INFO_V1("SetBackgroundRequestEnabled(%s)\n", isEnabled ? "true" : "false");

    if (m_IsBackgroundRequestEnabled != isEnabled)
    {
        m_IsBackgroundRequestEnabled = isEnabled;
        NN_RESULT_DO(UpdateAggregatedRequest());
        m_SubmitEvent.Signal();
    }

    NN_RESULT_SUCCESS;
}

void RequestManager::Dump() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_DETAIL_NIFM_INFO_V1("--------------RequestManager-------------\n");
    for( const auto& userRequest : m_RequestList )
    {
        userRequest.Dump();
    }
    NN_DETAIL_NIFM_INFO_V1("-----------------------------------------\n");
}

bool RequestManager::IsProhibitedConnectionConfirmationOptionRequestAvailable() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (const auto& userRequest : m_RequestList)
    {
        if(userRequest.GetRequestState() == RequestState_Accepted)
        {
            const auto& requirement = userRequest.GetRequirement();
            if( true
                && requirement._networkType == NetworkType_Internet
                && requirement._connectionConfirmationOption == ConnectionConfirmationOption_Prohibited
                )
            {
                return true;
            }
        }
    }

    return false;
}

}
}
}
