﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nifm/detail/core/nifm_PowerStateCoordinator.h>

#include <nn/nifm/detail/core/nifm_RequestManager.h>
#include <nn/nifm/detail/core/nifm_ConnectionSelector.h>
#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterfaceManager.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/util/util_LockGuard.h>


namespace nn
{
namespace nifm
{
namespace detail
{

PowerStateCoordinator::PowerStateCoordinator(
    RequestManager* pRequestManager,
    ConnectionSelector* pConnectionSelector,
    EthernetInterfaceManager* pEthernetInterfaceManager) NN_NOEXCEPT
    : m_pRequestManager(pRequestManager),
      m_pConnectionSelector(pConnectionSelector),
      m_pEthernetInterfaceManager(pEthernetInterfaceManager),
      m_PowerStateEventCallback(this),
      m_CurrentState(State::FullAwake),
      m_GoalState(State::FullAwake),
      m_DispatchTriggerEvent(nn::os::EventClearMode_AutoClear),
      m_DispatchTriggerEventHandler(m_DispatchTriggerEvent)
{
    const nn::psc::PmModuleId pmDependencies[] = {
        nn::psc::PmModuleId_Pcie,           // wlan
        nn::psc::PmModuleId_SocketWlan,     // wlan
        nn::psc::PmModuleId_Usb,            // eth
        nn::psc::PmModuleId_Eth,            // eth
        nn::psc::PmModuleId_Socket,         // bscsocket, tsc,
        nn::psc::PmModuleId_Fs,             // settings
        nn::psc::PmModuleId_Btm,            // ldn
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_PmModule.Initialize(
            nn::psc::PmModuleId_Nifm,                   // this module's ID
            pmDependencies,                             // list of ID's for dependencies
            NN_ARRAY_SIZE(pmDependencies),      // number of ID's in dependency list
            nn::os::EventClearMode_AutoClear            // clear mode for state change event
        )
    );

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.GetRequest(&m_PmState, &m_PmFlagSet));

    m_PowerStateEventHandler.emplace(*m_PmModule.GetEventPointer());
    m_PowerStateEventHandler->Register(&m_PowerStateEventCallback);
}

PowerStateCoordinator::~PowerStateCoordinator() NN_NOEXCEPT
{
}

PowerStateCoordinator::State PowerStateCoordinator::GetGoalState() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    return m_GoalState;
}

void PowerStateCoordinator::PowerStateEventCallback() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.GetRequest(&m_PmState, &m_PmFlagSet));

    NN_DETAIL_NIFM_TRACE("PmState: %d\n", m_PmState);

    // PM modules are expected to handle at least
    // PmState_FullAwake
    // PmState_MinimumAwake
    // PmState_SleepReady
    switch (m_PmState)
    {
    case nn::psc::PmState_FullAwake:
        if (m_GoalState != State::FullAwake)
        {
            m_GoalState = State::FullAwake;
            m_DispatchTriggerEvent.Signal();
        }
        // MinimumAwake->FullAwake は保留されていた FG 要求が評価されるようになるだけなのですぐに Acknowledge
        if (m_CurrentState == State::MinimumAwake)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(m_PmState, ResultSuccess()));
            NN_DETAIL_NIFM_TRACE("Acknowledged in PowerStateEventCallback: %d\n", m_PmState);
            m_CurrentState = State::FullAwake;
        }
        break;

    case nn::psc::PmState_MinimumAwake:
        if (m_GoalState != State::MinimumAwake)
        {
            m_GoalState = State::MinimumAwake;
            m_DispatchTriggerEvent.Signal();
        }
        // FullAwake->MinimumAwake は停留せずに SleepReady に移るので、ここではすぐに Acknowledge
        if (m_CurrentState == State::FullAwake)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(m_PmState, ResultSuccess()));
            NN_DETAIL_NIFM_TRACE("Acknowledged in PowerStateEventCallback: %d\n", m_PmState);
            m_CurrentState = State::MinimumAwake;
        }
        break;

    case nn::psc::PmState_SleepReady:
        if (m_GoalState != State::Sleep)
        {
            m_GoalState = State::Sleep;
            CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::SleepRequired>();
            m_DispatchTriggerEvent.Signal();
        }
        break;

    case nn::psc::PmState_ShutdownReady:
        if (m_GoalState != State::Shutdown)
        {
            m_GoalState = State::Shutdown;
            CancelFlagHolder::GetSingleton().RequireCancel<CancelFlagHolder::ReasonFlag::ShutdownRequired>();
            m_DispatchTriggerEvent.Signal();
        }
        break;

    default:
        // no-op, but still return ResultSuccess()
        // for unhandled states - new states may be
        // added in the future.
        m_PmModule.Acknowledge(m_PmState, ResultSuccess());
        break;
    }
}

void PowerStateCoordinator::AcknowledgeSleep() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_PmState == nn::psc::PmState_SleepReady && m_CurrentState != State::Sleep)
    {
        NN_SDK_ASSERT(m_CurrentState == State::MinimumAwake || m_CurrentState == State::FullAwake);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(m_PmState, ResultSuccess()));
        NN_DETAIL_NIFM_TRACE("Acknowledged in DispatchPuttingNetworkInterfacesToSleep: %d\n", m_PmState);
        m_CurrentState = State::Sleep;
    }
}

void PowerStateCoordinator::AcknowledgeAwake() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_PmState == nn::psc::PmState_FullAwake && m_CurrentState != State::FullAwake)
    {
        NN_SDK_ASSERT(m_CurrentState == State::Sleep); // MimimumAwake -> FullAwake の Acknowledge は PowerStateEventCallback で処理される

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(m_PmState, ResultSuccess()));
        NN_DETAIL_NIFM_TRACE("Acknowledged in DispatchReadyToSleep: %d\n", m_PmState);
        m_CurrentState = State::FullAwake;
    }
    else if(m_PmState == nn::psc::PmState_MinimumAwake && m_CurrentState != State::MinimumAwake)
    {
        NN_SDK_ASSERT(m_CurrentState == State::Sleep);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(m_PmState, ResultSuccess()));
        NN_DETAIL_NIFM_TRACE("Acknowledged in DispatchReadyToSleep: %d\n", m_PmState);
        m_CurrentState = State::MinimumAwake;
    }
}

void PowerStateCoordinator::AcknowledgeShutdown() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_PmState == nn::psc::PmState_ShutdownReady && m_CurrentState != State::Shutdown)
    {
        NN_SDK_ASSERT(m_CurrentState == State::MinimumAwake || m_CurrentState == State::FullAwake);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(m_PmState, ResultSuccess()));
        NN_DETAIL_NIFM_TRACE("Acknowledged in DispatchWaitingForConnectionFree: %d\n", m_PmState);
        m_CurrentState = State::Shutdown;
    }
}

nn::Result PowerStateCoordinator::PutToSleep() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_CurrentState != State::Shutdown, ResultInternalError());
    NN_RESULT_THROW_UNLESS(m_GoalState != State::Shutdown, ResultInternalError());

    m_GoalState = State::Sleep;
    m_DispatchTriggerEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result PowerStateCoordinator::WakeUp() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_CurrentState != State::Shutdown, ResultInternalError());
    NN_RESULT_THROW_UNLESS(m_GoalState != State::Shutdown, ResultInternalError());

    switch (m_GoalState)
    {
    case nn::nifm::detail::PowerStateCoordinator::State::MinimumAwake:
        m_GoalState = State::FullAwake;
        break;
    case nn::nifm::detail::PowerStateCoordinator::State::Sleep:
        m_GoalState = State::MinimumAwake;
        break;
    default:
        NN_DETAIL_NIFM_WARN("PowerStateCoordinator::WakeUp, m_PmState:%d\n", m_PmState);
        m_GoalState = State::FullAwake;
        break;
    }
    m_DispatchTriggerEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result PowerStateCoordinator::Shutdown() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_DETAIL_NIFM_TRACE("m_CurrentState = %d, m_GoalState = %d\n", m_CurrentState, m_GoalState);
    NN_RESULT_THROW_UNLESS(m_CurrentState == State::MinimumAwake || m_CurrentState == State::FullAwake, ResultInternalError());

    m_GoalState = State::Shutdown;
    m_DispatchTriggerEvent.Signal();

    NN_RESULT_SUCCESS;
}

}
}
}
