﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/networkInterface/nifm_WirelessInterface.horizon.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_WirelessAccessPoint.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>
#include <nn/nifm/detail/core/connectionConfirmation/nifm_AuthenticationClient.h>
#include <nn/nifm/detail/util/nifm_WlanUtility.h>
#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/nifm/detail/util/nifm_DebugUtility.h>
#include <nn/nifm/detail/util/nifm_SocketHolder.h>

#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_Types.h>

#include <nn/socket.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_ScopeExit.h>

#include <cstring>
#include <new>
#include <mutex>

namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{

inline void InAddrToWlanIpv4Address(nn::wlan::WlanIpv4Address* pOutWlanIpv4Address, nn::socket::InAddr inAddr) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutWlanIpv4Address);

    const Bit8* pInAddr = reinterpret_cast<const Bit8*>(&inAddr);

    // ともにネットワークバイトオーダーで格納される
    pOutWlanIpv4Address->addr[0] = pInAddr[0];
    pOutWlanIpv4Address->addr[1] = pInAddr[1];
    pOutWlanIpv4Address->addr[2] = pInAddr[2];
    pOutWlanIpv4Address->addr[3] = pInAddr[3];
}

inline nn::socket::InAddr IpV4AddressToInAddr(const IpV4Address ipV4Address) NN_NOEXCEPT
{
    nn::socket::InAddr inAddr = { (static_cast<uint32_t>(ipV4Address.data[0]) <<  0) |
                                  (static_cast<uint32_t>(ipV4Address.data[1]) <<  8) |
                                  (static_cast<uint32_t>(ipV4Address.data[2]) << 16) |
                                  (static_cast<uint32_t>(ipV4Address.data[3]) << 24) };

    return inAddr;
}

bool IsRetryNeeded(Result result)
{
    if (// CauseOfInfo_NoBss
        result <= ResultBssNotFound()
        )
    {
        return true;
    }

    return false;
}

}

void WirelessInterface::WlanConnectionEventCallback::ExecuteImpl() NN_NOEXCEPT
{
    nn::wlan::ConnectionStatus connectionStatus;
    nn::Result result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);

    NN_SDK_ASSERT(result.IsSuccess());
    if (result.IsFailure())
    {
        return;
    }

    // 意図しない切断のシグナル以外は ConnectImpl, DisconnectImpl でクリアされるはず
    NN_SDK_ASSERT_EQUAL(connectionStatus.state, nn::wlan::ConnectionState_Idle);
    if (connectionStatus.state != nn::wlan::ConnectionState_Idle)
    {
        return;
    }

    m_pWirelessInterface->NotifyLinkDisconnected(ConvertConnectionStatusToResult(connectionStatus));
}

/*** ScanRunner ***/

void WirelessInterface::ScanRunner::ThreadFunc(void* pContext) NN_NOEXCEPT
{
    reinterpret_cast<WirelessInterface::ScanRunner*>(pContext)->ThreadFunc();
}

void WirelessInterface::ScanRunner::ThreadFunc() NN_NOEXCEPT
{
    m_ScanResult = m_WirelessInterface.ExecuteScan(m_pScanBuffer, m_ScanBufferSize, m_pAccessPoint);
    m_Event.Signal();
}

WirelessInterface::ScanRunner::ScanRunner(WirelessInterface& wirelessInterface, const AccessPointData* pAccessPoint, size_t scanBufferSize) NN_NOEXCEPT
    : m_WirelessInterface(wirelessInterface),
      m_pAccessPoint(pAccessPoint),
      m_pScanBuffer(nullptr),
      m_ScanBufferSize(scanBufferSize),
      m_pThreadStack(nullptr),
      m_Thread(),
      m_Event(nn::os::EventClearMode_AutoClear),
      m_ScanResult(nn::ResultSuccess()),
      m_CancelReasonResult(nn::ResultSuccess())
{
}

WirelessInterface::ScanRunner::~ScanRunner() NN_NOEXCEPT
{
    if (m_pScanBuffer != nullptr)
    {
        delete[] m_pScanBuffer;
    }
    if (m_pThreadStack != nullptr)
    {
        delete[] m_pThreadStack;
    }
}

nn::Result WirelessInterface::ScanRunner::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pScanBuffer == nullptr);
    m_pScanBuffer = new(std::nothrow) char[m_ScanBufferSize];
    NN_RESULT_THROW_UNLESS(m_pScanBuffer != nullptr, ResultOutOfMemory());

    NN_SDK_ASSERT(m_pThreadStack == nullptr);
    m_pThreadStack = new(std::nothrow) char[ThreadStackSize + nn::os::ThreadStackAlignment - 1];
    NN_RESULT_THROW_UNLESS(m_pThreadStack != nullptr, ResultOutOfMemory());

    void* pThreadStack = reinterpret_cast<void*>(nn::util::align_up(reinterpret_cast<uintptr_t>(m_pThreadStack), nn::os::ThreadStackAlignment));
    NN_RESULT_DO(nn::os::CreateThread(&m_Thread, ThreadFunc, this, pThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(nifm, Scan)));

    nn::os::SetThreadName(&m_Thread, NN_SYSTEM_THREAD_NAME(nifm, Scan));
    nn::os::StartThread(&m_Thread);

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ScanRunner::Cancel(nn::Result result) NN_NOEXCEPT
{
    m_CancelReasonResult = result;
    NN_RESULT_THROW(m_WirelessInterface.CancelScan());
}

nn::Result WirelessInterface::ScanRunner::Wait() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    NN_SDK_ASSERT_NOT_NULL(m_pThreadStack);
    delete[] m_pThreadStack;
    m_pThreadStack = nullptr;

    // nn::wlan::Infra::StartScan() は中断されても
    // その時点までのスキャン結果と成功を返す仕様であり、
    // 成功後に内部状態が変化する副作用もないので、
    // キャンセル理由に基づく失敗で結果を上書き可能
    NN_RESULT_DO(m_CancelReasonResult);

    NN_RESULT_THROW(m_ScanResult);
}

nn::Result WirelessInterface::ScanRunner::ExportTo(AccessPointListBase* pAccessPointList) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pAccessPointList);

    m_WirelessInterface.InterpretScanData(pAccessPointList, m_pScanBuffer);

    NN_RESULT_SUCCESS;
}

/*** ConnectRunner ***/

void WirelessInterface::ConnectRunner::ThreadFunc(void* pContext) NN_NOEXCEPT
{
    reinterpret_cast<WirelessInterface::ConnectRunner*>(pContext)->ThreadFunc();
}

void WirelessInterface::ConnectRunner::ThreadFunc() NN_NOEXCEPT
{
    AccessPointData accessPoint;
    m_WirelessAccessPoint.TryExport(&accessPoint);

    NetworkProfileData networkProfile;
    m_NetworkProfile.Export(&networkProfile);

    m_ConnectResult = m_WirelessInterface.ExecuteConnect(accessPoint, networkProfile);
    m_Event.Signal();
}

WirelessInterface::ConnectRunner::ConnectRunner(WirelessInterface& wirelessInterface, const WirelessAccessPoint& wirelessAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
    : m_WirelessInterface(wirelessInterface),
      m_WirelessAccessPoint(wirelessAccessPoint),
      m_NetworkProfile(networkProfile),
      m_pThreadStack(nullptr),
      m_Thread(),
      m_Event(nn::os::EventClearMode_AutoClear),
      m_ConnectResult(ResultLinkLayerNotAvailable()),
      m_CancelReasonResult(ResultSuccess())
{
}

WirelessInterface::ConnectRunner::~ConnectRunner() NN_NOEXCEPT
{
    if (m_pThreadStack != nullptr)
    {
        delete[] m_pThreadStack;
    }
}

nn::Result WirelessInterface::ConnectRunner::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pThreadStack == nullptr);
    m_pThreadStack = new(std::nothrow) char[ThreadStackSize + nn::os::ThreadStackAlignment - 1];
    NN_RESULT_THROW_UNLESS(m_pThreadStack != nullptr, ResultOutOfMemory());

    void* pThreadStack = reinterpret_cast<void*>(nn::util::align_up(reinterpret_cast<uintptr_t>(m_pThreadStack), nn::os::ThreadStackAlignment));
    NN_RESULT_DO(nn::os::CreateThread(&m_Thread, ThreadFunc, this, pThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(nifm, Connect)));

    nn::os::SetThreadName(&m_Thread, NN_SYSTEM_THREAD_NAME(nifm, Connect));
    nn::os::StartThread(&m_Thread);

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ConnectRunner::Cancel(nn::Result result) NN_NOEXCEPT
{
    m_CancelReasonResult = result;
    NN_RESULT_THROW(m_WirelessInterface.CancelConnect());
}

nn::Result WirelessInterface::ConnectRunner::Wait() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    NN_SDK_ASSERT_NOT_NULL(m_pThreadStack);
    delete[] m_pThreadStack;
    m_pThreadStack = nullptr;

    if (m_ConnectResult.IsSuccess())
    {
        // 成功を失敗に捻じ曲げると後処理が必要になるので、
        // キャンセル要求があったとしても次のキャンセルポイントへ進める
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(m_CancelReasonResult);

    NN_RESULT_THROW(m_ConnectResult);
}

void WirelessInterface::GetAllMacAddresses( MacAddress* pOutMacAddresses, int* pOutCount, int inCount ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutMacAddresses);
    NN_SDK_ASSERT_NOT_NULL(pOutCount);

    nn::wlan::MacAddress macAddress;
    nn::Result result = nn::wlan::Infra::GetMacAddress(&macAddress);

    NN_DETAIL_NIFM_INFO(
        "MAC Address of Wireless I/F: %02x:%02x:%02x:%02x:%02x:%02x\n",
        macAddress.GetMacAddressData()[0], macAddress.GetMacAddressData()[1], macAddress.GetMacAddressData()[2],
        macAddress.GetMacAddressData()[3], macAddress.GetMacAddressData()[4], macAddress.GetMacAddressData()[5]
    );

    if( result.IsSuccess() )
    {
        if( inCount >= 1 )
        {
            std::memcpy(pOutMacAddresses[0].data, macAddress.GetMacAddressData(), sizeof(pOutMacAddresses[0].data));
        }
        *pOutCount = 1;
    }
    else
    {
        *pOutCount = 0;
    }

    return;
}

nn::Result WirelessInterface::GetAllowedChannels(int16_t(&pOutChannels)[WirelessChannelsCountMax], int *pOutCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);

    uint32_t outCount;
    NN_RESULT_DO(nn::wlan::Infra::GetAllowedChannels(pOutChannels, &outCount));
    *pOutCount = outCount;

    NN_RESULT_SUCCESS;
}

WirelessInterface::WirelessInterface() NN_NOEXCEPT
    : m_IsInitialized(false),
      m_MacAddress(InvalidMacAddress),
      m_OpenInfraModeCount(0),
      m_IsCloseInfraModeNeeded(false),
      m_ScanCompleteEvent(nn::os::EventClearMode_AutoClear),
      m_WlanConnectionEvent(nn::util::nullopt),
      m_WlanConnectionEventCallback(nn::util::nullopt),
      m_WlanConnectionEventHandler(nn::util::nullopt),
      m_ScanResult(ResultSuccess()),
      m_ConnectionAttemptResult(ResultSuccess()),
      m_Id(-1), // TODO: TORIAEZU
      m_PassivelyScannedAccessPointList(nn::TimeSpan::FromSeconds(ScanResultLifeTimeInSeconds)),    // 過去のスキャン結果をマージをおこなうので寿命を指定
      m_ActiveScanResults(m_ActivelyScannedAccessPointList),
      m_PassiveScanResults(m_PassivelyScannedAccessPointList),
      m_pLastScannedResults(&m_ActiveScanResults),    // この指定により m_LastSuccessfulScanTime に関わらずアクティブスキャンから始まる
      m_ActiveScanIndex(0),
      m_ResultHealth(nn::ResultSuccess()),
      m_ScanChannelCount(InvalidScanChannelCount),
      m_ScanChannels()
{
    NN_STATIC_ASSERT(static_cast<int>(nn::nifm::LinkLevel_0) == static_cast<int>(nn::wlan::LinkLevel_0));
    NN_STATIC_ASSERT(static_cast<int>(nn::nifm::LinkLevel_1) == static_cast<int>(nn::wlan::LinkLevel_1));
    NN_STATIC_ASSERT(static_cast<int>(nn::nifm::LinkLevel_2) == static_cast<int>(nn::wlan::LinkLevel_2));
    NN_STATIC_ASSERT(static_cast<int>(nn::nifm::LinkLevel_3) == static_cast<int>(nn::wlan::LinkLevel_3));

    NN_STATIC_ASSERT(nn::nifm::WirelessChannelsCountMax == nn::wlan::WirelessChannelsCountMax);
}

WirelessInterface::~WirelessInterface() NN_NOEXCEPT
{
    // WirelessInterface が破棄されるまでにすべての要求は却下され、
    // 無線デバイスのインフラ通信モードも終了しているべきだが、
    // 終了処理は念のためにおこなっておく

    NN_SDK_ASSERT(m_OpenInfraModeCount == 0);
    if (m_OpenInfraModeCount > 0)
    {
        nn::wlan::Infra::CloseMode();
    }

    NN_SDK_ASSERT(m_WlanConnectionEventHandler);
    if (m_WlanConnectionEventHandler)
    {
        RemoveEventHandler(*m_WlanConnectionEventHandler);
    }

    NN_SDK_ASSERT(m_WlanConnectionEventCallback);
    NN_SDK_ASSERT(m_WlanConnectionEvent);
}

nn::Result WirelessInterface::Initialize() NN_NOEXCEPT
{
    return Initialize( InvalidMacAddress );
}

nn::Result WirelessInterface::Initialize( const MacAddress& macAddress ) NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    if( !IsInitialized() )
    {
        NN_RESULT_DO(m_IpConfiguration.Initialize(GetInterfaceName(), NetworkInterfaceType_Ieee80211));

        m_MacAddress = macAddress;

        m_IsInitialized = true;
    }

    NN_RESULT_THROW(result);
}

bool WirelessInterface::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

bool WirelessInterface::IsAvailable() const NN_NOEXCEPT
{
    // TODO: handle の有効性もチェック
    return m_IsInitialized;
}

MacAddress* WirelessInterface::GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT
{
    std::memcpy( pOutMacAddress->data, m_MacAddress.data, MacAddress::Size );
    return pOutMacAddress;
}

nn::Result WirelessInterface::UpdateRssiAndLinkLevel(AccessPointData* pInOutAccessPointData) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInOutAccessPointData);

    int rssi;
    NN_RESULT_DO(nn::wlan::Infra::GetRssi(&rssi));

    pInOutAccessPointData->rssi = rssi;
    pInOutAccessPointData->linkLevel = static_cast<nn::nifm::LinkLevel>(nn::wlan::Infra::ConvertRssiToLinkLevel(rssi));

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ExecuteScan(void* pScanBuffer, size_t size, const AccessPointData* pInAccessPoint) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pScanBuffer);

    nn::wlan::ScanParameters scanParameters;
    nn::wlan::Ssid ssid;

    if (pInAccessPoint == nullptr)
    {
        scanParameters.scanType = nn::wlan::ScanType_Passive;
        scanParameters.channelCount = m_ScanChannelCount == InvalidScanChannelCount ? 0 : m_ScanChannelCount;
        for (int i = 0; i < m_ScanChannelCount; ++i)
        {
            scanParameters.channelList[i] = m_ScanChannels[i];
        }
        scanParameters.ssidList = nullptr;                                          // all
        scanParameters.ssidCount = 0;                                               // all
        scanParameters.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();   // all
    }
    else
    {
        ssid.Set(pInAccessPoint->ssid.hex, pInAccessPoint->ssid.length);

        scanParameters.scanType = nn::wlan::ScanType_Active;
        scanParameters.channelCount = 1;
        scanParameters.channelList[0] = pInAccessPoint->channel;
        scanParameters.ssidList = &ssid;
        scanParameters.ssidCount = 1;
        scanParameters.bssid.Set(pInAccessPoint->bssid.data);
    }

    scanParameters.channelScanTime = -1;    // default(110ms)
    scanParameters.homeChannelTime = -1;    // default(45ms)

    NN_RESULT_DO(OpenInfraMode());

    NN_UTIL_SCOPE_EXIT
    {
        CloseInfraMode();
    };

    NN_DETAIL_NIFM_INFO("nn::wlan::Infra::StartScan. (%d)\n", scanParameters.channelCount);
    NN_RESULT_THROW(nn::wlan::Infra::StartScan(pScanBuffer, size, scanParameters));
}

nn::Result WirelessInterface::CancelScan() NN_NOEXCEPT
{
    NN_RESULT_THROW(nn::wlan::Infra::StopScan());
}

void WirelessInterface::InterpretScanData(AccessPointListBase* pOutAccessPointList, const void* pScanBuffer) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());

    nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
    uint32_t findNum = resultReader.GetCount();
    NN_DETAIL_NIFM_INFO("resultReader.GetCount: %d\n", findNum);
    for (uint32_t i = 0; i < findNum; i++)
    {
        WirelessAccessPoint wirelessAccessPoint(this);
        ReadAccessPointFromBeaconDescriptionReader(&wirelessAccessPoint, resultReader.GetNextDescription());

        auto pWirelessAccessPoint = pOutAccessPointList->CreateTempAndUpdate<WirelessAccessPoint>(wirelessAccessPoint);

        while (pWirelessAccessPoint == nullptr && pOutAccessPointList->DeleteOldest())
        {
            NN_DETAIL_NIFM_TRACE("DeleteOldest\n");
            pWirelessAccessPoint = pOutAccessPointList->CreateTempAndUpdate<WirelessAccessPoint>(wirelessAccessPoint);
        }

        if (pWirelessAccessPoint == nullptr)
        {
            NN_DETAIL_NIFM_WARN("WirelessInterface::InterpretScanData() - pOutAccessPointList is full.\n");
            break;
        }
    }

    pOutAccessPointList->Commit();
}

nn::Result WirelessInterface::ExecuteConnect(const AccessPointData& accessPoint, const NetworkProfileData& networkProfile) NN_NOEXCEPT
{
    nn::wlan::Ssid ssid;
    ConvertToWlanSsidFromNifm(&ssid, networkProfile.wirelessSetting.ssidConfig.ssid);

    nn::wlan::MacAddress bssid;
    ConvertToWlanBssidFromNifm(&bssid, accessPoint.bssid);

    nn::wlan::Security security;
    ConvertToWlanSecurityFromNifm(&security, networkProfile, accessPoint);

    int16_t channel = accessPoint.channel;

    nn::Result result = ResultLinkLayerNotAvailable();

    for (int i = 0; i < RetryConnectCountMax; ++i)
    {
        Dump(ssid, bssid, channel, security);

        NN_DETAIL_NIFM_INFO("nn::wlan::Infra::Connect started.\n");
        result = nn::wlan::Infra::Connect(ssid, bssid, channel, security, true, 10);
        NN_DETAIL_NIFM_TRACE("nn::wlan::Infra::Connect ended.\n");

        // nn::wlan::Infra::Connect() は事前条件違反以外では失敗せず、
        // 接続試行の結果は nn::wlan::Infra::GetConnectionStatus() から取得する
        if (result.IsSuccess())
        {
            nn::wlan::ConnectionStatus connectionStatus;
            result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);

            if (result.IsSuccess())
            {
                result = ConvertConnectionStatusToResult(connectionStatus);
            }
        }

        if (CancelFlagHolder::GetSingleton().ConfirmInternetAdmitted(NetworkInterfaceType_Ieee80211).IsSuccess() && IsRetryNeeded(result))
        {
            // 電波環境によって応答を落とすことがあるので、成功率を上げるためにリトライする
            continue;
        }

        break;
    }

    if (result <= ResultBssNotFound())
    {
        if (networkProfile.wirelessSetting.ssidConfig.nonBroadcast && WirelessAccessPoint::IsStealth(accessPoint.ssid))
        {
            result = ResultBssNotFoundStealthIncluded();
        }
        else
        {
            result = ResultBssNotFoundStealthNotIncluded();
        }
    }

    NN_RESULT_THROW(result);
}

nn::Result WirelessInterface::CancelConnect() NN_NOEXCEPT
{
    NN_RESULT_THROW(nn::wlan::Infra::CancelConnect());
}

nn::Result WirelessInterface::Scan() NN_NOEXCEPT
{
    NN_RESULT_THROW(Scan(nullptr, nullptr));
}

nn::Result WirelessInterface::Scan(AccessPointListBase* pOutAccessPointList, const AccessPointData* pAccessPoint) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsEnabled(), ResultNetworkInterfaceNotAvailable());

    NN_RESULT_DO(CancelFlagHolder::GetSingleton().ConfirmScanAdmitted());

    ScanRunner scanRunner(*this, pAccessPoint);
    NN_RESULT_DO(scanRunner.Start());

    nn::Result cancelResult = ResultInternalError();
    while (!scanRunner.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(100)))
    {
        auto result = CancelFlagHolder::GetSingleton().ConfirmScanAdmitted();

        if (result.IsFailure() && cancelResult.IsFailure())
        {
            cancelResult = scanRunner.Cancel(result);
            NN_DETAIL_NIFM_TRACE_V3("Cancel scan: %s (%x).\n",
                cancelResult.IsSuccess() ? "Success" : "Failed", cancelResult.GetInnerValueForDebug());
        }
    }

    auto result = scanRunner.Wait();

    NN_UTIL_LOCK_GUARD(m_Mutex);

    ScannedResultsHolder* pScannedResultsHolder = nullptr;

    if (pAccessPoint == nullptr)
    {
        pScannedResultsHolder = &m_PassiveScanResults;
        // パッシブスキャンでは、取りこぼしへの対処として、前回までのスキャン結果をクリアしない
    }
    else
    {
        pScannedResultsHolder = &m_ActiveScanResults;
        pScannedResultsHolder->GetAccessPointListPointer()->Clear();
    }

    NN_SDK_ASSERT_NOT_NULL(pScannedResultsHolder);

    if (result.IsSuccess())
    {
        result = scanRunner.ExportTo(pScannedResultsHolder->GetAccessPointListPointer());
    }

    if (result.IsSuccess())
    {
        if (pAccessPoint != nullptr || m_ScanChannelCount == 0) // パッシブスキャンの場合は、全チャンネルスキャン時にのみタイムスタンプ更新
        {
            pScannedResultsHolder->SetTimestamp(nn::os::GetSystemTick().ToTimeSpan());
        }
        pScannedResultsHolder->SetValid(true);

        if (pOutAccessPointList != nullptr)
        {
            pScannedResultsHolder->GetAccessPointList().ExportTo(pOutAccessPointList);
        }
    }
    else
    {
        pScannedResultsHolder->SetValid(false);
    }

    m_pLastScannedResults = pScannedResultsHolder;

    NN_RESULT_THROW(result);
}

nn::Result WirelessInterface::GetScanData(AccessPointListBase* pOutAccessPointList) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_PassivelyScannedAccessPointList.ExportTo(pOutAccessPointList);

    NN_RESULT_SUCCESS;
}

bool WirelessInterface::UpdateAccessPointList(AccessPointListBase* pOutAccessPointList) NN_NOEXCEPT
{
    AccessPointData* pAccessPoint;
    bool isScanNeeded = true;

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        // 前回のスキャンから一定時間たっていたら最初からやり直す (ClearAccessPointList)
        // まずアクティブスキャンをおこなって、直近でつないだアクセスポイントを探す
        // アクティブスキャンで接続先が見つからなかったら全チャンネルのパッシブスキャンをする

        if (m_ActiveScanIndex < 0)
        {
            return false;
        }

        if (m_ActiveScanIndex < m_AccessPointHistory.GetCount())
        {
            pAccessPoint = &m_AccessPointHistory[m_ActiveScanIndex];
            ++m_ActiveScanIndex;
            if (pAccessPoint != nullptr)
            {
                isScanNeeded = !CopyToAccessPointListFromAccessPointData(pOutAccessPointList, *pAccessPoint);
            }
        }
        else
        {
            pAccessPoint = nullptr;     // パッシブスキャン
            m_ActiveScanIndex = -1;     // 次回以降はスキャンしない
        }
    }

    if (isScanNeeded && Scan(pOutAccessPointList, pAccessPoint).IsFailure())
    {
        return false;
    }

    return true;
}

nn::Result WirelessInterface::GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    // TODO: 更新との排他を図る
    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    NN_UTIL_LOCK_GUARD(m_Mutex);

    bool isScanDataExpired = nn::os::GetSystemTick().ToTimeSpan() - m_pLastScannedResults->GetTimestamp() >= timeSpan;

    if (isScanDataExpired || !m_pLastScannedResults->IsValid())
    {
        m_ActiveScanIndex = 0;      // UpdateAccessPointList() が呼ばれたときにスキャンを最初からやり直す
    }
    else if (m_pLastScannedResults == &m_PassiveScanResults)
    {
        m_ActiveScanIndex = -1;     // UpdateAccessPointList() で何もしない
    }

    {
        // ローカル通信用のダミーアクセスポイントの追加
        if (pAccessPointList->CreateAndAdd<LocalAccessPoint>(this) == nullptr)
        {
            NN_DETAIL_NIFM_WARN("Accesspoint list is full. (Local)\n");
            NN_RESULT_SUCCESS;
        }
        // すれいちがい通信用のダミーアクセスポイントの追加
        if (pAccessPointList->CreateAndAdd<NeighborDetectionAccessPoint>(this) == nullptr)
        {
            NN_DETAIL_NIFM_WARN("Accesspoint list is full. (Neighbor)\n");
            NN_RESULT_SUCCESS;
        }
    }

    // インターネット通信用のアクセスポイントの列挙

    if (m_pLastScannedResults->IsValid() && !isScanDataExpired)
    {
        m_pLastScannedResults->GetAccessPointList().ExportTo(pAccessPointList);
    }

    NN_RESULT_THROW(result);
}

nn::Result WirelessInterface::ConnectImpl(
    const WirelessAccessPoint& wirelessAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ConnectImpl(const WirelessAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    NN_RESULT_DO(CancelFlagHolder::GetSingleton().ConfirmInternetAdmitted(NetworkInterfaceType_Ieee80211));

    NN_RESULT_DO(OpenInfraMode());

    m_IsCloseInfraModeNeeded = true;

    ConnectRunner connectRunner(*this, wirelessAccessPoint, networkProfile);
    NN_RESULT_DO(connectRunner.Start());

    nn::Result cancelResult = ResultInternalError();
    while (!connectRunner.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(100)))
    {
        auto result = CancelFlagHolder::GetSingleton().ConfirmInternetAdmitted(NetworkInterfaceType_Ieee80211);

        if (result.IsFailure() && cancelResult.IsFailure())
        {
            cancelResult = connectRunner.Cancel(result);
            NN_DETAIL_NIFM_INFO("Connection is cancelled (%x).\n", cancelResult.GetInnerValueForDebug());
        }
    }

    auto result = connectRunner.Wait();

    m_WlanConnectionEvent->TryWait();

    if (result.IsSuccess())
    {
        result = InitializeInternetConnection(networkProfile, aggregatedRequest);

        if (result.IsFailure())
        {
            DisconnectImpl(wirelessAccessPoint);
        }
    }

    NN_RESULT_THROW(result);

} // NOLINT(impl/function_size)

nn::Result WirelessInterface::DisconnectImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(wirelessAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::DisconnectImpl(const WirelessAccessPoint&, ...)\n");

    NN_DETAIL_NIFM_INFO("nn::wlan::Infra::Disconnect() started.\n");
    nn::wlan::Infra::Disconnect();
    NN_DETAIL_NIFM_TRACE("nn::wlan::Infra::Disconnect() ended.\n");

    NN_SDK_ASSERT(m_WlanConnectionEvent);
    if (m_WlanConnectionEvent)
    {
        // 切断は把握しているので、シグナルをここで消すことによりわざわざ WlanConnectionEventCallback は通さない
        m_WlanConnectionEvent->TryWait();
    }

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ReleaseImpl(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(wirelessAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ReleaseImpl(const WirelessAccessPoint&, ...)\n");

    // 外的要因で切断された場合は DisconnectImpl を呼ばないことがあるので、
    // 必ず通るここで CloseMode する
    if (m_IsCloseInfraModeNeeded)  // StartLocalCommunication の場合 OpenInfraMode を呼ばないのでここでクローズしてはいけない
    {
        CloseInfraMode();
        m_IsCloseInfraModeNeeded = false;
    }

    NN_SDK_ASSERT(!m_WlanConnectionEvent);
    if (m_WlanConnectionEvent)
    {
        // nn::wlan::Connect, TryWait を抜けた直後に切断されて、
        // TryWait 後に取得した ConnectionState が Disonnected だった場合にシグナルが残るので、
        // Free 状態になる前に wlan ライブラリの SystemEvent がシグナルしていないことを保証する
        // ここに来るまでに wlan ライブラリが ConnectionState_Idle になっていることは保証されているので
        // それ以降 nn::wlan::Connect を呼ばずに SystemEvent がシグナルすることはない

        // …が、そもそもここで m_pWlanConnectionEvent は破棄されているはずなので念のため
        m_WlanConnectionEvent->TryWait();
    }

    NN_RESULT_SUCCESS;
}

void WirelessInterface::UpdateAccessPoint(WirelessAccessPoint* pInOutWirelessAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInOutWirelessAccessPoint);

    // ステルスの AP だとビーコン情報に SSID が入っていないので、プロファイルから埋める
    Ssid ssid;
    networkProfile.GetWirelessSetting().GetSsid(&ssid);
    pInOutWirelessAccessPoint->SetSsid(ssid);

    // WEP の認証設定はビーコンからは判断できないので、プロファイルから埋める
    if (pInOutWirelessAccessPoint->GetEncryption() == Encryption_Wep)
    {
        NN_SDK_ASSERT_EQUAL(pInOutWirelessAccessPoint->GetEncryption(), networkProfile.GetWirelessSetting().GetEncryption());

        pInOutWirelessAccessPoint->SetWepAuthenticationAndEncryption(networkProfile.GetWirelessSetting().GetAuthentication(), pInOutWirelessAccessPoint->GetEncryption());
    }

    // TODO: [TORIAEZU]
    // nn::wlan::Infra::Connect が SSID 指定でブロードキャストなアクティブスキャンをおこない、
    // 応答を受けた中でいちばん電波強度の強いものに接続試行をおこなうため、接続したアクセスポイントが当初指定と異なることがある
    {
        nn::wlan::ConnectionStatus connectionStatus;

        if (nn::wlan::Infra::GetConnectionStatus(&connectionStatus).IsSuccess())
        {
            MacAddress bssid;
            std::memcpy(bssid.data, connectionStatus.bssid.GetMacAddressData(), bssid.Size);
            pInOutWirelessAccessPoint->SetBssid(bssid);
        }
    }

    UpdateAccessPointHistory(*pInOutWirelessAccessPoint);
}

// Local

nn::Result WirelessInterface::ConnectImpl(
    const LocalAccessPoint& localAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(localAccessPoint);
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ConnectImpl(const LocalAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::DisconnectImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(localAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::DisconnectImpl(const LocalAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ReleaseImpl(const LocalAccessPoint& localAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(localAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ReleaseImpl(const LocalAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

void WirelessInterface::UpdateAccessPoint(LocalAccessPoint* pInOutLocalAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_UNUSED(pInOutLocalAccessPoint);
    NN_UNUSED(networkProfile);
}

// NeighborDetection
nn::Result WirelessInterface::ConnectImpl(
    const NeighborDetectionAccessPoint& neighborDetectionAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(neighborDetectionAccessPoint);
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ConnectImpl(const NeighborDetectionAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::DisconnectImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(neighborDetectionAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::DisconnectImpl(const NeighborDetectionAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ReleaseImpl(const NeighborDetectionAccessPoint& neighborDetectionAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(neighborDetectionAccessPoint);

    NN_DETAIL_NIFM_TRACE("WirelessInterface::ReleaseImpl(const NeighborDetectionAccessPoint&, ...)\n");

    NN_RESULT_SUCCESS;
}

void WirelessInterface::UpdateAccessPoint(NeighborDetectionAccessPoint* pInOutNeighborDetectionAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_UNUSED(pInOutNeighborDetectionAccessPoint);
    NN_UNUSED(networkProfile);
}

bool WirelessInterface::CopyToAccessPointListFromAccessPointData(AccessPointListBase* pOutAccessPointList, const AccessPointData& accessPoint) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutAccessPointList);

    WirelessAccessPoint wirelessAccessPoint(this);
    wirelessAccessPoint.SetBssid(accessPoint.bssid);
    wirelessAccessPoint.SetSsid(accessPoint.ssid);
    wirelessAccessPoint.SetChannel(accessPoint.channel);
    wirelessAccessPoint.SetRssi(accessPoint.rssi);
    wirelessAccessPoint.SetLinkLevel(accessPoint.linkLevel);

    switch (accessPoint.authentication)
    {
    case Authentication_WpaPsk:
        NN_FALL_THROUGH;
    case Authentication_Wpa2Psk:
        wirelessAccessPoint.SetWpaAuthenticationAndEncryption(accessPoint.authentication, accessPoint.encryption, accessPoint.groupEncryption);
        break;
    case Authentication_Open:
        NN_FALL_THROUGH;
    case Authentication_Shared:
        NN_FALL_THROUGH;
    case Authentication_Unknown:
        wirelessAccessPoint.SetWepAuthenticationAndEncryption(accessPoint.authentication, accessPoint.encryption);
        break;
    default:
        NN_SDK_ASSERT(false);
        return false;
    }

    auto pWirelessAccessPoint = pOutAccessPointList->CreateTempAndUpdate<WirelessAccessPoint>(wirelessAccessPoint);
    while (pWirelessAccessPoint == nullptr && pOutAccessPointList->DeleteOldest())
    {
        pWirelessAccessPoint = pOutAccessPointList->CreateTempAndUpdate<WirelessAccessPoint>(wirelessAccessPoint);
    }

    if (pWirelessAccessPoint == nullptr)
    {
        return false;
    }

    pOutAccessPointList->Commit();

    return true;
}

nn::Result WirelessInterface::OpenInfraMode() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::Result result = nn::ResultSuccess();

    if (m_OpenInfraModeCount == 0)
    {
        result = nn::wlan::Infra::OpenMode();

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::wlan::ResultWlanDeviceAbnormal)
            {
                m_ResultHealth = ResultNetworkInterfaceInTrouble();
                NN_RESULT_THROW(ResultNetworkInterfaceInTrouble());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_DETAIL_NIFM_WARN_V3("nn::wlan::OpenMode failed (%08x).\n", result.GetInnerValueForDebug());

                m_ResultHealth = nn::ResultSuccess();   // すくなくとも 2.0.0 NUP 時点では ResultWlanDeviceAbnormal のときだけ nn::nifm::ConfirmSystemAvailability() を失敗にする
                NN_RESULT_THROW(ResultNetworkInterfaceNotAvailable());
            }
        NN_RESULT_END_TRY

        m_ResultHealth = nn::ResultSuccess();

        NN_SDK_ASSERT(!m_WlanConnectionEvent);
        NN_SDK_ASSERT(!m_WlanConnectionEventCallback);
        NN_SDK_ASSERT(!m_WlanConnectionEventHandler);

        m_WlanConnectionEvent.emplace();
        nn::wlan::Infra::GetConnectionEvent(m_WlanConnectionEvent->GetBase());
        m_WlanConnectionEventCallback.emplace(this);
        m_WlanConnectionEventHandler.emplace(*m_WlanConnectionEvent);
        m_WlanConnectionEventHandler->Register(&*m_WlanConnectionEventCallback);

        AddEventHandler(*m_WlanConnectionEventHandler);
    }

    if (result.IsSuccess())
    {
        ++m_OpenInfraModeCount;
    }

    NN_RESULT_THROW(result);
}

nn::Result WirelessInterface::CloseInfraMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_OpenInfraModeCount > 0);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::Result result = nn::ResultSuccess();

    if( m_OpenInfraModeCount > 0 )
    {
        if( m_OpenInfraModeCount == 1 )
        {
            result = nn::wlan::Infra::CloseMode();

            if( result.IsSuccess() )
            {
                NN_SDK_ASSERT(m_WlanConnectionEvent);
                NN_SDK_ASSERT(m_WlanConnectionEventCallback);
                NN_SDK_ASSERT(m_WlanConnectionEventHandler);

                RemoveEventHandler(*m_WlanConnectionEventHandler);

                m_WlanConnectionEventHandler = nn::util::nullopt;
                m_WlanConnectionEventCallback = nn::util::nullopt;
                m_WlanConnectionEvent = nn::util::nullopt;
            }
            else
            {
                NN_DETAIL_NIFM_TRACE_V3("nn::wlan::CloseMode failed (%08x).\n", result.GetInnerValueForDebug());
            }
        }

        if( result.IsSuccess() )
        {
            --m_OpenInfraModeCount;
        }
    }

    NN_RESULT_THROW(result);
}

const char* WirelessInterface::GetInterfaceName() const NN_NOEXCEPT
{
    return "wl0";
}

bool WirelessInterface::IsLinkUp() const NN_NOEXCEPT
{
    nn::wlan::ConnectionStatus connectionStatus;
    auto result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);

    if (result.IsFailure())
    {
        return false;
    }

    return connectionStatus.state == nn::wlan::ConnectionState_Connected;
}

nn::Result WirelessInterface::ConfirmHealth() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW(m_ResultHealth);
}

void WirelessInterface::UpdateAccessPointHistory(const WirelessAccessPoint& wirelessAccessPoint) NN_NOEXCEPT
{
    AccessPointData* pAccessPoint = nullptr;

    {
        MacAddress bssid;
        wirelessAccessPoint.GetBssid(&bssid);

        // 一致しているものがあれば上書きして繰り上げ
        for (auto p : m_AccessPointHistory)
        {
            if (p->bssid == bssid)
            {
                pAccessPoint = p;
                break;
            }
        }
    }

    // 空きがあれば追加、空きがなければ末尾を追い出して追加
    if (pAccessPoint == nullptr)
    {
        pAccessPoint = m_AccessPointHistory.Create();

        if (pAccessPoint == nullptr)
        {
            pAccessPoint = &m_AccessPointHistory[m_AccessPointHistory.GetCount() - 1];
        }
    }

    NN_SDK_ASSERT_NOT_NULL(pAccessPoint);

    wirelessAccessPoint.TryExport(pAccessPoint);

    // TODO: [TORIAEZU] MoveToTop の代わり
    for (int i = 0; i < m_AccessPointHistory.GetCount(); ++i)
    {
        if (&m_AccessPointHistory[i] == pAccessPoint)
        {
            m_AccessPointHistory.Move(i, 0);
            break;
        }
    }

    m_ActiveScanIndex = 0;

    for (int i = 0; i < m_AccessPointHistory.GetCount(); ++i)
    {
        NN_DETAIL_NIFM_TRACE(".oOo.oOo.oOo.oOo.oOo.oOo.oOo.oOo.oOo.\n");
        NN_DETAIL_NIFM_TRACE("m_AccessPointHistory[%d]\n", i);
        NN_DETAIL_NIFM_TRACE("  SSID: %.32s\n", m_AccessPointHistory[i].ssid.hex);
        NN_DETAIL_NIFM_TRACE("  Channel: %u\n", m_AccessPointHistory[i].channel);
        const MacAddress& bssid = m_AccessPointHistory[i].bssid;
        NN_UNUSED(bssid);
        NN_DETAIL_NIFM_TRACE("  BSSID: %02x:%02x:%02x:%02x:%02x:%02x\n", bssid.data[0], bssid.data[1], bssid.data[2], bssid.data[3], bssid.data[4], bssid.data[5]);
        NN_DETAIL_NIFM_TRACE(".oOo.oOo.oOo.oOo.oOo.oOo.oOo.oOo.oOo.\n");
    }
}

nn::Result WirelessInterface::PutToSleepImpl() NN_NOEXCEPT
{
    nn::wlan::WlanState state;
    NN_UNUSED(state);
    NN_SDK_ASSERT(nn::wlan::Infra::GetState(&state).IsSuccess()
        && (state == nn::wlan::WlanState_Ready ||
            state == nn::wlan::WlanState_InfraIdle ||
            state == nn::wlan::WlanState_InfraSta));

    NN_RESULT_THROW(nn::wlan::Infra::RequestSleep());
}

nn::Result WirelessInterface::WakeUpImpl() NN_NOEXCEPT
{
    nn::wlan::WlanState state;
    NN_UNUSED(state);
    NN_SDK_ASSERT(nn::wlan::Infra::GetState(&state).IsSuccess()
        && (state == nn::wlan::WlanState_Sleep ||
            state == nn::wlan::WlanState_Ready ||
            state == nn::wlan::WlanState_InfraIdle ||
            state == nn::wlan::WlanState_InfraSta));

    NN_RESULT_THROW(nn::wlan::Infra::RequestWakeUp());
}

nn::Result WirelessInterface::WisprAuthenticate(HttpResponse* pHttpResponse) NN_NOEXCEPT
{
    HttpResponse authenticationResponse;
    HttpPostData authenticationPostData;
    AuthenticationClient authenticationClient(pHttpResponse);

    authenticationClient.SetResponseBuffer(&authenticationResponse);
    authenticationClient.SetPostDataBuffer(&authenticationPostData);
    authenticationClient.SetNetworkInterfaceType(m_NetworkProfileData.networkInterfaceType);
    authenticationClient.SetProxySetting(m_NetworkProfileData.ipSetting.proxy);

    // リクエストパラメータの設定
    MacAddress macAddress;
    authenticationClient.SetMacAddress(*GetMacAddress(&macAddress));
    MacAddress bssid;
    authenticationClient.SetBssid(*reinterpret_cast<const WirelessAccessPoint*>(m_pAccessPoint)->GetBssid(&bssid));
    authenticationClient.SetSsid(m_NetworkProfileData.wirelessSetting.ssidConfig.ssid);

    return authenticationClient.Authenticate(AuthenticationClient::AuthenticationUrl);
}

nn::Result WirelessInterface::SetTcpSessionInformation(const SocketInfo& socketInfo) NN_NOEXCEPT
{
    // TODO: [TORIAEZU]
    // 最初から SocketHolder で管理して、それが渡ってくるようにしたい
    SocketHolder origin(socketInfo.socketDescriptor, socketInfo.processId);
    SocketHolder duplicated(origin);

    nn::socket::TcpInfo tcpInfo;
    NN_RESULT_DO(duplicated.GetTcpInfo(&tcpInfo));

    nn::socket::SockAddrIn sockAddrInClient;
    NN_RESULT_DO(duplicated.GetSource(&sockAddrInClient));

    nn::socket::SockAddrIn sockAddrInServer;
    NN_RESULT_DO(duplicated.GetDestination(&sockAddrInServer));

    IpAddressSetting ipAddressSetting;
    DnsSetting dnsSetting;
    NN_RESULT_DO(m_IpConfiguration.GetIpConfigInfo(&ipAddressSetting, &dnsSetting));

    if (sockAddrInClient.sin_addr.S_addr == nn::socket::InAddr_Any)
    {
        // bind していないとソースが 0.0.0.0:(ephemeral port) になるので
        // デフォルトルートの IP アドレスを埋める
        sockAddrInClient.sin_addr = IpV4AddressToInAddr(ipAddressSetting.ipAddress);
    }

    MacAddress macAddress;

    {
        nn::socket::InAddr inAddrSubnetMask = IpV4AddressToInAddr(ipAddressSetting.ipAddress);
        nn::socket::InAddr inAddrDefaultGateway = IpV4AddressToInAddr(ipAddressSetting.defaultGateway);

        if ((sockAddrInServer.sin_addr.S_addr & inAddrSubnetMask.S_addr) == (inAddrDefaultGateway.S_addr & inAddrSubnetMask.S_addr))
        {
            // TODO: 現在の実装では、サブネット内に通信先がいるとき、 ARP エントリ失効をカバーできていない
            NN_RESULT_DO(SocketHolder::LookUpArpEntry(&macAddress, sockAddrInServer.sin_addr));
        }
        else
        {
            NN_RESULT_DO(m_IpConfiguration.GetDefaultGatewayMacAddress(&macAddress));
        }
    }

    NN_DETAIL_NIFM_INFO(">       seq: % 16u[0x%08x]\n", tcpInfo.tcpi_snd_nxt, tcpInfo.tcpi_snd_nxt);
    NN_DETAIL_NIFM_INFO(">       ack: % 16u[0x%08x]\n", tcpInfo.tcpi_rcv_nxt, tcpInfo.tcpi_rcv_nxt);
    NN_DETAIL_NIFM_INFO(">      rwin: % 16u\n", tcpInfo.tcpi_rcv_space);
    NN_DETAIL_NIFM_INFO(">     local: % 16s:%u\n", nn::socket::InetNtoa(sockAddrInClient.sin_addr), nn::socket::InetNtohs(sockAddrInClient.sin_port));
    NN_DETAIL_NIFM_INFO(">    remote: % 16s:%u\n", nn::socket::InetNtoa(sockAddrInServer.sin_addr), nn::socket::InetNtohs(sockAddrInServer.sin_port));
    NN_DETAIL_NIFM_INFO("> next addr: [%02x:%02x:%02x:%02x:%02x:%02x]\n",
        macAddress.data[0], macAddress.data[1], macAddress.data[2], macAddress.data[3], macAddress.data[4], macAddress.data[5]);

    // 以下、 wlan の形式への変換

    nn::wlan::MacAddress dstMac(macAddress.data);

    nn::wlan::WlanIpv4Address src;
    InAddrToWlanIpv4Address(&src, sockAddrInClient.sin_addr);

    nn::wlan::WlanIpv4Address dst;
    InAddrToWlanIpv4Address(&dst, sockAddrInServer.sin_addr);

    auto result = nn::wlan::Infra::SetTcpSessionInformation(
        dstMac,
        src, dst, nn::socket::InetNtohs(sockAddrInClient.sin_port), nn::socket::InetNtohs(sockAddrInServer.sin_port),
        tcpInfo.tcpi_rcv_nxt, static_cast<uint16_t>(tcpInfo.tcpi_rcv_space)
    );

    if (result.IsFailure())
    {
        NN_DETAIL_NIFM_WARN("SetTcpSessionInformation failed (result 0x%08x)\n", result.GetInnerValueForDebug());

        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::SetScanChannels(const int16_t scanChannels[], int count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_ScanChannelCount = std::min(static_cast<size_t>(count), NN_ARRAY_SIZE(m_ScanChannels));
    for (int i = 0; i < m_ScanChannelCount; ++i)
    {
        m_ScanChannels[i] = scanChannels[i];
    }

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::GetScanChannels(int16_t* pOutScanChannels, int* pOutCount) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutScanChannels);
    NN_SDK_ASSERT_NOT_NULL(pOutCount);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    *pOutCount = m_ScanChannelCount;
    for (int i = 0; i < m_ScanChannelCount; ++i)
    {
        pOutScanChannels[i] = m_ScanChannels[i];
    }

    NN_RESULT_SUCCESS;
}

nn::Result WirelessInterface::ClearScanChannels() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_ScanChannelCount = InvalidScanChannelCount;
    memset(m_ScanChannels, 0, sizeof(m_ScanChannels));

    NN_RESULT_SUCCESS;
}

}
}
}

