﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/networkInterface/nifm_NetworkInterfaceBase.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointBase.h>
#include <nn/nifm/detail/core/connectionConfirmation/nifm_HttpResponse.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>
#include <nn/nifm/detail/core/connectionConfirmation/nifm_ConnectionTestClient.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>
#include <nn/nifm/detail/util/nifm_DebugUtility.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>
#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

const MacAddress NetworkInterfaceBase::InvalidMacAddress = { { 0x1, 0x0, 0x0, 0x0, 0x0, 0x0 } };
extern const AggregatedRequestType InvalidAggregatedRequestType = {};
const int NetworkInterfaceBase::InvalidScanChannelCount = -1;

NetworkInterfaceBase::NetworkInterfaceBase() NN_NOEXCEPT
    : m_IsEnabled(true),
      m_NetworkType(NetworkType_None),
      m_NetworkResourceState(NetworkResourceState::Free),
      m_RequestRevision(0),
      m_AggregatedRequestType(InvalidAggregatedRequestType),
      m_ConnectionEvent(nn::os::EventClearMode_AutoClear),
      m_ConnectionSingleEventHandler(m_ConnectionEvent),
      m_ConnectionEventCallback(*this),
      m_IsInternetConnectionInitialized(false),
      m_InternetAvailability(InternetAvailability_Invalid),
      m_InternetResult(ResultInternetConnectionNotAvailable()),
      m_AdditionalInfo(),
      m_IsDisconnectionBlocking(false),
      m_ConnectionResult(ResultLinkLayerNotAvailable()),
      m_pAccessPoint(nullptr)
{
    memset(&m_NetworkProfileData, 0, sizeof(m_NetworkProfileData));
    m_ConnectionEventHandler.Register(&m_ConnectionEventCallback);
    m_ConnectionEventHandler.Add(m_ConnectionSingleEventHandler);
}

NetworkInterfaceBase::~NetworkInterfaceBase() NN_NOEXCEPT
{
    // 派生クラスのデストラクタでネットワークリソースを解放しておくことが必要
    // （Disconnect や Release は純粋仮想関数を呼ぶので基底クラスのデストラクタからは呼べない）
    NN_SDK_ASSERT(m_NetworkResourceState == NetworkResourceState::Free);
}

nn::Result NetworkInterfaceBase::Connect(
    const AccessPointBase& accessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("NetworkInterfaceBase(%d)::Connect\n", GetNetworkInterfaceType());

    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        NN_RESULT_THROW_UNLESS(m_NetworkResourceState == NetworkResourceState::Free, nn::nifm::ResultInvalidNetworkResourceState());
        NN_RESULT_THROW_UNLESS(m_IsEnabled, nn::nifm::ResultNetworkInterfaceDisabled());

        NN_SDK_ASSERT_EQUAL(m_NetworkType, NetworkType_None);
        NN_SDK_ASSERT_EQUAL(m_pAccessPoint, nullptr);
        NN_SDK_ASSERT_LESS_EQUAL(accessPoint.GetSize(), GetAccessPointStorageSize());

        m_pAccessPoint = accessPoint.CopyTo(GetAccessPointStoragePointer(), GetAccessPointStorageSize());
        m_NetworkType = m_pAccessPoint->GetNetworkType();   // TODO: 外す
        networkProfile.Export(&m_NetworkProfileData);
        m_NetworkResourceState = NetworkResourceState::ToBePrepared;
        m_RequestRevision = aggregatedRequest.revision;
        m_AggregatedRequestType = aggregatedRequest;
        m_IsDisconnectionBlocking = false;
        m_ConnectionResult = ResultLinkLayerNotAvailable();
    }

    NN_SDK_ASSERT_NOT_NULL(m_pAccessPoint);

    nn::Result result =
        m_pAccessPoint->IsPossiblyAvailable(networkProfile, aggregatedRequest) ?
        m_pAccessPoint->ConnectImpl(networkProfile, aggregatedRequest) :
        nn::nifm::ResultLinkLayerNotAvailable();    // TODO

    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    // ConnectImpl は成功で抜けたが、ここに来る前に外部要因の切断が発生していた場合
    if (result.IsSuccess() && m_NetworkResourceState == NetworkResourceState::Lost)
    {
        result = nn::nifm::ResultNotConnected();    // TODO
    }

    if (result.IsSuccess())
    {
        // 接続が成功して初めてわかるアクセスポイントの情報を更新する（ステルスで隠されていた SSID など）
        m_pAccessPoint->UpdateAccessPoint(networkProfile);
    }

    m_ConnectionResult = result;

    SetStateAndSignal(result.IsSuccess() ? NetworkResourceState::Available : NetworkResourceState::Lost);

    NN_RESULT_THROW(result);
}

nn::Result NetworkInterfaceBase::Disconnect(nn::Result connectionResult) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pAccessPoint != nullptr, ResultInvalidNetworkResourceState());

    NN_RESULT_THROW(Disconnect(nullptr, connectionResult));
}

nn::Result NetworkInterfaceBase::Disconnect(
    const AccessPointBase* pAccessPoint,
    nn::Result connectionResult) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("NetworkInterfaceBase(%d)::Disconnect\n", GetNetworkInterfaceType());

    NN_SDK_ASSERT(connectionResult.IsFailure());

    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        NN_RESULT_THROW_UNLESS(m_NetworkResourceState != NetworkResourceState::Lost, nn::ResultSuccess());
        NN_RESULT_THROW_UNLESS(m_NetworkResourceState == NetworkResourceState::Available, nn::nifm::ResultInvalidNetworkResourceState());

        if (pAccessPoint != nullptr && m_pAccessPoint != pAccessPoint)
        {
            // TODO: [TORIAEZU] 本当はアクセスポイントの同一性が正しく判定できているべき
            if (m_pAccessPoint->GetNetworkType() == pAccessPoint->GetNetworkType())
            {
                NN_DETAIL_NIFM_WARN("The wrong access point may be being disconnected.\n");
            }
            else
            {
                NN_RESULT_THROW(nn::nifm::ResultInternalError());   // TODO
            }
        }

        if (!m_AggregatedRequestType.isAcceptedRejectable)
        {
            m_IsDisconnectionBlocking = true;
            m_ConnectionResult = connectionResult;
            NN_RESULT_THROW(nn::nifm::ResultDisconnectionBlocked());
        }
    }

    NN_RESULT_DO(m_pAccessPoint->DisconnectImpl());

    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        m_ConnectionResult = connectionResult;
        SetStateAndSignal(NetworkResourceState::Lost);
    }
    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::Release() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pAccessPoint != nullptr, ResultInvalidNetworkResourceState());

    NN_RESULT_THROW(Release(nullptr));
}

nn::Result NetworkInterfaceBase::Release(
    const AccessPointBase* pAccessPoint) NN_NOEXCEPT
{
    NN_DETAIL_NIFM_INFO("NetworkInterfaceBase(%d)::Release\n", GetNetworkInterfaceType());

    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        NN_RESULT_THROW_UNLESS(m_NetworkResourceState != NetworkResourceState::Free, nn::ResultSuccess());
        NN_RESULT_THROW_UNLESS(m_NetworkResourceState == NetworkResourceState::Lost, nn::nifm::ResultInvalidNetworkResourceState());

        if (pAccessPoint != nullptr && m_pAccessPoint != pAccessPoint)
        {
            // TODO: [TORIAEZU] ConnectionSelector::ConnectAny() 内の Release を通すためのワークアラウンド
            // 本当はアクセスポイントの同一性が正しく判定できているべき
            if (m_pAccessPoint->GetNetworkType() == pAccessPoint->GetNetworkType())
            {
                NN_DETAIL_NIFM_WARN("The wrong access point may be being released.\n");
            }
            else
            {
                NN_RESULT_THROW(nn::nifm::ResultInternalError());   // TODO
            }
        }
    }

    FinalizeInternetConnectionIfNecessary();

    auto result = m_pAccessPoint->ReleaseImpl();

    if (result.IsSuccess())
    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        // 派生クラス内に実体があり、参照は基底クラス内にしかない前提
        m_pAccessPoint->~AccessPointBase();

        m_pAccessPoint = nullptr;
        m_NetworkType = NetworkType_None;
        memset(&m_NetworkProfileData, 0, sizeof(m_NetworkProfileData));
        m_NetworkResourceState = NetworkResourceState::Free;
        m_RequestRevision = 0;
        m_AggregatedRequestType = InvalidAggregatedRequestType;
        m_ConnectionEvent.Clear();
        m_IsDisconnectionBlocking = false;
        m_ConnectionResult = ResultLinkLayerNotAvailable();
    }

    NN_RESULT_THROW(result);
}

nn::Result NetworkInterfaceBase::ConfirmKeepingCurrentConnection(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    nn::Result result = ConfirmAcceptability(aggregatedRequest);

    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    if (result.IsSuccess() && aggregatedRequest.networkType != NetworkType_None)
    {
        m_AggregatedRequestType = aggregatedRequest;
    }
    else
    {
        // TODO: [TORIAEZU]
        // 失敗でも内容を更新するのが気持ち悪いのを何とかする
        // AggregatedRequest の中で、以下のパラメータは新しく満たすべき接続の状態を表しているのではないのが原因
        m_AggregatedRequestType.isAcceptedRejectable = aggregatedRequest.isAcceptedRejectable;
        m_AggregatedRequestType.lastRevision = aggregatedRequest.lastRevision;
    }

    NN_RESULT_THROW(result);
}

nn::Result NetworkInterfaceBase::UpdateCurrentNetworkProfile(const NetworkProfileData& networkProfileData) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    if (m_NetworkResourceState == NetworkResourceState::Available)
    {
        NN_SDK_ASSERT_EQUAL(m_NetworkProfileData.id, networkProfileData.id);
        m_NetworkProfileData = networkProfileData;

        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultNotConnected());
}

nn::Result NetworkInterfaceBase::ConfirmAcceptability(const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        // 接続がなく、接続が不要な場合は、ほかに何もチェックせずに終了
        if (m_NetworkType == NetworkType_None && aggregatedRequest.networkType == NetworkType_None)
        {
            NN_RESULT_SUCCESS;
        }

        if (m_NetworkType != aggregatedRequest.networkType)
        {
            NN_RESULT_THROW(ResultCannotShareCurrentConnection());
        }
        else if ((m_NetworkType == NetworkType_Local || m_NetworkType == NetworkType_NeighborDetection)
            && m_AggregatedRequestType.revision != aggregatedRequest.revision)
        {
            NN_RESULT_THROW(ResultCannotShareCurrentConnection());
        }

        if (m_NetworkResourceState != NetworkResourceState::Available)
        {
            NN_RESULT_THROW(ResultCannotShareCurrentConnection());
        }

        if (aggregatedRequest.profileId != nn::util::InvalidUuid &&
            aggregatedRequest.profileId != m_NetworkProfileData.id)
        {
            NN_RESULT_THROW(ResultCannotShareCurrentConnection());
        }
    }

    nn::Result result = UpdateInternetAvailability(
        aggregatedRequest.connectionConfirmationOption,
        aggregatedRequest.connectionConfirmationOptionSub);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultHotspotAuthenticationViaWebAuthAppletNeeded)
        {
            m_AdditionalInfo.profileId = m_NetworkProfileData.id;
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

bool NetworkInterfaceBase::IsEnabled() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    return m_IsEnabled;
}

nn::Result NetworkInterfaceBase::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    if (m_IsEnabled != isEnabled)
    {
        m_ConnectionEvent.Signal();
        NN_DETAIL_NIFM_TRACE("ConnectionEvent is signaled.\n");
    }

    m_IsEnabled = isEnabled;

    NN_DETAIL_NIFM_INFO("NIC(Type: %d) is now %s.\n", GetNetworkInterfaceType(), isEnabled ? "enabled" : "disabled");

    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::ConnectionEventCallback() NN_NOEXCEPT
{
    // 無線オフなのに接続があれば切断する

    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        if (m_IsEnabled)
        {
            NN_RESULT_SUCCESS;
        }

        NN_SDK_ASSERT_NOT_EQUAL(m_NetworkResourceState, NetworkResourceState::ToBePrepared);

        if (m_NetworkResourceState != NetworkResourceState::Available)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(Disconnect(ResultNetworkInterfaceNoLongerAvailable()));
};

nn::Result NetworkInterfaceBase::IpEventCallback() NN_NOEXCEPT
{
    auto result = m_IpConfiguration.GetInterfaceStatus();

    if (result.IsFailure())
    {
        NN_RESULT_THROW(Disconnect(result));
    }

    NN_RESULT_SUCCESS;
};

void NetworkInterfaceBase::GetNetworkResourceInfo(NetworkResourceInfo* pOutNetworkResourceInfo) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    pOutNetworkResourceInfo->networkType = m_NetworkType;
    pOutNetworkResourceInfo->networkResourceState = m_NetworkResourceState;
    pOutNetworkResourceInfo->connectionResult = m_ConnectionResult;
    pOutNetworkResourceInfo->isDisconnectionBlocking = m_IsDisconnectionBlocking;
    pOutNetworkResourceInfo->internetAvailability = m_InternetAvailability;
    pOutNetworkResourceInfo->internetResult = m_InternetResult;
    pOutNetworkResourceInfo->networkInterfaceType = GetNetworkInterfaceType();
    pOutNetworkResourceInfo->networkProfileId = m_NetworkProfileData.id;
    pOutNetworkResourceInfo->requestRevision = m_AggregatedRequestType.revision;
    pOutNetworkResourceInfo->isGreedy = m_AggregatedRequestType.isGreedy;
    pOutNetworkResourceInfo->isSharable = m_AggregatedRequestType.isSharable;
    pOutNetworkResourceInfo->restrictedRevision = m_AggregatedRequestType.restrictedRevision;
}

nn::Result NetworkInterfaceBase::GetCurrentAccessPoint(AccessPointData* pOutAccessPointData) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutAccessPointData);

    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    NN_RESULT_THROW_UNLESS(m_NetworkResourceState == NetworkResourceState::Available, ResultNotConnected());
    NN_RESULT_THROW_UNLESS(m_pAccessPoint != nullptr, ResultNotConnected());    // 念のため

    NN_RESULT_DO(m_pAccessPoint->TryExport(pOutAccessPointData));

    UpdateRssiAndLinkLevel(pOutAccessPointData);    // もともと何らかの値は入っているはずなので、失敗は無視

    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::UpdateRssiAndLinkLevel(AccessPointData* pInOutAccessPointData) const NN_NOEXCEPT
{
    NN_UNUSED(pInOutAccessPointData);

    NN_RESULT_THROW(ResultNotImplemented());
}


nn::Result NetworkInterfaceBase::GetCurrentNetworkProfile(NetworkProfileData* pOutNetworkProfile) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutNetworkProfile);

    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    NN_RESULT_THROW_UNLESS(m_NetworkResourceState == NetworkResourceState::Available, ResultNotConnected());
    switch (m_NetworkType)
    {
    case NetworkType_Internet:
        *pOutNetworkProfile = m_NetworkProfileData;
        NN_RESULT_SUCCESS;

    case NetworkType_Local:
        NN_RESULT_THROW(ResultLocalConnection());
    case NetworkType_NeighborDetection:
        NN_RESULT_THROW(ResultNeighborDetectConnection());

    case NetworkType_None:
        NN_FALL_THROUGH;
    default:
        NN_RESULT_THROW(ResultNotConnected());
    }
}

NetworkResourceState NetworkInterfaceBase::GetNetworkResourceState() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    return m_NetworkResourceState;
}

nn::Result NetworkInterfaceBase::Scan() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::GetScanData(AccessPointListBase* pOutAccessPointList) const NN_NOEXCEPT
{
    NN_UNUSED(pOutAccessPointList);

    NN_RESULT_SUCCESS;
}

bool NetworkInterfaceBase::UpdateAccessPointList(AccessPointListBase* pOutAccessPointList) NN_NOEXCEPT
{
    NN_UNUSED(pOutAccessPointList);

    return false;
}

void NetworkInterfaceBase::NotifyLinkDisconnected(nn::Result connectionResult) NN_NOEXCEPT
{
    NN_SDK_ASSERT(connectionResult.IsFailure());

    NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

    switch( m_NetworkResourceState )
    {
    case NetworkResourceState::ToBePrepared:
    case NetworkResourceState::Available:
        m_ConnectionResult = connectionResult;
        SetStateAndSignal(NetworkResourceState::Lost);
        break;

    case NetworkResourceState::Lost:
        // Disconnect とほぼ同時に外部要因の切断が発生し、 Disconnect が先行した場合に通る
        break;

    default:
        // Windows でネットワーク接続が自動で再確立される場合や、 Ethernet が自動で Link up/down する場合に通る
        break;
    }
}

void NetworkInterfaceBase::AddEventHandler(EventHandler& eventHandler) NN_NOEXCEPT
{
    m_ConnectionEventHandler.Add(eventHandler);
}

void NetworkInterfaceBase::RemoveEventHandler(EventHandler& eventHandler) NN_NOEXCEPT
{
    m_ConnectionEventHandler.Remove(eventHandler);
}

const char* NetworkInterfaceBase::GetInterfaceName() const NN_NOEXCEPT
{
    return "";
}

void NetworkInterfaceBase::SetStateAndSignal(NetworkResourceState state) NN_NOEXCEPT
{
    NN_SDK_ASSERT(state == NetworkResourceState::Available || state == NetworkResourceState::Lost);
    NN_SDK_ASSERT(m_NetworkResourceStateMutex.IsLockedByCurrentThread());

    if( state != m_NetworkResourceState )
    {
        m_NetworkResourceState = state;
        m_ConnectionEvent.Signal();
    }
}

nn::Result NetworkInterfaceBase::GetCurrentIpAddress( IpV4Address* pOutIpAddress ) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_IpConfiguration.GetIpAddress(pOutIpAddress));
}

nn::Result NetworkInterfaceBase::GetCurrentIpConfigInfo( IpAddressSetting* pOutIpAddressSetting, DnsSetting* pOutDnsSetting ) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_IpConfiguration.GetIpConfigInfo(pOutIpAddressSetting, pOutDnsSetting));
}

nn::Result NetworkInterfaceBase::GetRedirectInfoIfAny(AdditionalInfo* pOutAdditionalInfo) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_AdditionalInfo.redirectUrl[0] != '\0', ResultInternalError());
    *pOutAdditionalInfo = m_AdditionalInfo;

    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::ConfirmHealth() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}


nn::Result NetworkInterfaceBase::UpdateInternetAvailability(
    ConnectionConfirmationOption connectionConfirmationOption,
    ConnectionConfirmationOption connectionConfirmationOptionSub) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(connectionConfirmationOption, ConnectionConfirmationOption_Invalid);
    NN_SDK_ASSERT_LESS(connectionConfirmationOption, ConnectionConfirmationOption_Count);
    NN_SDK_ASSERT_GREATER_EQUAL(connectionConfirmationOptionSub, ConnectionConfirmationOption_Invalid);
    NN_SDK_ASSERT_LESS(connectionConfirmationOptionSub, ConnectionConfirmationOption_Count);
    NN_SDK_ASSERT_LESS_EQUAL(connectionConfirmationOption, connectionConfirmationOptionSub);

    NN_RESULT_THROW_UNLESS(connectionConfirmationOption >= ConnectionConfirmationOption_Invalid, ResultInvalidArgument());   // TODO
    NN_RESULT_THROW_UNLESS(connectionConfirmationOption < ConnectionConfirmationOption_Count, ResultInvalidArgument());   // TODO
    NN_RESULT_THROW_UNLESS(connectionConfirmationOptionSub >= ConnectionConfirmationOption_Invalid, ResultInvalidArgument());   // TODO
    NN_RESULT_THROW_UNLESS(connectionConfirmationOptionSub < ConnectionConfirmationOption_Count, ResultInvalidArgument());   // TODO
    NN_RESULT_THROW_UNLESS(connectionConfirmationOption <= connectionConfirmationOptionSub, ResultInvalidArgument());   // TODO

    InternetAvailability internetAvailability = m_InternetAvailability;
    nn::Result result = ResultInternetConnectionNotAvailable();     // 表示されてもよいように初期値を代入しておくけれど使われないはず

    Dump(m_NetworkType, GetNetworkInterfaceType(), internetAvailability, connectionConfirmationOption, connectionConfirmationOptionSub);

    NN_UTIL_SCOPE_EXIT
    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        m_InternetAvailability = internetAvailability;
        m_InternetResult = result;
        NN_DETAIL_NIFM_TRACE_V1("Internet connection status is now %d(NIC type: %d).\n", internetAvailability, GetNetworkInterfaceType());
    };

    // インターネット接続でなければ何もせず成功
    if (m_NetworkType != NetworkType_Internet)
    {
        NN_SDK_ASSERT(internetAvailability == InternetAvailability_Invalid);
        NN_RESULT_SUCCESS;
    }

    // リンク層の接続確立直後などでまだ疎通確認状態が Invalid であれば NotConfirmed に変更する
    if (internetAvailability == InternetAvailability_Invalid)
    {
        internetAvailability = InternetAvailability_NotConfirmed;
    }

    // 最優先の利用要求が疎通確認を禁止していれば、 それで終了
    if (connectionConfirmationOption == ConnectionConfirmationOption_Prohibited)
    {
        NN_RESULT_SUCCESS;
    }

    // 誰も疎通確認状態に興味がなければ、それで終了
    if (connectionConfirmationOptionSub < ConnectionConfirmationOption_Preferred)
    {
        NN_RESULT_SUCCESS;
    }

    // ここに来ているということは、疎通が確認できればよいと思っている誰かがいる
    NN_SDK_ASSERT(connectionConfirmationOptionSub >= ConnectionConfirmationOption_Preferred);

    // 疎通を明確に再確認してほしいと思っている利用要求があるか、
    // 現在のネットワークが疎通確認されていなければ、疎通確認してみる
    if (connectionConfirmationOptionSub == ConnectionConfirmationOption_Forced ||
        internetAvailability != InternetAvailability_Confirmed)
    {
        NN_DETAIL_NIFM_INFO("Connecting to the Connection Test Server...\n");

        m_InternetAvailability = InternetAvailability_InProgress;
        // 疎通確認
        HttpResponse connTestResponse;
        ConnectionTestClient connectionTestClient;

        connectionTestClient.SetResponseBuffer(&connTestResponse);
        connectionTestClient.SetNetworkInterfaceType(m_NetworkProfileData.networkInterfaceType);
        connectionTestClient.SetProxySetting(m_NetworkProfileData.ipSetting.proxy);
        result = connectionTestClient.ConfirmInternetConnection(ConnectionTestClient::ConnTestUrl);

        internetAvailability = result.IsSuccess() ? InternetAvailability_Confirmed : InternetAvailability_NotConfirmed;

        // リダイレクトされた ，または疎通確認サーバー以外から StatusCode:200 応答が返った
        if (result <= ResultConnectionTestHttpStatusFound() ||
            result <= ResultConnectionTestHttpStatusMovedPermanently() ||
            result <= ResultConnectionTestHttpStatusSeeOther() ||
            result <= ResultConnectionTestHttpStatusTemporaryRedirect() ||
            result <= ResultConnectionTestHttpStatusPermanentRedirect() ||
            result <= ResultConnectionTestResponseInvalidOrganization())
        {
            //  現在のプロファイルが SSID リストに含まれる，かつリダイレクトされた場合：WISPr
            // TODO: かつ Auth, Enc が一致
            if (m_NetworkProfileData.networkProfileType == NetworkProfileType_SsidList &&
                !(result <= ResultConnectionTestResponseInvalidOrganization()))
            {
                result = WisprAuthenticate(&connTestResponse);

                if (result.IsSuccess())
                {
                    result = connectionTestClient.ConfirmInternetConnection(ConnectionTestClient::ConnTestUrl);

                    if (result.IsSuccess())
                    {
                        internetAvailability = InternetAvailability_Confirmed;
                    }
                }
            }

            // SSID リストに含まれない または WISPr 認証（その後の疎通確認含む）に失敗 または 200 応答
            if (result.IsFailure())
            {
                // Web 認証アプレットには渡されないが，エラーレポート向けにリダイレクト URL を保存はしておく
                const auto redirectUrlPointer = connectionTestClient.GetRedirectUrlPointer();
                if (redirectUrlPointer != nullptr)
                {
                    nn::util::Strlcpy(m_AdditionalInfo.redirectUrl, redirectUrlPointer, m_AdditionalInfo.RedirectUrlSize);
                }

                result = ResultHotspotAuthenticationViaWebAuthAppletNeeded();
            }
        }
    }

    // ARP エントリが有効であることが強く期待できる疎通確認直後のタイミングで空呼びして、
    // あとで実際に必要になるときのために、内部にキャッシュをためておくための呼び出し
    MacAddress defaultGatewayMacAddress;
    m_IpConfiguration.GetDefaultGatewayMacAddress(&defaultGatewayMacAddress);

    NN_SDK_ASSERT(
        internetAvailability == InternetAvailability_Confirmed ||
        connectionConfirmationOption <= ConnectionConfirmationOption_Preferred ||
        result.GetDescription() != ResultInternetConnectionNotAvailable().GetDescription());

    // 最優先の利用要求が、疎通が必ず確認されていることを求めているのに確認できていなければ失敗とする
    NN_RESULT_THROW_UNLESS(
        internetAvailability == InternetAvailability_Confirmed || connectionConfirmationOption <= ConnectionConfirmationOption_Preferred,
        result);

    // 最優先の利用要求が、疎通が必ず確認されていることを求めていなければ、確認結果はどうあれ成功とする
    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::WisprAuthenticate(HttpResponse* pHttpResponse) NN_NOEXCEPT
{
    NN_UNUSED(pHttpResponse);
    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::InitializeInternetConnection(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsInternetConnectionInitialized);
    NN_RESULT_THROW_UNLESS(!m_IsInternetConnectionInitialized, ResultInternalError());

    {
        auto result = m_IpConfiguration.ApplyIpSetting(networkProfile.GetIpSetting());

        // 認証なし・暗号化ありで IP 設定に失敗するのは、復号が正しくできていない可能性がある

        if (result <= ResultDhcpFailed())
        {
            if (networkProfile.GetNetworkInterfaceType() == NetworkInterfaceType_Ieee80211)
            {
                const auto& wirelessSetting = networkProfile.GetWirelessSetting();

                if (wirelessSetting.GetAuthentication() == Authentication_Open &&
                    wirelessSetting.GetEncryption() != Encryption_None)
                {
                    result = ResultDhcpFailedOrPassphraseWrong();
                }
            }
        }

        NN_RESULT_DO(result);
    }

    m_IsInternetConnectionInitialized = true;   // 疎通確認の成否によらず、 IP 設定が完了した時点で終了処理は必要

    nn::os::SystemEvent* pSystemEvent = m_IpConfiguration.GetSystemEventPointer();

    if (pSystemEvent != nullptr)
    {
        m_IpSingleEventHandler.emplace(*pSystemEvent);
        m_IpEventCallback.emplace(*this);
        m_IpSingleEventHandler->Register(&*m_IpEventCallback);
        AddEventHandler(*m_IpSingleEventHandler);
    }

    auto result = UpdateInternetAvailability(
        aggregatedRequest.connectionConfirmationOption,
        aggregatedRequest.connectionConfirmationOptionSub);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultHotspotAuthenticationViaWebAuthAppletNeeded)
        {
            m_AdditionalInfo.profileId = networkProfile.GetId();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::FinalizeInternetConnectionIfNecessary() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_NetworkResourceState, NetworkResourceState::Lost);

    if (m_IsInternetConnectionInitialized)
    {
        if (m_IpSingleEventHandler)
        {
            RemoveEventHandler(*m_IpSingleEventHandler);
            m_IpEventCallback = nn::util::nullopt;
            m_IpSingleEventHandler = nn::util::nullopt;
        }

        // ここに来る時点で Lost 状態なのは保証されるので ReleaseImpl の結果を見ずにインターフェースダウンを通知する
        m_IpConfiguration.NotifyInterfaceDown();

        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        m_InternetAvailability = InternetAvailability_Invalid;
        m_IsInternetConnectionInitialized = false;
    }

    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::SetTcpSessionInformation(const SocketInfo& socketInfo) NN_NOEXCEPT
{
    NN_UNUSED(socketInfo);

    NN_RESULT_THROW(ResultNotImplemented());
}

nn::Result NetworkInterfaceBase::SetScanChannels(const int16_t scanChannels[], int count) NN_NOEXCEPT
{
    NN_UNUSED(scanChannels);
    NN_UNUSED(count);

    NN_RESULT_THROW(ResultNotImplemented());
}

nn::Result NetworkInterfaceBase::GetScanChannels(int16_t* pOutScanChannels, int* pOutCount) const NN_NOEXCEPT
{
    NN_UNUSED(pOutScanChannels);
    NN_UNUSED(pOutCount);

    NN_RESULT_THROW(ResultNotImplemented());
}

nn::Result NetworkInterfaceBase::ClearScanChannels() NN_NOEXCEPT
{
    NN_RESULT_THROW(ResultNotImplemented());
}

nn::Result NetworkInterfaceBase::PutToSleep() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_NetworkResourceStateMutex);

        // ndd からの利用要求受理時は ndd が、スリープに必要な処理を行う
        if (m_NetworkType != NetworkType_NeighborDetection)
        {
            NN_RESULT_DO(PutToSleepImpl());
        }
    }

    // スリープ中に IP アドレスのリース更新失敗による切断が走らないように
    if (m_IpSingleEventHandler)
    {
        RemoveEventHandler(*m_IpSingleEventHandler);
    }

    NN_RESULT_SUCCESS;
}

nn::Result NetworkInterfaceBase::WakeUp() NN_NOEXCEPT
{
    NN_RESULT_DO(WakeUpImpl());

    if (m_IpSingleEventHandler)
    {
        AddEventHandler(*m_IpSingleEventHandler);
    }

    NN_RESULT_SUCCESS;
}

}
}
}
