﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterface.win32.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>
#include <nn/nifm/detail/core/accessPoint/nifm_EthernetAccessPoint.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>

#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

EthernetInterface::EthernetInterface() NN_NOEXCEPT
    : m_IsInitialized(false),
      m_IsAvailable(false),
      m_Index(0),
      m_MacAddress(InvalidMacAddress),
      m_IpInterfaceChangeNotificationHandle(NULL)
{
}

EthernetInterface::~EthernetInterface() NN_NOEXCEPT
{
}

nn::Result EthernetInterface::Initialize() NN_NOEXCEPT
{
    NN_RESULT_THROW(Initialize( InvalidMacAddress ));
}

nn::Result EthernetInterface::Initialize( const MacAddress& macAddress ) NN_NOEXCEPT
{
    if( !IsInitialized() )
    {
        // インターフェースが見つからなくても初期化済みにして、関数が成功する状態にする
        m_IsInitialized = true;

        PMIB_IF_TABLE2 pMibIfTable2;
        if( ::GetIfTable2( &pMibIfTable2 ) == NO_ERROR )
        {
            for ( uint32_t i = 0; i < pMibIfTable2->NumEntries; ++i )
            {
                const MIB_IF_ROW2& mibIfRow2 = pMibIfTable2->Table[i];
                if( memcmp( macAddress.data, mibIfRow2.PhysicalAddress, MacAddress::Size ) == 0 &&
                    mibIfRow2.InterfaceAndOperStatusFlags.HardwareInterface == TRUE )
                {
                    NN_RESULT_DO(m_IpConfiguration.Initialize(macAddress));

                    m_MacAddress = macAddress;
                    m_Index = mibIfRow2.InterfaceIndex;
                    m_IsAvailable = true;

                    ::NotifyIpInterfaceChange(AF_INET, IpInterfaceChangeNotificationCallback, this, FALSE, &m_IpInterfaceChangeNotificationHandle);

                    break;
                }
            }
            ::FreeMibTable( pMibIfTable2 );
        }
    }

    NN_RESULT_SUCCESS;
}

bool EthernetInterface::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

bool EthernetInterface::IsAvailable() const NN_NOEXCEPT
{
    return IsInitialized() && m_IsAvailable;
}

nn::Result EthernetInterface::GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT
{
    NN_UNUSED(timeSpan);

    nn::Result result = ResultSuccess();

    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    MIB_IF_ROW2 mibIfRow2;
    ::SecureZeroMemory( (PVOID)&mibIfRow2, sizeof(mibIfRow2) );
    mibIfRow2.InterfaceIndex = m_Index;
    DWORD error = ::GetIfEntry2( &mibIfRow2 );

    if( error == NO_ERROR )
    {
        if( mibIfRow2.Type == IF_TYPE_ETHERNET_CSMACD && mibIfRow2.MediaConnectState == ::MediaConnectStateConnected )
        {
            if (pAccessPointList->CreateAndAdd<EthernetAccessPoint>(this) == nullptr)
            {
                NN_DETAIL_NIFM_WARN("Accesspoint list is full.\n");
                NN_RESULT_SUCCESS;
            }
        }
    }

    NN_RESULT_THROW(result);
}

MacAddress* EthernetInterface::GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT
{
    memcpy( pOutMacAddress->data, m_MacAddress.data, MacAddress::Size );
    return pOutMacAddress;
}

nn::Result EthernetInterface::ConnectImpl(
    const EthernetAccessPoint& ethernetAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(ethernetAccessPoint);
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("EthernetInterface::ConnectImpl(const EthernetAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultNetworkInterfaceNotAvailable());

    MIB_IF_ROW2 mibIfRow2;
    ::SecureZeroMemory( (PVOID)&mibIfRow2, sizeof(mibIfRow2) );
    mibIfRow2.InterfaceIndex = m_Index;
    DWORD error = ::GetIfEntry2( &mibIfRow2 );

    NN_RESULT_THROW_UNLESS(error == NO_ERROR, ResultNotConnected());
    NN_RESULT_THROW_UNLESS(mibIfRow2.Type == IF_TYPE_ETHERNET_CSMACD && mibIfRow2.MediaConnectState == ::MediaConnectStateConnected, ResultNotConnected());

    // TODO: accessPoint の検証

    nn::Result result = InitializeInternetConnection(networkProfile, aggregatedRequest);

    if (result.IsFailure())
    {
        DisconnectImpl(ethernetAccessPoint);
    }

    NN_RESULT_THROW(result);
}

nn::Result EthernetInterface::DisconnectImpl(const EthernetAccessPoint& ethernetAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(ethernetAccessPoint);

    NN_DETAIL_NIFM_TRACE("EthernetInterface::DisconnectImpl(const EthernetAccessPoint&)\n");

    NN_RESULT_SUCCESS;
}

nn::Result EthernetInterface::ReleaseImpl(const EthernetAccessPoint& ethernetAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(ethernetAccessPoint);

    NN_DETAIL_NIFM_TRACE("EthernetInterface::ReleaseImpl(const EthernetAccessPoint&)\n");

    NN_RESULT_SUCCESS;
}

void EthernetInterface::UpdateAccessPoint(EthernetAccessPoint* pInOutEthernetAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_UNUSED(pInOutEthernetAccessPoint);
    NN_UNUSED(networkProfile);
}

bool EthernetInterface::IsMediaConnected() const NN_NOEXCEPT
{
    if( IsAvailable() )
    {
        MIB_IFROW mibIfRow;
        mibIfRow.dwIndex = m_Index;

        DWORD error = ::GetIfEntry(&mibIfRow);

        if( error == NO_ERROR )
        {
            if( mibIfRow.dwOperStatus == IF_OPER_STATUS_OPERATIONAL )
            {
                return true;
            }
        }
    }

    return false;
}

VOID WINAPI EthernetInterface::IpInterfaceChangeNotificationCallback(PVOID pContext, ::PMIB_IPINTERFACE_ROW pIpInterfaceRow, ::MIB_NOTIFICATION_TYPE notificationType) NN_NOEXCEPT
{
    reinterpret_cast<EthernetInterface*>(pContext)->IpInterfaceChangeNotificationCallbackImpl(pIpInterfaceRow, notificationType);
}

void EthernetInterface::IpInterfaceChangeNotificationCallbackImpl(::PMIB_IPINTERFACE_ROW pIpInterfaceRow, ::MIB_NOTIFICATION_TYPE notificationType) NN_NOEXCEPT
{
    NN_UNUSED(pIpInterfaceRow);

    if( notificationType == ::MibDeleteInstance )
    {
        NotifyLinkDisconnected(ResultEthernetLinkDown());
    }
}

bool EthernetInterface::IsLinkUp() const NN_NOEXCEPT
{
    MIB_IF_ROW2 mibIfRow2;
    ::SecureZeroMemory((PVOID)&mibIfRow2, sizeof(mibIfRow2));
    mibIfRow2.InterfaceIndex = m_Index;
    DWORD error = ::GetIfEntry2(&mibIfRow2);

    return error == NO_ERROR &&
           mibIfRow2.Type == IF_TYPE_ETHERNET_CSMACD &&
           mibIfRow2.MediaConnectState == ::MediaConnectStateConnected;
}

nn::Result EthernetInterface::PutToSleepImpl() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result EthernetInterface::WakeUpImpl() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

}
}
}
