﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include "../detail/ngc_WorkBufAllocator.h"
#include "./ngc_ErrnoT.h"

namespace nn { namespace ngc { namespace detail {

// BIT must be uint32_t or uint64_t
// いくつのブロックが必要か返します。(端数切り上げ)
// ブロックは 32bit or 64bit で 1 ブロックです。
template<class BIT>
inline uint32_t GetBlockCount(uint32_t num_bits) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(BIT) * 8;  // NOLINT
    return (num_bits + kBitCount - 1) / kBitCount;
}

template<class BIT>
inline uint64_t GetBlockCount(uint64_t num_bits) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(BIT) * 8;  // NOLINT
    return (num_bits + kBitCount - 1) / kBitCount;
}

// ビット列 bv の idx 番目のビットを 1 にします。(0 始まり)
// 0011 (0x3) のとき、 0 ビット目は 1 , 1 ビット目 は 1 , 2 ビット目は 0 , 3 ビット目は 0です。
template<class BIT>
inline void SetBit(BIT* bv, uint32_t idx) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(*bv) * 8;
    uint32_t blk = idx / kBitCount;
    uint32_t ofs = idx % kBitCount;
    bv[blk] |= static_cast<BIT>(1) << ofs;
}

template<class BIT>
inline void SetBit(BIT* bv, uint64_t idx) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(*bv) * 8;
    uint64_t blk = idx / kBitCount;
    uint32_t ofs = static_cast<uint32_t>(idx % kBitCount);
    bv[blk] |= static_cast<BIT>(1) << ofs;
}

// ビット列 bv の idx 番目のビットを 0 にします。
template<class BIT>
inline void ResetBit(BIT* bv, uint32_t idx) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(*bv) * 8;
    uint32_t blk = idx / kBitCount;
    uint32_t ofs = idx % kBitCount;
    bv[blk] &= ~(static_cast<BIT>(1) << ofs);
}

template<class BIT>
inline void ResetBit(BIT* bv, uint64_t idx) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(*bv) * 8;
    uint64_t blk = idx / kBitCount;
    uint32_t ofs = static_cast<uint32_t>(idx % kBitCount);
    bv[blk] &= ~(static_cast<BIT>(1) << ofs);
}

// ビット列 bv の idx 番目のビットを返します。
template<class BIT>
inline bool GetBit(const BIT* bv, uint32_t idx) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(*bv) * 8;
    uint32_t blk = idx / kBitCount;
    uint32_t ofs = idx % kBitCount;
    return 0 != (bv[blk] & (static_cast<BIT>(1) << ofs));
}

template<class BIT>
inline bool GetBit(const BIT* bv, uint64_t idx) NN_NOEXCEPT
{
    static const int kBitCount = sizeof(*bv) * 8;
    uint64_t blk = idx / kBitCount;
    uint32_t ofs = static_cast<uint32_t>(idx % kBitCount);
    return 0 != (bv[blk] & (static_cast<BIT>(1) << ofs));
}

template <class T>
class Resetter final
{
public:
    inline explicit Resetter(T* obj) NN_NOEXCEPT
        : m_Obj(obj), m_IsOk(false)
    {
    }
    inline ~Resetter() NN_NOEXCEPT
    {
        if (!m_IsOk) m_Obj->Reset();
    }
    inline bool Ok() NN_NOEXCEPT
    {
        m_IsOk = true;
        return true;
    }

private:
    T* m_Obj;
    bool m_IsOk;
};

template <class T>
class PlainArray32 final
{
public:
    PlainArray32() NN_NOEXCEPT : m_Count(0), m_Value(), m_pAllocator(NULL)
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
    }
    explicit PlainArray32(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Count(0), m_Value(), m_pAllocator(pAllocator)
    {
        NN_STATIC_ASSERT(std::is_pod<T>::value);
    }
    ~PlainArray32() NN_NOEXCEPT
    {
        this->Reset();
    }
    inline PlainArray32(PlainArray32&& rhs) : m_Count(), m_Value(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline PlainArray32& operator=(PlainArray32&& rhs) NN_NOEXCEPT
    {
        PlainArray32 tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }

    void Swap(PlainArray32& rhs) NN_NOEXCEPT
    {
        using std::swap;
        swap(m_Count, rhs.m_Count);
        m_Value.Swap(rhs.m_Value);
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }

    bool Init(uint32_t count) NN_NOEXCEPT;
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    void ReleaseAllocator() NN_NOEXCEPT;
    T& operator[](size_t idx)
    {
        NN_SDK_ASSERT(idx < m_Count);
        return m_Value[idx];
    }
    const T& operator[](size_t idx) const
    {
        NN_SDK_ASSERT(idx < m_Count);
        return m_Value[idx];
    }
    size_t Size() const NN_NOEXCEPT
    {
        return m_Count;
    }
    size_t MemSize() const NN_NOEXCEPT
    {
        size_t array_size = sizeof(T) * m_Count;
        return sizeof(m_Count) + sizeof(m_Value) + array_size + sizeof(m_pAllocator);
    }
    void Reset() NN_NOEXCEPT;
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    uint32_t m_Count;
    ArrayManager<T> m_Value;
    WorkBufAllocator* m_pAllocator;

    PlainArray32(const PlainArray32&) = delete;
    void operator=(const PlainArray32&) = delete;
};

template <class T>
bool PlainArray32<T>::Init(uint32_t count) NN_NOEXCEPT
{
    if (m_Value)
    {
        return false;
    }
    if (count == 0)
    {
        m_Count = 0;
        return true;
    }
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(T) * count);
        m_Value.Reset(pTmp ? new (pTmp)T[count] : NULL, m_pAllocator);
    }
    else
    {
        m_Value.Reset(new (std::nothrow) T[count]);
    }
    if (!m_Value)
    {
        return false;
    }
    m_Count = count;
    memset(m_Value.Get(), 0, sizeof(T) * count);
    return true;
}

template <class T>
bool PlainArray32<T>::SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;
    return true;
}

template <class T>
void PlainArray32<T>::ReleaseAllocator() NN_NOEXCEPT
{
    this->Reset();
}

template <class T>
inline void PlainArray32<T>::Reset() NN_NOEXCEPT
{
    m_Count = 0;
    m_Value.Reset();
    m_pAllocator = NULL;
}

template <class T>
bool PlainArray32<T>::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!w->Write(m_Count))
    {
        return false;
    }
    if (m_Count > 0 && !w->WriteArray(m_Value.Get(), m_Count))
    {
        return false;
    }
    return true;
}

template <class T>
bool PlainArray32<T>::Import(BinaryReader* r) NN_NOEXCEPT
{
    Resetter<PlainArray32<T> > rst(this);
    if (!r->Read(&m_Count))
    {
        return false;
    }
    if (m_Count == 0)
    {
        m_Value.Reset();
        return true;
    }
    if (m_pAllocator)
    {
        void* pTmp = m_pAllocator->Allocate(sizeof(T) * m_Count);
        m_Value.Reset(pTmp ? new (pTmp)T[m_Count] : NULL, m_pAllocator);
    }
    else
    {
        m_Value.Reset(new (std::nothrow) T[m_Count]);
    }
    if (!m_Value)
    {
        return false;
    }
    if (m_Count != r->ReadArray(m_Value.Get(), m_Count))
    {
        return false;
    }
    rst.Ok();
    return true;
}

/**
 * @brief   ビットベクトルを表現するクラスです。
 * @details Rank や Select 操作のない、単純な 32bit 区切りのビットを操作するだけのクラス
 */
class BitVector32 final
{
public:
    BitVector32() NN_NOEXCEPT : m_Size(0), m_BitVec(), m_pAllocator(NULL) {}
    explicit BitVector32(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Size(0), m_BitVec(), m_pAllocator(pAllocator) {}
    ~BitVector32() NN_NOEXCEPT;
    inline BitVector32(BitVector32&& rhs) : m_Size(), m_BitVec(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline BitVector32& operator=(BitVector32&& rhs) NN_NOEXCEPT
    {
        BitVector32 tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    /**
     * @brief   rhs と m_Size と m_BitVec を入れ替えます。
     */
    void Swap(BitVector32& rhs) NN_NOEXCEPT
    {
        using std::swap;
        swap(m_Size, rhs.m_Size);
        m_BitVec.Swap(rhs.m_BitVec);
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }
    /**
     * @brief       初期化します。
     * @param[in]   size    ビットの長さ
     */
    bool Init(uint32_t size) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief   集合に32bit符号なし整数を追加します。
     *          nth 番目のビットを 1 にします。
     */
    bool TurnOn(uint32_t nth) NN_NOEXCEPT
    {
        if (nth >= m_Size) return false;
        SetBit(m_BitVec.Get(), nth);
        return true;
    }
    /**
     * @brief   nth 番目のビットを 0 にします。
     */
    bool TurnOff(uint32_t nth) NN_NOEXCEPT
    {
        if (nth >= m_Size) return false;
        ResetBit(m_BitVec.Get(), nth);
        return true;
    }
    // ビットでアクセスするための [] 定義
    bool operator[](uint32_t nth) const NN_NOEXCEPT
    {
        return nth < m_Size ? GetBit(m_BitVec.Get(), nth) : false;
    }
    /**
     * @brief   ビット列の長さを返します。
     */
    uint32_t Size() const NN_NOEXCEPT
    {
        return m_Size;
    }
    /**
     * @brief   32bit 配列の生ポインタを返します。
     */
    const uint32_t* Data() const NN_NOEXCEPT
    {
        return reinterpret_cast<const uint32_t*>(m_BitVec.Get());
    }
    /**
     * @brief   コンストラクタ呼び出し直後の状態に戻します。
     * @details m_Allocator の設定も NULL にします。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;
    /**
     * @brief   オブジェクトを(ファイルに)書き出します。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief   書き出されたオブジェクトを読み出します。
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    uint32_t m_Size;  // m_Size is bit length
    ArrayManager<uint32_t> m_BitVec;
    WorkBufAllocator* m_pAllocator;

    BitVector32(const BitVector32&) = delete;
    void operator=(const BitVector32&) = delete;
};

/**
 * @brief   rank 用の情報を持つクラスです。
 * @details SbvRank はビットベクトル自体は内部に持たず、 2 段階の Rank 用情報しか持たない
 *          Rank を O(1) で行える。 Select は二分探索するので O(lg(n))
 */
class SbvRank final {
    // LV-A has cumulative popCounts for each 256bits block.
    // LV-B has popCounts in a 256bits block.
    // We can calcuate 'rank' in O(1).
public:
    SbvRank() NN_NOEXCEPT : m_LvA(), m_LvB(), m_pAllocator(NULL) {}
    explicit SbvRank(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_LvA(), m_LvB(), m_pAllocator(pAllocator) {}
    ~SbvRank() NN_NOEXCEPT;
    inline SbvRank(SbvRank&& rhs) : m_LvA(), m_LvB(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline SbvRank& operator=(SbvRank&& rhs) NN_NOEXCEPT
    {
        SbvRank tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    /**
     * @brief   rhs と中身を入れ替えます。
     */
    void Swap(SbvRank& rhs) NN_NOEXCEPT
    {
        m_LvA.Swap(rhs.m_LvA);
        m_LvB.Swap(rhs.m_LvB);
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief       Rank 用辞書を作成します。
     * @param[in]   pBv     作成元のビットベクトル
     * @param[in]   size    pBv の長さ(ビットの数)
     */
    bool Build(const uint32_t* pBv, uint32_t size) NN_NOEXCEPT;
    /**
     * @brief   Rank操作を行います。[0..pos] 内に含まれる 1 の数を返します。
     */
    uint32_t Rank1(const uint32_t* pBv, uint32_t pos) const NN_NOEXCEPT
    {
        using ::nn::ngc::detail::CalcRank1;
        return CalcRank1(pos, pBv, m_LvA.Get(), m_LvB.Get());
    }
    /**
     * @brief   Rank操作を行います。[0..pos] 内に含まれる 0 の数を返します。
     */
    uint32_t Rank0(const uint32_t* pBv, uint32_t pos) const NN_NOEXCEPT
    {
        return pos + 1 - this->Rank1(pBv, pos);
    }
    /**
     * @brief   nth 番目の 1 ビットの場所を返します。 nth は 0 から開始します。
     */
    int32_t Select1(const uint32_t* pBv, uint32_t size, uint32_t nth) const NN_NOEXCEPT
    {
        using ::nn::ngc::detail::CalcSelect1;
        if (nth >= this->Rank1(pBv, size - 1))
        {
            return -1;
        }
        int32_t ans = CalcSelect1(nth, size, pBv, m_LvA.Get(), m_LvB.Get());
        NN_SDK_ASSERT(ans < static_cast<int>(size));
        return ans;
    }
    /**
     * @brief   nth 番目の 0 ビットの場所を返します。 nth は 0 から開始します。
     */
    int32_t Select0(const uint32_t* pBv, uint32_t size, uint32_t nth) const NN_NOEXCEPT
    {
        using ::nn::ngc::detail::CalcSelect0;
        if (nth >= this->Rank0(pBv, size - 1)) return -1;
        int32_t ans = CalcSelect0(nth, size, pBv, m_LvA.Get(), m_LvB.Get());
        NN_SDK_ASSERT(ans < static_cast<int>(size));
        return ans;
    }
    /**
     * @brief   コンストラクタ直後の状態に戻します。
     * @details SetAllocator() を呼び出している場合アロケータ情報は初期化されません。
     *          アロケータ情報の初期化には Reset() の後に ReleaseAllocator() を呼び出してください。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief   明示的に確保するサイズを返します。
     */
    size_t MemSize(uint32_t size) const NN_NOEXCEPT;
    /**
     * @brief   オブジェクトを(ファイルに)書き出します。
     */
    bool Export(BinaryWriter* w, uint32_t size) const NN_NOEXCEPT;
    /**
     * @brief   書き出されたオブジェクトを読み出します。
     */
    bool Import(BinaryReader* r, uint32_t size) NN_NOEXCEPT;

private:
    // lv_a 作成先辞書、レベル A(256ビット区切り)
    //      256 ビットごとに、その位置での Rank1 の結果を保存
    ArrayManager<uint32_t> m_LvA;
    // lv_b 作成先辞書、レベル B(32 ビット区切り)
    //      32 ビットごとに、直前の lv_a からの Rank1 の差分を保存
    ArrayManager<uint8_t> m_LvB;
    WorkBufAllocator* m_pAllocator;

    SbvRank(const SbvRank&) = delete;
    void operator=(const SbvRank&) = delete;
};

/**
 * @brief   Set を与えてもらいそれから定数時間で Select 操作をすることができるクラス
 */
class SbvSelect final {
public:
    // We can calcuate 'select' in O(1).
    SbvSelect() NN_NOEXCEPT : m_pAllocator(NULL){}
    explicit SbvSelect(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_pAllocator(pAllocator) {}
    ~SbvSelect() NN_NOEXCEPT;
    inline SbvSelect(SbvSelect&& rhs) : m_ClumpArray(), m_Blocks(), m_Asserted(), m_RankForBlocks(), m_RankForAsserted()
    {
        this->Swap(rhs);
    }
    inline SbvSelect& operator=(SbvSelect&& rhs) NN_NOEXCEPT
    {
        SbvSelect tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    /**
     * @brief   要素をスワップします。
     */
    void Swap(SbvSelect& rhs) NN_NOEXCEPT
    {
        using std::swap;
        m_ClumpArray.Swap(rhs.m_ClumpArray);
        m_Blocks.Swap(rhs.m_Blocks);
        m_Asserted.Swap(rhs.m_Asserted);
        m_RankForBlocks.Swap(rhs.m_RankForBlocks);
        m_RankForAsserted.Swap(rhs.m_RankForAsserted);
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief   Select 用補助データを生成します
     */
    bool Build(const uint32_t* pBv, uint32_t size) NN_NOEXCEPT;
    /**
     * @brief   コンストラクタ直後の状態に戻します。
     * @details SetAllocator() を呼び出している場合アロケータ情報は初期化されません。
     *          アロケータ情報の初期化には Reset() の後に ReleaseAllocator() を呼び出してください。
     */
    void Reset() NN_NOEXCEPT;
    int32_t Select(const Set& set, uint32_t nth) const NN_NOEXCEPT;
    size_t MemSize() const NN_NOEXCEPT;
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    // [0 であるブロック(の塊)に関するビットベクトル]
    // 0 であるブロックの塊の数ぶんの配列
    // 要素数 :       0 であるブロックの塊の数
    // 保存される値 : ここまでの 0 ブロックの総数
    PlainArray32<uint32_t> m_ClumpArray;
    // [0 でないブロックに関するビットベクトル]
    // 長さ(ビットの数) : 0 でないブロックの数ぶんのビットベクトル
    // 1 になっているビット : 自分が 0 ブロックの直後のブロックである
    // m_Blocks に対する Rank(i) で i つめの 0 でないブロックの前に
    // いくつの 0 のブロックの塊があるかがわかる
    BitVector32 m_Blocks;
    // [1 であるビットに関するビットベクトル]
    // 長さ(ビットの数) : 1 になっているビットの数ぶんのビットベクトル
    // 1 になっているビット : ブロックごとの 1 の数の区切り
    //                        1 が出た次からの 0 の連続数がそのブロックの 1 の数
    // m_Asserted に対する Rank(i) で i 番目の 1 が(0 の塊のブロックを除いて)何番目のブロックに
    // 所属しているかがわかる
    BitVector32 m_Asserted;
    SbvRank m_RankForBlocks;    // m_Blocks の Rank 用補助データ
    SbvRank m_RankForAsserted;  // m_Asserted の Rank 用補助データ
    WorkBufAllocator* m_pAllocator;

    SbvSelect(const SbvSelect&) = delete;
    void operator=(const SbvSelect&) = delete;
};

}}} // nn::ngc::detail
