﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <utility>

#include <nn/nn_Common.h>
#include "../detail/ngc_WorkBufAllocator.h"

#include "./ngc_ErrnoT.h"
#include "./ngc_SmartBitmap.h"
#include "./ngc_BinaryReader.h"
#include "./ngc_BinaryWriter.h"

namespace nn { namespace ngc { namespace detail {

static const unsigned int SbvBlockSize = 32;

/**
 * @brief   Rank/Select操作つきの32bit符号なし整数の集合を保持する簡潔データ構造です。
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 *          このクラスは32bit符号なし整数の集合をビットベクトルの形で保持していて、Rank操作と呼ばれる操作を でサポートしています。 Rank操作についての詳しい説明は Rank0()/Rank1() を御覧ください。
 *          このクラスの利用方法は、以下のとおりです。
 *              1. コンストラクタでオブジェクトを作成します。
 *              2. Init() で定義域を設定してビットベクトルを初期化します。
 *              3. TurnOn()で整数を集合に加えます。
 *              4. Build()でRank操作を定数時間で行うための辞書を構築します。
 *              5. Rank1(), Select1()といった操作が実行可能になります。
 *              6. なお集合の内容を変更したい場合は、TurnOn(), TurnOff()で変更した後再びBuild()を実行します。
 *          データをExport()で書きだした場合はコンストラクタの実行後にImport()を呼び出すことで利用できるようになります。 このクラスは定義域のビットベクトルをすべて保持していて高速なRank操作が可能ですが、メモリを多く消費します。 疎な集合の場合は、SparseSetを用いる方をお勧めします。
 */
class  Set final
{
public:
    /**
     * @brief   インデックスとなる整数型のtypedefです。
     */
    typedef uint32_t IdxType;

    Set() NN_NOEXCEPT : m_Prv(NULL), m_pAllocator(NULL) {}
    explicit Set(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Prv(NULL), m_pAllocator(pAllocator) {}
    ~Set() NN_NOEXCEPT;
    inline Set(Set&& rhs) NN_NOEXCEPT : m_Prv(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline Set& operator=(Set&& rhs) NN_NOEXCEPT
    {
        Set tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    /**
     * @brief   rhs と中身を入れ替えます。
     */
    void Swap(Set& rhs) NN_NOEXCEPT
    {
        SetPrivate* tmp = rhs.m_Prv;
        rhs.m_Prv = m_Prv;
        m_Prv = tmp;
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }

    /**
     * @brief       初期化を行います。
     * @param[in]   bvSize  ビットベクトルのサイズ
     * @return      成功した場合は true
     * @details     0 以上 bvSize 未満の値を集合に加えることができるようになります。
     */
    bool Init(uint32_t bvSize) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief   集合に32bit符号なし整数を追加します。
     * @return  成功した場合は true
     * @details idx 番目のビットを立てます。
     */
    bool TurnOn(uint32_t idx) NN_NOEXCEPT;
    /**
     * @brief   集合から32bit符号なし整数を取り除きます。
     * @return  成功した場合は true
     * @details idx 番目のビットを 0 に戻します。
     */
    bool TurnOff(uint32_t idx) NN_NOEXCEPT;
    /**
     * @brief   Rank 辞書を構築します。
     * @return  成功した場合は true
     */
    bool Build() NN_NOEXCEPT;
    /**
     * @brief       値が集合に含まれているかどうかをテストします。
     * @param[in]   idx 32bit符号なし整数
     * @return      集合にidx が含まれている場合はtrue
     */
    bool Has(uint32_t idx) const NN_NOEXCEPT;

    /**
     * @brief   Rank操作を行います。[0..idx] 内に含まれる 1 の数を返します。
     * @param[in]   idx 32bit 符号なし整数
     * @return      集合に含まれるidx 以下の値の数
     * @details     集合内のidx 以下の値の数を返します。 すわなち、ビットベクトル[0 .. idx]内に含まれる1の数を返します。 この操作は定数時間で行われます。
     */
    uint32_t Rank1(uint32_t idx) const NN_NOEXCEPT;
    /**
     * @brief       Rank操作を行います。[0..idx] 内に含まれる 0 の数を返します。
     * @param[in]   idx 32bit 符号なし整数
     * @return      集合に含まれないidx 以下の値の数
     * @details     idx 以下の値で集合に含まれない要素の数を返します。 すわなち、ビットベクトル[0 .. idx]内に含まれる1の数を返します。 この操作は定数時間で行われます。
     */
    uint32_t Rank0(uint32_t idx) const NN_NOEXCEPT
    {
        uint32_t rank1 = this->Rank1(idx);
        return idx + 1 - rank1;
    }
    /**
     * @brief   nth 番目の 1 ビットの場所を返します。 nth は 0 から開始します。
     * @param[in]   nth 0 以上の整数
     * @return      nth 番目(基数は0)の要素が存在するならばその値。なければ-1
     * @details     すなわち、ビットベクトルのnth 番目のONになっっているビットの位置を返します。 この操作は内部でRank1()を利用して二分探索で行われます。 通常の場合は、Select操作を O(1) でサポートするSbvを利用してください。
     */
    int32_t Select1(uint32_t nth) const NN_NOEXCEPT;
    /**
     * @brief       nth 番目の 0 ビットの場所を返します。 nth は 0 から開始します。
     * @param[in]   nth 0 以上の整数
     * @return      nth 番目(基数は0)の要素が存在するならばその値。なければ-1
     * @details     すなわち、ビットベクトルのnth 番目のOFFになっっているビットの位置を返します。 この操作は内部でRank1()を利用して二分探索で行われます。
     */
    int32_t Select0(uint32_t nth) const NN_NOEXCEPT;

    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     * @details 実際にシステムが確保するメモリ量はアライメントやヒープの管理領域の関係上この関数の返り値より大きい可能性があります。
     */
    size_t MemSize() const NN_NOEXCEPT;
    /**
     * @brief   ビットベクトルのサイズを返します。
     * @return  集合の定義域のサイズ(ビットベクトルのサイズ)
     * @details Init() に渡した数が返ります。
     */
    uint32_t GetBitVectorSize() const NN_NOEXCEPT;
    /**
     * @brief   ビットベクトルへのポインタを返します。
     * @return  ビットベクトルへのポインタ
     * @details  LSB（下の桁）から順番に格納されています。
     */
    const uint32_t* GetBitVector() const NN_NOEXCEPT;
    /**
     * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
     * @details SetAllocator() を呼び出している場合アロケータ情報は初期化されません。
     *          アロケータ情報の初期化には Reset() の後に ReleaseAllocator() を呼び出してください。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出しようオブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータは Import()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      インポートが成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    struct SetPrivate;
    struct SetPrivateDeleter;

    mutable SetPrivate* m_Prv;
    WorkBufAllocator* m_pAllocator;

    Set(const Set&) = delete;
    void operator=(const Set&) = delete;
};

/**
 * @brief   Rank/Select操作つきの32bit符号なし整数の集合を保持する簡潔データ構造です。
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 *          このクラスは32bit符号なし整数の集合をビットベクトルの形で保持していて、Rank操作に加えてSelect操作を O(1) でサポートしています。 Rank/Select操作を利用することで近年提案されているコンパクトなデータ構造に対する高速な操作を実現することができます。
 *          このクラスの利用方法は、以下のとおりです。
 *              - コンストラクタでオブジェクトを作成します。
 *              - Init()で定義域を設定してビットベクトルを初期化します。
 *              - TurnOn()で整数を集合に加えます。
 *              - Build()でRank/Select操作を定数時間で行うための辞書を構築します。
 *              - 以上でRank/Select操作が利用できるようになります。
 *          データをExport()で書きだした場合はコンストラクタの実行後にImport()を呼び出すことで利用できるようになります。 このクラスは定義域のビットベクトルをすべて保持していて高速なRank操作が可能ですが、メモリを多く消費します。 疎な集合の場合は、SparseSetを用いる方をお勧めします。
 */
class  Sbv final
{
public:
    /**
     * @brief   インデックスとなる整数型のtypedefです。
     */
    typedef uint32_t IdxType;

    Sbv() NN_NOEXCEPT : m_Prv(NULL), m_pAllocator(NULL) {}
    explicit Sbv(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Prv(NULL), m_pAllocator(pAllocator) {}
    ~Sbv() NN_NOEXCEPT;
    inline Sbv(Sbv&& rhs) NN_NOEXCEPT : m_Prv(), m_Set(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline Sbv& operator=(Sbv&& rhs) NN_NOEXCEPT
    {
        Sbv tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }

    /**
     * @brief   rhs と中身を入れ替えます。
     */
    void Swap(Sbv& rhs) NN_NOEXCEPT
    {
        using std::swap;
        m_Set.Swap(rhs.m_Set);
        SbvPrivate* tmp = rhs.m_Prv;
        rhs.m_Prv = m_Prv;
        m_Prv = tmp;
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }
    /**
     * @brief       オブジェクトを初期化します。
     * @param[in]   bvSize ビットベクトルのサイズ
     * @return      成功した場合は true
     * @details     0以上bvSize 未満の値を集合に加えることができるようになります。
     */
    bool Init(uint32_t bvSize) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief   集合に32bit符号なし整数を追加します。
     * @return  成功した場合は true
     * @details idx 番目のビットを立てます。
     */
    bool TurnOn(uint32_t idx) NN_NOEXCEPT
    {
        return m_Set.TurnOn(idx);
    }
    /**
     * @brief   集合から32bit符号なし整数を取り除きます。
     * @return  成功した場合は true
     * @details idx 番目のビットを 0 に戻します。
     */
    bool TurnOff(uint32_t idx) NN_NOEXCEPT
    {
        return m_Set.TurnOff(idx);
    }
    /**
     * @brief   Rank/Select 辞書を構築します。
     * @return  成功した場合は true
     */
    bool Build() NN_NOEXCEPT;
    /**
     * @brief       値が集合に含まれているかどうかをテストします。
     * @param[in]   idx 32bit 符号なし整数
     * @return      集合にidx が含まれている場合はtrue
     */
    bool Has(uint32_t idx) const NN_NOEXCEPT
    {
        return m_Set.Has(idx);
    }
    /**
     * @brief       Rank操作を行います。[0..idx] 内に含まれる 1 の数を返します。
     * @param[in]   idx 32bit 符号なし整数
     * @return      集合に含まれるidx 以下の値の数
     * @details     集合内のidx 以下の値の数を返します。 すわなち、ビットベクトル[0 .. idx]内に含まれる1の数を返します。 この操作は O(1) で行われます。
     */
    uint32_t Rank1(uint32_t idx) const NN_NOEXCEPT
    {
        return m_Set.Rank1(idx);
    }
    /**
     * @brief       Rank操作を行います。[0..idx] 内に含まれる 0 の数を返します。
     * @param[in]   idx 32bit 符号なし整数
     * @return      集合に含まれないidx 以下の値の数
     * @details     idx 以下の値で集合に含まれない要素の数を返します。 すわなち、ビットベクトル[0 .. idx]内に含まれる1の数を返します。 この操作は O(1) で行われます。
     */
    uint32_t Rank0(uint32_t idx) const NN_NOEXCEPT
    {
        return m_Set.Rank0(idx);
    }
    /**
     * @brief       nth 番目の 1 ビットの場所を返します。 nth は 0 から開始します。
     * @param[in]   nth 0 以上の整数
     * @return      nth 番目(基数は0)の1ビットが存在するならばその値。なければ-1
     */
    int32_t Select1(uint32_t nth) const NN_NOEXCEPT;
    /**
     * @brief   nth 番目の 0 ビットの場所を返します。 nth は 0 から開始します。
     * @param[in]   nth 0 以上の整数
     * @return      nth 番目(基数は0)の0ビットが存在するならばその値。なければ-1
     */
    int32_t Select0(uint32_t nth) const NN_NOEXCEPT
    {
        return m_Set.Select0(nth);
    }
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;
    /**
     * @brief   ビットベクトルのサイズを返します。
     * @details Init() に渡した数が返ります。
     */
    uint32_t GetBitVectorSize() const NN_NOEXCEPT
    {
        return m_Set.GetBitVectorSize();
    }
    /**
     * @brief   ビットベクトルへのポインタを返します。
     * @details  LSB（下の桁）から順番に格納されています。
     */
    const uint32_t* GetBitVector() const NN_NOEXCEPT
    {
        return m_Set.GetBitVector();
    }
    /**
     * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
     * @details SetAllocator() を呼び出している場合アロケータ情報は初期化されません。
     *          アロケータ情報の初期化には Reset() の後に ReleaseAllocator() を呼び出してください。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出し用オブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータはImport()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    struct SbvPrivate;
    struct SbvPrivateDeleter;

    SbvPrivate* m_Prv;
    Set m_Set;
    WorkBufAllocator* m_pAllocator;

    Sbv(const Sbv&) = delete;
    void operator=(const Sbv&) = delete;
};

/**
 * @brief   疎な64bit符号なし整数の集合を保持する簡潔データ構造です。
 * @pre     64bit の時 1 <= bvSize <= 0x1FFFFFFFFF
 * @pre     32bit の時 1 <= bvSize <= 0xFFFFFFFF
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 *          疎な集合の場合、SetやSbvよりも小さなメモリ領域でRank/Select操作をサポートした整数の集合(ビットベクトル)をサポートします。 疎な集合というのは、目安としては、全体の10パーセントかそれ以下がONになっている集合と考えてください。
 *          また、集合にある整数が含まれているかどうはを判断する操作(Has())は直観と異なり、Rank操作を実行した後にSelect操作を実行しているので両者より低速になります。
 *          このクラスの利用方法は、以下のとおりです。
 *              - コンストラクタでオブジェクトを作成します。
 *              - Init()で定義域を設定してビットベクトルを初期化します。
 *              - TurnOn()で整数を集合に加えます。
 *              - Build()でRank/Select操作を定数時間で行うための辞書を構築します。
 *              - 以上でRank/Select操作が利用できるようになります。
 *          データをExport()で書きだした場合はコンストラクタの実行後にImport()を呼び出すことで利用できるようになります。
 *          このオブジェクトに対するHas()等による参照は、Build()の実行後に行う必要があります。 内部で参照動作のためのインデックスを作成する必要があるからです。
 */
class  SparseSet final
{
public:
    // index is 64-bit and # of on-bits is 32-bit.
    // In other words, rank is 64-bit index, and select is 32-bit index
    typedef uint64_t IdxType;
    SparseSet() NN_NOEXCEPT : m_BitvectorSize(0),
                              m_LowestIdx(0xFFFFFFFF),
                              m_HighestIdx(0),
                              m_NumOnBits(0),
                              m_Width(0),
                              m_LowerBitsLength(0),
                              m_pLowerBits(NULL),
                              m_pAllocator(NULL) {}
    explicit SparseSet(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_BitvectorSize(0),
                                                                   m_LowestIdx(0xFFFFFFFF),
                                                                   m_HighestIdx(0),
                                                                   m_NumOnBits(0),
                                                                   m_Width(0),
                                                                   m_LowerBitsLength(0),
                                                                   m_pLowerBits(NULL),
                                                                   m_pAllocator(pAllocator) {}
    ~SparseSet() NN_NOEXCEPT;
    SparseSet(SparseSet&& rhs) NN_NOEXCEPT;
    SparseSet& operator=(SparseSet&& rhs) NN_NOEXCEPT;
    /**
     * @brief       初期化を行います。
     * @param[in]   bvSize ビットベクトルのサイズ
     * @return  成功した場合は true
     */
    bool Init(uint64_t bvSize) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief       集合に64bit符号なし整数を追加します。
     * @param[in]   idx 集合に加える整数
     * @return      成功した場合は true
     */
    bool TurnOn(uint64_t idx) NN_NOEXCEPT;
    /**
     * @brief   Rank/Select 辞書を構築します。
     * @return  成功した場合は true
     */
    bool Build() NN_NOEXCEPT;
    /**
     * @brief       値が集合に含まれているかどうかをテストします。
     * @param[out]  pOutRank    Rank の値
     * @param[in]   idx         64bit 符号なし整数
     * @pre         Build() 後である
     * @details     NULL でなければランク値が格納されます
     */
    bool Has(uint32_t* pOutRank, uint64_t idx) const NN_NOEXCEPT
    {
        // called after Build()
        uint32_t rank1 = this->Rank1(idx);
        if (pOutRank)
        {
            *pOutRank = rank1;
        }
        if (rank1 == 0)
        {
            return false;
        }
        int64_t select1 = this->Select1Ex(rank1 - 1);
        NN_SDK_ASSERT(select1 >= 0);
        return idx == static_cast<uint64_t>(select1);
    }
    /**
     * @brief       値が集合に含まれているかどうかをテストします。
     */
    bool Has(uint64_t idx) const NN_NOEXCEPT
    {
        return this->Has(NULL, idx);
    }
    /**
     * @brief       Rank操作を行います。[0..idx] 内に含まれる 1 の数を返します。
     * @param[in]   idx 64bit 符号なし整数
     * @return      集合に含まれるidx 以下の値の数
     */
    uint32_t Rank1(uint64_t idx) const NN_NOEXCEPT;
    /**
     * @brief       Rank操作を行います。[0..idx] 内に含まれる 0 の数を返します。
     * @param[in]   idx 64bit 符号なし整数
     * @return      集合に含まれないidx 以下の値の数
     */
    uint32_t Rank0(uint64_t idx) const NN_NOEXCEPT
    {
        uint32_t rank1 = this->Rank1(idx);
        return static_cast<uint32_t>(idx + 1 - rank1);
    }
    /**
     * @brief       nth 番目の 1 ビットの場所を返します。 nth は 0 から開始します。
     * @param[in]   nth 0以上の32bit整数
     * @return      nth 番目(基数は0)の要素が存在するならばその値。なければ-1
     */
    int32_t Select1(uint32_t nth) const NN_NOEXCEPT;
    /**
     * @brief       nth 番目の 1 ビットの場所を返します。 nth は 0 から開始します。
     * @param[in]   nth 0以上の整数
     * @return      nth 番目(基数は0)の要素が存在するならばその値。なければ-1
     * @details     64bit の値を返す可能性があるときに利用する
     */
    int64_t Select1Ex(uint32_t nth) const NN_NOEXCEPT;
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;
    /**
     * @brief   ビットベクトルのサイズを返します。
     * @details Init()関数に渡した値と同じ値を返します。
     */
    uint64_t GetBitVectorSize() const NN_NOEXCEPT
    {
        return m_BitvectorSize;
    }
    /**
     * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
     * @details SetAllocator() を呼び出している場合アロケータ情報は初期化されません。
     *          アロケータ情報の初期化には Reset() の後に ReleaseAllocator() を呼び出してください。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出し用オブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータはImport()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      インポートが成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;
    void Swap(SparseSet& rhs) NN_NOEXCEPT;

private:
    /**
     * @brief   下位ビットデータの中の任意の位置のビット列を返します。
     * @details m_pLowerBits の中の pos 番目のビットから num ビットぶんの値を返す
     *          e.g. b_ = 010101010 pos = 6 num = 3 の場合 010 が帰る
     */
    uint32_t GetBits(uint32_t pos, size_t num) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(num <= 32);
        uint64_t idx64 = pos / detail::SbvBlockSize;
        NN_SDK_ASSERT(idx64 <= SIZE_MAX);
        size_t idx = static_cast<size_t>(idx64);
        size_t ofs = static_cast<size_t>(pos % detail::SbvBlockSize);
        if (ofs + num > 32)
        {
            uint64_t b = m_pLowerBits[idx + 1];
            b <<= 32;
            b += m_pLowerBits[idx];
            return static_cast<uint32_t>((b >> ofs) & ((1 << num) - 1));
        }
        else
        {
            return (m_pLowerBits[idx] >> ofs) & ((1 << num) - 1);
        }
    }

private:
    uint64_t m_BitvectorSize;   // ビットベクトルのサイズ(ビット数)
    uint64_t m_LowestIdx;       // 1 になっているビットの中で、一番小さいインデックス
    uint64_t m_HighestIdx;      // 1 になっているビットの中で、一番大きいインデックス
    uint32_t m_NumOnBits;       // 1 になっているビットの数
    uint32_t m_Width;           // 下位ビットの幅
    uint32_t m_LowerBitsLength; // m_pLowerBits の配列の長さ
    uint32_t* m_pLowerBits;     // 下位ビットデータ（途中 temp としてビットベクトルを 32bit ごとに区切ったブロックの配列を入れられる)
    Sbv m_Sbv;                  // 上位ビットの数のビット列
    WorkBufAllocator* m_pAllocator;

    SparseSet(const SparseSet&) = delete;
    void operator=(const SparseSet&) = delete;
};

/**
 * @brief   追記可能な圧縮された整数配列です。
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 *          整数配列を64個ずつのブロックにわけ、ブロックの代表値と代表値からの64個の差分を記録しています。 差分値はブロックごとに一定のビット幅に詰めて格納しています。 代表値と差分を足すというごく簡単な計算で値に対するランダムアクセスが可能です。
 *          簡単な方法の割には小さい値の割合が多いデータを格納する場合の圧縮率が高く、そのまま配列を格納した場合の1/2から1/4のサイズにすることができる場合が多いです。
 */
class  CompressedArray final
{
    static const size_t UnitSize = 64;

public:
    CompressedArray() NN_NOEXCEPT
        : m_pHeader(NULL), m_pData(NULL), m_HeaderSize(0), m_HeaderCapacity(0),
          m_DataSize(0), m_DataCapacity(0), m_WorkIdx(0), m_pAllocator(NULL) {}
    explicit CompressedArray(WorkBufAllocator* pAllocator) NN_NOEXCEPT
        : m_pHeader(NULL), m_pData(NULL), m_HeaderSize(0), m_HeaderCapacity(0),
          m_DataSize(0), m_DataCapacity(0), m_WorkIdx(0), m_pAllocator(pAllocator) {}
    ~CompressedArray() NN_NOEXCEPT;
    CompressedArray(CompressedArray&& rhs) NN_NOEXCEPT;
    CompressedArray& operator=(CompressedArray&& rhs) NN_NOEXCEPT;
    /**
     * @brief   オブジェクトの内容をスワップします。
     */
    void Swap(CompressedArray& rhs) NN_NOEXCEPT;
    /**
     * @brief       データを追加します。64個単位の長さになるとデータを圧縮します。
     * @param[in]   x   追加する値
     * @return      成功した場合は true
     */
    bool PushBack(uint32_t x) NN_NOEXCEPT
    {
        m_pWork[m_WorkIdx++] = x;
        if (m_WorkIdx == UnitSize)
        {
            return this->Build();
        }
        return true;
    }
    /**
     * @brief   インデックスを指定して値を取り出します。
     * @return  配列の値。idx が範囲外の場合は0
     */
    uint32_t operator[](size_t idx) const NN_NOEXCEPT;
    /**
     * @brief   配列の長さを取得します。
     */
    size_t Size() const NN_NOEXCEPT
    {
        return m_WorkIdx + m_HeaderCapacity * UnitSize / 2;
    }
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
     */
    void Reset() NN_NOEXCEPT;
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出し用オブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータはImport()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      インポートが成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    bool Build() NN_NOEXCEPT;
    // Header:
    //  baseidx: index of data_(lower 27bits)
    //  k: bit width of diff data(upper 5bits)
    //  base: base value(uint32_t)

    // Data:
    //  UnitSize * kbit: diff data
    uint32_t* m_pHeader;        // 圧縮後のデータの復元に必要な baseidx と base を保存するための配列
    uint32_t* m_pData;          // 圧縮後のデータの実体
    uint32_t m_HeaderSize;      // m_pHeader の実際に使用済みの場所の最後尾インデックス
    uint32_t m_HeaderCapacity;  // m_pHeader の確保済みサイズ(Bytes)
    uint32_t m_DataSize;        // m_pData の実際に使用済みの場所の最後尾インデックス
    uint32_t m_DataCapacity;    // m_pData の確保済みサイズ(Bytes)
    uint8_t m_WorkIdx;          // m_pWork にどれだけデータが溜まっているか
    uint32_t m_pWork[UnitSize]; // 圧縮前のデータが置かれる。ここに UnitSize ぶんデータが溜まると逐次圧縮される
    WorkBufAllocator* m_pAllocator;

    CompressedArray(const CompressedArray&) = delete;
    void operator=(const CompressedArray&) = delete;
};

/**
 * @brief   整数から整数へのコンパクトなリードオンリーの連想配列です。
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 *          キーとなる整数をSparseSetクラスを用いて格納し、値となる整数をCompressedArrayクラスを用いて格納しています。 疎な連想配列の実装に向いています。
 *          このクラスの利用方法は、以下のとおりです。
 *              - コンストラクタでオブジェクトを作成します。
 *              - Init()で定義域を設定してビットベクトルを初期化します。
 *              - TurnOn()でキーと対応する値を追加します。
 *              - Build()でRank/Select操作を定数時間で行うための辞書を構築します。
 *              - 以上で連想配列が使えるようになります。
 *          データをExport()で書きだした場合はコンストラクタの実行後にImport()を呼び出すことで利用できるようになります。
 */
class Map final
{
    typedef CompressedArray ArrayType;
    typedef SparseSet SetType;

public:
    /**
     * @brief   整数インデックスです。
     */
    typedef SetType::IdxType IdxType;
    /**
     * @brief   ブール値と整数のペアです。値が見つからなかった場合はブール値がfalseです。
     */
    typedef std::pair<bool, uint32_t> FindType;

    Map() NN_NOEXCEPT : m_Set(), m_Value(), m_pAllocator(NULL) {}
    explicit Map(WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_Set(), m_Value(), m_pAllocator(pAllocator) {}
    ~Map() NN_NOEXCEPT {}
    inline Map(Map&& rhs) NN_NOEXCEPT : m_Set(), m_Value(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline Map& operator=(Map&& rhs) NN_NOEXCEPT
    {
        Map tmp(std::move(rhs));
        this->Swap(tmp);
        rhs.ReleaseAllocator();
        return *this;
    }
    /**
     * @brief   オブジェクトの内容をスワップします。
     */
    void Swap(Map& rhs) NN_NOEXCEPT
    {
        m_Set.Swap(rhs.m_Set);
        m_Value.Swap(rhs.m_Value);
        WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }
    /**
     * @brief       オブジェクトを初期化します。
     * @param[in]   bvSize  キーの取りうる値の上限(bvSize 自体は含まず)
     * @return      成功した場合は true
     * @details     0以上bvSize 未満の値をキーとすることができるようになります。
     */
    bool Init(IdxType bvSize) NN_NOEXCEPT
    {
        if (m_Value.Size() != 0)
        {
            return false;
        }
        return m_Set.Init(bvSize);
    }
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(WorkBufAllocator* pAllocator) NN_NOEXCEPT
    {
        if (!pAllocator)
        {
            return false;
        }
        m_pAllocator = pAllocator;
        m_Set.SetAllocator(m_pAllocator);
        m_Value.SetAllocator(m_pAllocator);
        return true;
    }
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT
    {
        m_Set.ReleaseAllocator();
        m_Value.ReleaseAllocator();
        m_pAllocator = NULL;
    }
    /**
     * @brief       キーと値を追加します。
     * @param[in]   idx     キーとなる整数値
     * @param[in]   data    値となる整数値
     * @return      成功した場合は true
     * @details     この関数を繰り返し使う場合、idxは単調増加である必要があります。 つまり小さいキーから順番に設定していく必要があります。
     */
    bool TurnOn(IdxType idx, uint32_t data) NN_NOEXCEPT
    {
        if (!m_Set.TurnOn(idx))
        {
            return false;
        }
        m_Value.PushBack(data);
        return true;
    }
    /**
     * @brief   連想配列を構築します。
     * @return  成功した場合は true
     */
    bool Build() NN_NOEXCEPT
    {
        return m_Set.Build();
    }
    /**
     * @brief       キーを指定して連想配列から値を取得します。
     * @param[in]   idx キーとなる整数値
     * @return      boolとuint32_tのペア。キーに対応する値が見つかった場合はbool値がtrue
     */
    FindType Find(IdxType idx) NN_NOEXCEPT
    {
        if (m_Set.Has(idx))
        {
            return std::make_pair(true, m_Value[m_Set.Rank1(idx) - 1]);
        }
        else
        {
            return std::make_pair(false, 0);
        }
    }
    const FindType Find(IdxType idx) const NN_NOEXCEPT
    {
        if (m_Set.Has(idx))
        {
            return std::make_pair(true, m_Value[m_Set.Rank1(idx) - 1]);
        }
        else
        {
            return std::make_pair(false, 0);
        }
    }
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT
    {
        size_t value_size = m_Value.MemSize();
        size_t set_size = m_Set.MemSize();
        return value_size + set_size + sizeof(m_pAllocator);
    }
    /**
     * @brief   キーの集合を取得します。
     */
    const SetType& GetKeys() const NN_NOEXCEPT
    {
        return m_Set;
    }
    /**
     * @brief   値の集合を取得します。
     */
    const ArrayType& GetValues() const NN_NOEXCEPT
    {
        return m_Value;
    }
    /**
     * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
     * @details アロケータ情報も初期化されます。
     */
    void Reset() NN_NOEXCEPT
    {
        m_Value.Reset();
        m_Set.Reset();
        m_pAllocator = NULL;
    }
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出し用オブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータはImport()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT
    {
        if (!m_Value.Export(w))
        {
            return false;
        }
        if (!m_Set.Export(w))
        {
            return false;
        }
        return true;
    }
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      インポートが成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT
    {
        if (!m_Value.Import(r) || !m_Set.Import(r))
        {
            this->Reset();
            return false;
        }
        return true;
    }

private:
    SetType m_Set;
    ArrayType m_Value;
    WorkBufAllocator* m_pAllocator;

    Map(const Map&) = delete;
    void operator=(const Map&) = delete;
};

}}} // nn::ngc::detail

namespace std {
template<> inline void swap<::nn::ngc::detail::Set>(::nn::ngc::detail::Set& lhs, ::nn::ngc::detail::Set& rhs) NN_NOEXCEPT
{
    lhs.Swap(rhs);
}
template<> inline void swap<::nn::ngc::detail::Sbv>(::nn::ngc::detail::Sbv& lhs, ::nn::ngc::detail::Sbv& rhs) NN_NOEXCEPT
{
    lhs.Swap(rhs);
}
template<> inline void swap<::nn::ngc::detail::SparseSet>(::nn::ngc::detail::SparseSet& lhs, ::nn::ngc::detail::SparseSet& rhs) NN_NOEXCEPT
{
    lhs.Swap(rhs);
}
template<> inline void swap<::nn::ngc::detail::CompressedArray>(::nn::ngc::detail::CompressedArray& lhs, ::nn::ngc::detail::CompressedArray& rhs) NN_NOEXCEPT
{
    lhs.Swap(rhs);
}
template<> inline void swap<::nn::ngc::detail::Map>(::nn::ngc::detail::Map& lhs, ::nn::ngc::detail::Map& rhs) NN_NOEXCEPT
{
    lhs.Swap(rhs);
}
}
