﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include "./ngc_InputStream.h"
#include "./ngc_OutputStream.h"

//
// InputStream.h
//
namespace nn { namespace ngc { namespace detail {

bool InputStream::Close() NN_NOEXCEPT
{
    // クローズ動作が成功し、かつ今までにエラーが設定されていなければtrue
    // Close_()が失敗する場合でも、リソース解放等は行われていなくては
    // ならず、再度Closeが必要になることはない。
    bool result = this->Close_();
    if (result)
    {
        result = (m_Errno == 0);
    }
    m_Pos = m_IsBufCurIdx = m_IsBufEndIdx = m_IsBufSize = 0;
    m_pIsBuf = NULL;
    m_Errno = 0;  // クローズ(のみ)でエラーはリセットされる
    return result;
}

size_t InputStream::Skip(size_t nbytes) NN_NOEXCEPT
{
    // NOTE:
    // INT01-Cによるとnbytesはrsize_t型にしておくのがスジだが、
    // あまり馴染みのない型なのでチェックだけをするようにしている。
    if (nbytes > RSIZE_MAX)
    {
        this->SetError(EINVAL);
        return 0;
    }
    if (m_IsBufEndIdx >= m_IsBufCurIdx + nbytes)
    {
        m_IsBufCurIdx += nbytes;
        return nbytes;
    }
    ptrdiff_t r = m_IsBufEndIdx - m_IsBufCurIdx;
    m_Pos += m_IsBufEndIdx;
    m_IsBufCurIdx = m_IsBufEndIdx = 0;
    size_t n = this->Skip_(static_cast<size_t>(nbytes - r));
    m_Pos += n;
    return n + r;
}

bool InputStream::GetWorkBuffer() NN_NOEXCEPT
{
    size_t nBytes = 0;
    void* p = this->GetWorkBuffer_(&nBytes);
    if (p)
    {  // 空のバッファも可能なのでnbytes==0もOK
        this->ResetBuffer(p, nBytes);
        return true;
    }
    else if (nBytes == 0)
    {
        this->SetError(ENOMEM);
        return false;
    }
    return true;  // FillBuffer_(NULL, 0)で取得を試みる場合
}

void InputStream::CheckBuffer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsBufCurIdx == m_IsBufEndIdx);
    if (!*this) return;
    if (!m_pIsBuf)
    {
        if (!this->GetWorkBuffer()) return;
    }
    size_t n = this->FillBuffer_(m_pIsBuf, m_IsBufSize);
    m_Pos += m_IsBufEndIdx;
    m_IsBufCurIdx = 0;
    m_IsBufEndIdx = static_cast<int>(n);
}

size_t InputStream::Read_(void* ptr, size_t nbytes) NN_NOEXCEPT
{
    // NOTE:
    // 現状の実装では最初にファイル全体を読み込むような場合でも
    // CheckBufferでバッファをセットアップする実装になっているので、
    // 2回デバイスにアクセスすることになっている。
    // 非効率なので修正が必要

    // 最初に呼ばれる場合にCheckBufferは必要になる(m_pIsBufがNULLだから)
    if (m_IsBufCurIdx == m_IsBufEndIdx)
    {
        this->CheckBuffer();
    }
    if (!*this)
    {
        return 0;
    }
    NN_SDK_ASSERT(m_pIsBuf != NULL);
    NN_SDK_ASSERT(m_IsBufEndIdx >= m_IsBufCurIdx);

    size_t nn = static_cast<size_t>(m_IsBufEndIdx - m_IsBufCurIdx);
    if (nbytes <= nn)
    {
        MemCpy(ptr, nbytes, &m_pIsBuf[m_IsBufCurIdx], nbytes);
        m_IsBufCurIdx += nbytes;
        return nbytes;
    }
    MemCpy(ptr, nbytes, &m_pIsBuf[m_IsBufCurIdx], nn);
    ptr = reinterpret_cast<unsigned char*>(ptr) + nn;
    nbytes -= nn;
    size_t cur = m_IsBufCurIdx;
    m_IsBufCurIdx = m_IsBufEndIdx = 0;
    m_Pos += nn + cur;
    size_t total = nn;
    // m_pIsBufの方が大きい場合、I/Oの発行数を抑えることができる。
    // また、FillBuffer_()の引数のnbytesがm_IsBufSize以上であることを確実にできる。
    // libcurl用のstreamに必要
    if (nbytes < m_IsBufSize)
    {
        size_t pos = 0;
        while (pos <= nbytes)
        {
            size_t rval = this->FillBuffer_(m_pIsBuf + pos, m_IsBufSize - pos);
            if (rval == 0)
            {
                MemCpy(ptr, nbytes, &m_pIsBuf[0], pos);
                m_Pos += pos;
                total += pos;
                return total;
            }
            pos += rval;
        }
        MemCpy(ptr, nbytes, &m_pIsBuf[0], nbytes);
        m_Pos += pos;
        m_IsBufCurIdx = nbytes;
        m_IsBufEndIdx = pos;
        total += nbytes;
        return total;
    }
    else
    {
        while (nbytes > 0)
        {
            size_t rval = this->FillBuffer_(ptr, nbytes);
            if (rval == 0)
            {
                break;
            }
            nbytes -= rval;
            ptr = reinterpret_cast<uint8_t*>(ptr) + rval;
            total += rval;
            m_Pos += rval;
        }
        return total;
    }
}

void* InputStream::GetWorkBuffer_(size_t* nbytes) NN_NOEXCEPT
{
    // -1に設定するとFillBuffer_(NULL, 0);
    // でバッファを設定しようとする(互換性用)。
    *nbytes = static_cast<size_t>(-1);
    return NULL;
}

size_t InputStream::Skip_(size_t nbytes) NN_NOEXCEPT
{
    // NOTE:
    // m_Posはこの関数内では不変でなくてはならない。
    NN_SDK_ASSERT(m_IsBufEndIdx == 0);
    size_t n = 0;
    for (;;)
    {
        CheckBuffer();
        if (m_IsBufCurIdx == m_IsBufEndIdx)
        {
            return n;
        }
        size_t r = m_IsBufEndIdx;
        if (n + r >= nbytes)
        {
            m_IsBufCurIdx = 0;
            m_IsBufEndIdx = static_cast<size_t>(r - (nbytes - n));
            MemMove(&m_pIsBuf[0], m_IsBufSize, &m_pIsBuf[nbytes - n], m_IsBufEndIdx);
            return nbytes;
        }
        n += m_IsBufEndIdx;
        m_IsBufEndIdx = 0;
    }
}

NullInputStream::NullInputStream() NN_NOEXCEPT
{
    memset(m_pDunnyBuf, 0, 256);
    this->ResetBuffer(&m_pDunnyBuf[0], 256);
}

size_t NullInputStream::FillBuffer_(void* p, size_t nbytes) NN_NOEXCEPT
{
    NN_UNUSED(p);
    NN_SDK_ASSERT(p != NULL);
    return nbytes;
}

size_t NullInputStream::Skip_(size_t nbytes) NN_NOEXCEPT
{
    return nbytes;
}

//
// OutputStream.h
//

bool OutputStream::Write_(const void* p, size_t n) NN_NOEXCEPT
{
    if (!m_pOsBuf)
    {
        if (!*this || !GetWorkBuffer_())
        {
            return false;
        }
        if (m_OsBufCuridx + n <= m_OsBufSize)
        {
            MemCpy(&m_pOsBuf[m_OsBufCuridx], n, p, n);
            m_OsBufCuridx += n;
            return true;
        }
    }
    if (!this->Flush_(false))
    {
        return false;
    }
    bool result = this->PushBuffer_(p, n, false);
    if (result)
    {
        // 成功した場合のみポジションが移動する
        m_Pos += n;
        m_OsBufCuridx = 0;
    }
    return result;
}

bool OutputStream::Flush_(bool flushDevice) NN_NOEXCEPT
{
    // OutputStreamのバッファをプッシュする。
    // flush_deviceがtrueならばデバイスのバッファ(もしあれば)もフラッシュする。
    if (!*this)
    {
        return false;
    }
    if (m_pOsBuf)
    {
        if (m_OsBufCuridx == 0)
        {
            return true;
        }
        bool result = this->PushBuffer_(m_pOsBuf, m_OsBufCuridx, flushDevice);
        if (result)
        {
            // 成功した場合のみポジションが移動する
            m_Pos += m_OsBufCuridx;
            m_OsBufCuridx = 0;
        }
        return result;
    }
    else
    {
        return this->GetWorkBuffer_();
    }
}

bool OutputStream::GetWorkBuffer_() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_OsBufCuridx == 0);
    size_t n = 0;
    void* ptr = this->GetWorkBuffer_(&n);
    if (ptr && n > 0)
    {
        this->ResetBuffer(ptr, n);
        return true;
    }
    else if (n != 0)
    {
        return this->PushBuffer_(NULL, 0, false);
    }
    else
    {
        this->SetError(ENOMEM);
        return false;
    }
}

void* OutputStream::GetWorkBuffer_(size_t* nbytes) NN_NOEXCEPT
{
    // -1に設定するとPushBuffer_(NULL, 0, false);
    // でバッファを設定しようとする(互換性用)。
    *nbytes = static_cast<size_t>(-1);
    return NULL;
}

bool OutputStream::Close() NN_NOEXCEPT
{
    // フラッシュ, クローズ動作が成功し、かつ今までにエラーが設定されていなければtrue
    // Close_()が失敗する場合でも、リソース解放等は行われていなくては
    // ならず、再度Closeが必要になることはない。
    bool r = m_pOsBuf && m_OsBufCuridx > 0 ? this->Flush_(false) : true;
    // Flushに失敗した場合でもClose動作をする。
    r = this->Close_() && r;
    r = r && m_Errno == 0;
    m_Pos = 0;
    m_OsBufCuridx = 0;
    m_pOsBuf = NULL;
    m_OsBufSize = 0;
    m_Errno = 0;
    return r;
}

bool OutputStream::WriteGather_(const FdIoVec* iov, int iovcnt) NN_NOEXCEPT
{
    for (int i = 0; i < iovcnt; ++i)
    {
        if (!this->Write(iov[i].iovBase, iov[i].iovLen))
        {
            return false;
        }
    }
    return true;
}

NullOutputStream::NullOutputStream() NN_NOEXCEPT
{
    this->ResetBuffer(&m_pDummyBuf[0], 256);
}

bool NullOutputStream::PushBuffer_(const void*, size_t, bool) NN_NOEXCEPT
{
    return true;
}

}}} // nn::ngc::detail
