﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>

#include "../detail/ngc_WorkBufAllocator.h"
#include "./ngc_Bp.h"
#include "./ngc_Sbv.h"

namespace nn { namespace ngc { namespace detail {

/**
 * @brief   AC法を用いて語やパターンの検出を行います。
 * @details Externals/nlib のものをコピーし、メモリ動的確保を無理やりアロケータを渡せるように修正したもの。
 *          クラスオブジェクトは、AhoCorasickBuilder::Build()で作成するかAhoCorasick::Import()関数で読み込むかすることで構築します。
 */
class  AhoCorasick final
{
public:
    static const size_t LabelWidth = 8;  // 1,2,4,8
    static const int LabelMask = (1 << LabelWidth) - 1;
    static const int NumLabelPerByte = 8 / LabelWidth;

public:
    AhoCorasick() NN_NOEXCEPT : m_pPrv(NULL), m_pAllocator(NULL) {}
    explicit AhoCorasick(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT : m_pPrv(NULL)
    {
        NN_ABORT_UNLESS(SetAllocator(pAllocator));
    }
    ~AhoCorasick() NN_NOEXCEPT;
    inline AhoCorasick(AhoCorasick&& rhs) : m_pPrv(), m_pAllocator()
    {
        this->Swap(rhs);
    }
    inline AhoCorasick& operator=(AhoCorasick&& rhs) NN_NOEXCEPT
    {
        AhoCorasick tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    void Swap(AhoCorasick& rhs) NN_NOEXCEPT
    {
        AhoCorasickPrivate* tmp = m_pPrv;
        m_pPrv = rhs.m_pPrv;
        rhs.m_pPrv = tmp;
        nn::ngc::detail::WorkBufAllocator* tmpAllocator = m_pAllocator;
        m_pAllocator = rhs.m_pAllocator;
        rhs.m_pAllocator = tmpAllocator;
    }
    /**
     * @brief           ユーザー定義のコールバック関数で、検索対象文字列が検出された場合に呼び出されます。
     * @param[in]       pFirst      検出語の先頭を指すポインタ
     * @param[in]       pLast       検出語の直後を指すポインタ
     * @param[in]       nodeid      検出した語のID(連続値ではない)
     * @param[in/out]   pUserObj    ユーザー指定オブジェクトへのポインタ
     * @return          trueの場合は引き続き解析を続ける。falseの場合は処理を打ち切り
     * @details         文字列(パターン)を検出した場合に呼び出されます。ユーザーはそれに対応した処理を記述します。
     */
    typedef bool (*MatchCallback)(const char* pFirst, const char* pLast, uint32_t nodeid,
                                  void* pUserObj);

    bool SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    void ReleaseAllocator() NN_NOEXCEPT;

    void Match(const char* pDoc, MatchCallback callback, void* pUserObj) NN_NOEXCEPT
    {
        this->Match(pDoc, strlen(pDoc), callback, pUserObj);
    }
    void Match(const char* pDoc, MatchCallback callback) NN_NOEXCEPT
    {
        this->Match(pDoc, strlen(pDoc), callback, NULL);
    }
    /**
     * @brief           データを検査して検出対象のパターンを検出します。
     * @param[in]       pData       検査対象のデータ
     * @param[in]       n           データサイズ
     * @param[in]       callback    パターンが検出された場合に呼び出されるコールバック関数
     * @param[in,out]   pUserObj    ユーザー指定オブジェクトへのポインタ
     */
    void Match(const void* pData, size_t n, MatchCallback callback, void* pUserObj) NN_NOEXCEPT;
    void Match(const void* pData, size_t n, MatchCallback callback) NN_NOEXCEPT
    {
        Match(pData, n, callback, NULL);
    }
    /**
     * @brief   このクラスが明示的に確保するメモリ量を返します。
     */
    size_t MemSize() const NN_NOEXCEPT;

    void MemSize(size_t* pLenStore, size_t* pGotoArc, size_t* pReportArc,
                 size_t* pFailureArc) const NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを(ファイルに)書き出します。
     * @param[in]   w   書き出し用オブジェクト
     * @return      成功した場合は true
     * @details     書きだしたデータはImport()関数で読みだして復元することができます。 また、データは常にリトルエンディアンで書き出されます。
     */
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    /**
     * @brief       書き出されたオブジェクトを読み出します。
     * @param[in]   r   読み出し用オブジェクト
     * @return      インポートが成功した場合は true
     */
    bool Import(BinaryReader* r) NN_NOEXCEPT;

private:
    struct AhoCorasickPrivate;
    AhoCorasickPrivate* m_pPrv;
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;
    friend class AhoCorasickBuilder;

    AhoCorasick(const AhoCorasick&) = delete;
    void operator=(const AhoCorasick&) = delete;
};

}}} // nn::ngc::detail

namespace std {
template<> inline void swap<::nn::ngc::detail::AhoCorasick>(::nn::ngc::detail::AhoCorasick& lhs, ::nn::ngc::detail::AhoCorasick& rhs)
{
    lhs.Swap(rhs);
}
}
