﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ngc_WorkBufAllocator.h"

namespace nn { namespace ngc { namespace detail {

    bool WorkBufAllocator::Initialize(void* ptr, size_t size) NN_NOEXCEPT
    {
#if !defined(NGC_PCTOOL)
        m_Handle = nn::lmem::CreateExpHeap(ptr, size, nn::lmem::CreationOption_NoOption);
        if (!m_Handle)
        {
            return false;
        }
#endif
        return true;
    }

    void WorkBufAllocator::Finalize() NN_NOEXCEPT
    {
#if !defined(NGC_PCTOOL)
        if (m_Handle)
        {
            nn::lmem::DestroyExpHeap(m_Handle);
            m_Handle = NULL;
        }
#endif
    }

    void* WorkBufAllocator::Allocate(size_t size) NN_NOEXCEPT
    {
#if !defined(NGC_PCTOOL)
        NN_SDK_ASSERT(m_Handle);
        return nn::lmem::AllocateFromExpHeap(m_Handle, size);
#else
        return std::malloc(size);
#endif
    }

    void* WorkBufAllocator::Allocate(size_t size, int alignment) NN_NOEXCEPT
    {
#if !defined(NGC_PCTOOL)
        NN_SDK_ASSERT(m_Handle);
        return nn::lmem::AllocateFromExpHeap(m_Handle, size, alignment);
#else
        return NULL;    // unsupported
#endif
    }

    void WorkBufAllocator::Free(void* ptr) const NN_NOEXCEPT
    {
#if !defined(NGC_PCTOOL)
        NN_SDK_ASSERT(m_Handle);
        if (!ptr)
        {
            return;
        }
        nn::lmem::FreeToExpHeap(m_Handle, ptr);
#else
        std::free(ptr);
#endif
    }

    void* WorkBufAllocator::Reallocate(void* ptr, size_t newSize) NN_NOEXCEPT
    {
#if !defined(NGC_PCTOOL)
        NN_SDK_ASSERT(m_Handle);
        if (newSize == 0)
        {
            nn::lmem::FreeToExpHeap(m_Handle, ptr);
            return NULL;
        }
        void* newPtr = nn::lmem::AllocateFromExpHeap(m_Handle, newSize);
        if (!newPtr)
        {
            return NULL;
        }
        if (!ptr)
        {
            return newPtr;
        }
        size_t oldSize = nn::lmem::GetExpHeapBlockSize(ptr);
        memcpy(newPtr, ptr, (oldSize < newSize) ? oldSize : newSize);
        nn::lmem::FreeToExpHeap(m_Handle, ptr);

        return newPtr;
#else
        return std::realloc(ptr, newSize);
#endif
    }

}}} // nn::ngc::detail

