﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/prepo/prepo_Api.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/server/util/nfc_ScopedLockShared.h>
#include <nn/nfp/server/nfp_Manager.h>
#include <nn/nfc/server/nfc_Manager.h>
#include "nfp_BackupDataStream.h"
#include <nn/nfc/server/util/nfc_UtilUtil.h>

namespace nn { namespace nfp {namespace server {

namespace
{
}

Manager::Manager(nn::nfc::server::Manager* manager) NN_NOEXCEPT : m_Manager(manager)
{
    nn::prepo::Initialize();
}

Manager::~Manager() NN_NOEXCEPT
{
}

nn::Result Manager::Mount(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, ModelType modelType, MountTarget mountTarget) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, Mount(service, modelType, mountTarget));
}

nn::Result Manager::Unmount(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, Unmount());
}

nn::Result Manager::OpenApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, nn::Bit32 accessId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, OpenApplicationArea(accessId));
}

nn::Result Manager::GetApplicationArea(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, size_t bufferSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetApplicationArea(pOutBuffer, pOutSize, bufferSize));
}

nn::Result Manager::SetApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, SetApplicationArea(pData, dataSize));
}

nn::Result Manager::RecreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, RecreateApplicationArea(createInfo));
}

nn::Result Manager::Flush(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, Flush(service));
}

nn::Result Manager::Restore(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, Restore(service));
}

nn::Result Manager::CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, CreateApplicationArea(service, createInfo));
}

nn::Result Manager::GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetRegisterInfo(pOutRegisterInfo));
}

nn::Result Manager::GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetCommonInfo(pOutCommonInfo));
}

nn::Result Manager::GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetModelInfo(pOutModelInfo));
}

nn::Result Manager::Format(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, Format(service));
}

nn::Result Manager::GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetAdminInfo(pOutAdminInfo));
}

nn::Result Manager::GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetRegisterInfo(pOutRegisterInfo));
}

nn::Result Manager::SetRegisterInfo(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, SetRegisterInfo(regInfo));
}

nn::Result Manager::DeleteRegisterInfo(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, DeleteRegisterInfo(service));
}

nn::Result Manager::DeleteApplicationArea(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, DeleteApplicationArea(service));
}

nn::Result Manager::ExistsApplicationArea(bool* outValue, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, ExistsApplicationArea(outValue));
}

nn::Result Manager::GetAll(nn::nfp::NfpData* pOutNfpData, nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, GetAll(pOutNfpData));
}

nn::Result Manager::SetAll(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const nn::nfp::NfpData& nfpData) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, SetAll(nfpData));
}

nn::Result Manager::FlushDebug(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, FlushDebug(service));
}

nn::Result Manager::BreakTag(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, BreakType breakType) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, BreakTag(service, breakType));
}


nn::Result Manager::ReadBackupData(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, size_t bufferSize) NN_NOEXCEPT
{
    NN_UNUSED(service);

    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutBuffer);
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutSize);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < bufferSize);

    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_RESULT_DO(BackupDataStream::GetInstance()->Initialize());
    NN_UTIL_SCOPE_EXIT
    {
        BackupDataStream::GetInstance()->Finalize();
    };

    return BackupDataStream::GetInstance()->Read(pOutBuffer, pOutSize, bufferSize);
}

nn::Result Manager::WriteBackupData(nn::nfc::server::core::Service* service, const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_UNUSED(service);

    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pData);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < dataSize);
    NN_NFC_SERVER_UTIL_REQUIRES(dataSize <= BackupDataStream::BackupTotalSize);

    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_RESULT_DO(BackupDataStream::GetInstance()->Initialize());
    NN_UTIL_SCOPE_EXIT
    {
        BackupDataStream::GetInstance()->Finalize();
    };

    return BackupDataStream::GetInstance()->Write(pData, dataSize);
}

nn::Result Manager::WriteNtf(nn::nfc::server::core::Service* service, const nn::nfc::DeviceHandle& deviceHandle, const void* pData, size_t dataSize, NtfWriteType ntfWriteType) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_Manager->m_ReaderWriterLock);

    NN_NFC_SERVER_MANAGER_ACCESS_DEVICE_RETURN(m_Manager, service, deviceHandle, WriteNtf(service, pData, dataSize, ntfWriteType));
}


size_t Manager::GetApplicationAreaSize() NN_NOEXCEPT
{
    return Device::GetApplicationAreaSize();
}

}}}  // namespace nn::nfp::server
