﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nfp/nfp_Types.h>
#include <nn/nfp/nfp_Result.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include "nfp_BackupDataHeader.h"
#include "nfp_BackupTocItem.h"

namespace nn { namespace nfp { namespace server {

/**
 * @brief         バックアップデータの操作クラスです。
 */
class BackupDataStream
{
public:
    //! バックアップデータエントリの総数
    static const size_t BackupEntryMax = 1000;

    //! バックアップデータのエントリサイズ (bytes) です。
    static const size_t BackupEntrySize = 2048;

    //! バックアップデータエントリの索引総サイズ
    static const size_t BackupEntryTocTotalSize = sizeof(BackupTocItem) * BackupEntryMax;

    //! バックアップデータエントリ総サイズです。
    static const size_t BackupEntryTotalSize = BackupEntrySize * BackupEntryMax;

    //! バックアップデータの総サイズです。
    static const size_t BackupTotalSize = sizeof(BackupDataHeader) + BackupEntryTocTotalSize + BackupEntryTotalSize;

    //! バックアップデータエントリの索引開始位置です。
    static const int64_t BackupEntryTocPos = sizeof(BackupDataHeader);

    //! バックアップデータエントリ開始位置です。
    static const int64_t BackupEntryPos = BackupEntryTocPos + BackupEntryTocTotalSize;

    class ScopedBackupFileStreamOpen
    {
    public:
        explicit ScopedBackupFileStreamOpen(nn::nfp::server::BackupDataStream& backupDataStream) NN_NOEXCEPT;
        ~ScopedBackupFileStreamOpen() NN_NOEXCEPT;
        void release() NN_NOEXCEPT;
    private:
        nn::nfp::server::BackupDataStream& m_BackupDataStream;
        bool m_IsReleased;
    };

    /*!
     * @brief          BackupDataStream の唯一のインスタンスを取得します。
     * @return         BackupDataStream の唯一のインスタンスです。
     */
    static BackupDataStream* GetInstance() NN_NOEXCEPT
    {
        static BackupDataStream instance;
        return &instance;
    }

    /**
     * @brief         バックアップデータをオープンします。存在しない場合は作成します。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 既に初期化されています。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Initialize() NN_NOEXCEPT;

    /**
     * @brief         バックアップデータをクローズします。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 初期化されていません。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Finalize() NN_NOEXCEPT;

    /**
     * @brief         システムセーブデータを再生成します。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 初期化されていません。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Format() NN_NOEXCEPT;

    /**
     * @brief         指定されたタグデータのバックアップを保存します。
     * @param[in]     pData                    書き込むタグデータです。
     * @param[in]     dataSize                 pData のサイズです。
     * @param[in]     tagId                    タグ情報です。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 初期化されていません。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Write(const void* pData, size_t dataSize, const nn::nfc::TagId& tagId) NN_NOEXCEPT;

    /**
     * @brief         システムセーブデータを取得します。
     * @param[out]    pOutBuffer               出力先です。
     * @param[out]    pOutSize                 出力サイズです。
     * @param[in]     bufferSize               バッファサイズです。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 初期化されていません。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Read(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT;

    /**
     * @brief         システムセーブデータを書き換えます。
     * @param[in]     pData                    出力先です。
     * @param[in]     dataSize                 バッファサイズです。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 初期化されていません。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Write(const void* pData, size_t dataSize) NN_NOEXCEPT;

    /**
     * @brief         指定されたタグデータのバックアップデータを取得します。
     * @param[out]    pOutBuffer               バックアップデータの出力先です。
     * @param[in]     readAndBufferSize        pOutBuffer のバッファサイズかつ読むべきサイズです。
     * @param[in]     tagId                    タグ情報です。
     * @retval        ResultSuccess            成功しました。
     * @retval        ResultInvalidDeviceState 初期化されていません。
     * @retval        ResultTagNotFound        バックアップデータが存在しません。
     * @retval        上記以外                 想定外のエラーです。
     */
    nn::Result Read(void* pOutBuffer, size_t readAndBufferSize, const nn::nfc::TagId& tagId) NN_NOEXCEPT;

    /**
     * @brief         指定されたタグデータのバックアップが存在することを確認します。
     * @param[in]     tagId                  タグ情報です。
     * @return        存在する場合に true です。
     */
    bool HasBackupData(const nn::nfc::TagId& tagId) NN_NOEXCEPT;

private:

    /**
     * @brief         コンストラクタです。
     */
    BackupDataStream() NN_NOEXCEPT;

    /**
     * @brief         デストラクタです。バックアップデータをアンマウントします。
     */
    ~BackupDataStream() NN_NOEXCEPT;

    nn::Result CreateAndOpenBackupData() NN_NOEXCEPT;
    const char* GetBackupFileName() const NN_NOEXCEPT;
    nn::Result FindEntry(int* pOutIndex, BackupTocItem* pOutToc, const nn::nfc::TagId& tagId) NN_NOEXCEPT;

    int m_FileHandle;
    BackupDataHeader   m_Header;
    bool               m_IsInitialized;
    mutable nn::os::MutexType m_Mutex;
};

}}}  // namespace nn::nfp::server
