﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nfp/nfp_DebugApi.h>
#include <nn/nfp/nfp_ErrorApi.h>
#include <nn/applet/applet.h>
#include "client/nfp_CommonInterface.h"
#include "client/nfp_McuVersionData.h"
#include "detail/nfp_Util.h"

namespace nn { namespace nfp {

namespace
{
}

void InitializeDebug() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!client::IsCreatedSystemInterface());
    NN_ABORT_UNLESS(!client::IsCreatedUserInterface());
    NN_ABORT_UNLESS(!client::IsCreatedDebugInterface());

    client::CreateDebugInterface();

    nn::sf::InArray<nn::xcd::McuVersionDataForNfc> requiredVersions(
        client::RequiredMcuVersionData,
        client::RequiredMcuVersionDataCount);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::applet::AppletResourceUserId id = nn::applet::GetAppletResourceUserId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(client::GetDebugInterface()->InitializeNfp(id, 0, requiredVersions));
#else
    nn::applet::AppletResourceUserId id = nn::applet::AppletResourceUserId::GetInvalidId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(client::GetDebugInterface()->InitializeNfp(id, 1, requiredVersions));
#endif

    SetErrorVisible(true);
}

void FinalizeDebug() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(client::GetDebugInterface()->FinalizeNfp());
    client::DestroyDebugInterface();
}

nn::Result GetAll(NfpData* pOutNfpData, const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNfpData);

    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(GetAll, (pOutNfpData, reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result SetAll(const DeviceHandle& deviceHandle, const NfpData& nfpData) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(SetAll, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), nfpData)));
}

nn::Result FlushDebug(const DeviceHandle& deviceHandle) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(FlushDebug, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle))));
}

nn::Result BreakTag(const DeviceHandle& deviceHandle, BreakType breakType) NN_NOEXCEPT
{
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(BreakTag, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), static_cast<nn::Bit32>(breakType))));
}

nn::Result ReadBackupData(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);

    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), bufferSize);
    uint32_t outSize = 0;
    nn::Result result = nn::nfp::detail::CheckError(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(ReadBackupData, (buffer, &outSize)));
    if(result.IsFailure())
    {
        return nn::nfp::ResultBackupError();
    }
    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

nn::Result WriteBackupData(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES(0 < dataSize);

    nn::sf::InBuffer data(reinterpret_cast<char*>(const_cast<void*>(pData)), dataSize);
    nn::Result result = nn::nfp::detail::CheckError(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(WriteBackupData,(data)));
    if(result.IsFailure())
    {
        return nn::nfp::ResultBackupError();
    }
    NN_RESULT_SUCCESS;
}

nn::Result WriteNtf(const DeviceHandle& deviceHandle, const void* pData, size_t dataSize, NtfWriteType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES(0 < dataSize);

    nn::sf::InBuffer data(reinterpret_cast<char*>(const_cast<void*>(pData)), dataSize);
    NN_NFP_DETAIL_RETURN(NN_NFP_CLIENT_GET_DEBUG_INTERFACE(WriteNtf, (reinterpret_cast<const nn::nfc::DeviceHandle&>(deviceHandle), data, static_cast<nn::Bit32>(type))));
}
}}
