﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/server/nfc_Tag.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfc/server/core/nfc_CoreUtil.h>
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace
{
}

namespace nn { namespace nfc { namespace server {

Tag::Tag(const nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& id, nn::nfc::NfcProtocol protocol, nn::nfc::TagType type, nn::os::SystemEventType* accessFinishEvent, nn::os::SystemEventType* accessResetEvent) NN_NOEXCEPT : m_Handle(deviceHandle), m_Id(id), m_Protocol(protocol), m_Type(type), m_pAccessFinishEvent(accessFinishEvent), m_pAccessResetEvent(accessResetEvent)
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);
}

Tag::~Tag() NN_NOEXCEPT
{
    nn::os::FinalizeMutex(&m_Mutex);
}

bool Tag::operator ==(const Tag& rhs) const NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return (m_Id.length == rhs.m_Id.length && std::memcmp(m_Id.uid, rhs.m_Id.uid, m_Id.length) == 0);
}

bool Tag::operator !=(const Tag& rhs) const NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return (m_Id.length != rhs.m_Id.length || std::memcmp(m_Id.uid, rhs.m_Id.uid, m_Id.length) != 0);
}

nn::nfc::TagId Tag::GetId() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_Id;
}

nn::nfc::NfcProtocol Tag::GetProtocol() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_Protocol;
}

nn::nfc::TagType Tag::GetType() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);
    return m_Type;
}

nn::Result Tag::SendCommand(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutBuffer);
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutSize);
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pData);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < dataSize);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < bufferSize);

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    std::unique_ptr<nn::nfc::server::core::PassThruParameter> parameter(new nn::nfc::server::core::PassThruParameter());
    parameter->timeoutMsec = static_cast<uint16_t>(timeout.GetMilliSeconds());
    parameter->tagId = m_Id;
    parameter->sendDataSize = static_cast<uint32_t>(dataSize);
    std::memcpy(parameter->sendData, pData, parameter->sendDataSize);

    ClearEvent();
    NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().SendRawData(service, m_Handle, m_Id, *parameter.get()));

    std::unique_ptr<nn::nfc::server::core::Info> info(new nn::nfc::server::core::Info());
    NN_RESULT_DO(WaitInfo(info.get(), service, m_Handle, parameter->timeoutMsec + 1000));
    switch(info->reason)
    {
    case nn::xcd::NfcEventReason_PassThruResult:
        {
            if(static_cast<size_t>(info->passThruData.responseSize) < bufferSize)
            {
                *pOutSize = static_cast<size_t>(info->passThruData.responseSize);
            }
            else
            {
                *pOutSize = bufferSize;
            }
            std::memcpy(pOutBuffer, info->passThruData.responseData, *pOutSize);
            NN_RESULT_SUCCESS;
        }
    case nn::xcd::NfcEventReason_Error:
        {
            return nn::nfc::server::core::ConvertToNfcResult(info->errorInfo.resultCode);
        }
    default:
        {
            return nn::nfc::ResultAccessError();
        }
    }
}

bool Tag::IsNtag() NN_NOEXCEPT
{
    return (m_Protocol == nn::nfc::NfcProtocol_TypeA && m_Type == nn::nfc::TagType_Type2);
}

nn::Result Tag::WaitInfo(nn::nfc::server::core::Info* pOutInfo, nn::nfc::server::core::Service* service, nn::nfc::DeviceHandle handle, int64_t timeoutMsec) NN_NOEXCEPT
{
    switch(nn::os::TimedWaitAny(nn::TimeSpan::FromMilliSeconds(timeoutMsec), m_pAccessFinishEvent, m_pAccessResetEvent))
    {
    case 0: // m_pAccessFinishEvent
        {
            nn::os::ClearSystemEvent(m_pAccessFinishEvent);
            return nn::nfc::server::core::Manager::GetInstance().GetInfo(pOutInfo, service, handle);
        }
        break;
    case 1: // m_pAccessResetEvent
        {
            nn::os::ClearSystemEvent(m_pAccessResetEvent);
            nn::nfc::server::core::ResetInfo resetInfo;
            NN_RESULT_DO(nn::nfc::server::core::Manager::GetInstance().GetResetInfo(&resetInfo, service, handle));
            return nn::nfc::server::core::ConvertToNfcResult(resetInfo.reason);
        }
        break;
    default: // timeout
        return nn::nfc::ResultTimeOutError();
    }
}

void Tag::ClearEvent() NN_NOEXCEPT
{
    nn::os::ClearSystemEvent(m_pAccessFinishEvent);
    nn::os::ClearSystemEvent(m_pAccessResetEvent);
}

}}}  // namespace nn::nfc::server
