﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/psc.h>
#endif
#include <nn/util/util_ScopeExit.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include <nn/nfc/server/core/nfc_SaveData.h>
#include <nn/nfc/server/util/nfc_ScopedLockShared.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/hid/hid_NpadCommon.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/settings/system/settings_Nfc.h>
#include <nn/nfc/server/util/nfc_ThreadStack.h>
#include "nfc_CoreDevice.h"
#include <nn/nfc/server/util/nfc_UtilUtil.h>

#define NN_NFC_SERVER_MANAGER_CHECK_RESULT_DO(device, x, service, xcdDeviceHandle) \
    do{                                                                \
        nn::Result result = device->x;                                 \
        if(result.IsFailure())                                         \
        {                                                              \
            NN_RESULT_DO(CheckAwakePrivate());                         \
            NN_RESULT_DO(CheckNfcEnabledPrivate(service));             \
            NN_RESULT_DO(CheckServicePrivate(service));                \
            nn::xcd::DeviceHandle handle;                              \
            NN_RESULT_DO(device->GetXcdDeviceHandle(&handle));         \
            if(handle != xcdDeviceHandle)                              \
            {                                                          \
                return nn::nfc::ResultNfcDeviceNotFound();     \
            }                                                          \
            return result;                                             \
        }                                                              \
    } while(NN_STATIC_CONDITION(false))

#define NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, x, service, xcdDeviceHandle) \
    NN_NFC_SERVER_MANAGER_CHECK_RESULT_DO(device, x, service, xcdDeviceHandle); \
    NN_RESULT_SUCCESS

namespace nn { namespace nfc { namespace server { namespace core {

namespace
{
const int LinkMonitoringThreadRetryInterval = 50;
}

//

Manager::Manager() NN_NOEXCEPT : m_DeviceUniqueId(0), m_InitializedCount(0), m_IsRunningThread(false), m_pLinkEventForNfcUser(nullptr), m_IsInitializedTerminalId(false), m_IsAwake(true)
{
    // 乱数の精度はそれほど重要でないので標準ライブラリの rand を使用します。
    int64_t tick = nn::os::GetSystemTick().GetInt64Value();
    std::srand(static_cast<unsigned>((tick >> 32) ^ tick));

    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);

    m_ForegroundApplet = nn::applet::AppletResourceUserId::GetInvalidId();

    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(m_DeviceList[i])
        {
            m_DeviceList[i].reset(nullptr);
        }
    }

    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        m_Service[i] = nullptr;
    }

    nn::hid::InitializeNpad();

    m_LinkMonitoringThreadStack.reset(new nn::nfc::server::util::ThreadStack(4 * 1024));

    nn::os::InitializeMultiWait(&m_LinkMonitoringMultiWait);
    nn::hid::system::BindNfcDeviceUpdateEvent(&m_LinkEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_LinkHolder, &m_LinkEvent);
    nn::os::LinkMultiWaitHolder(&m_LinkMonitoringMultiWait, &m_LinkHolder);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(&m_TerminateLinkMonitoringEvent, nn::os::EventClearMode_ManualClear, false));
    nn::os::InitializeMultiWaitHolder(&m_TerminateLinkMonitoringHolder, &m_TerminateLinkMonitoringEvent);
    nn::os::LinkMultiWaitHolder(&m_LinkMonitoringMultiWait, &m_TerminateLinkMonitoringHolder);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    m_PmModule.reset(new nn::psc::PmModule);
    const nn::psc::PmModuleId dependencies[] = {
        nn::psc::PmModuleId_Hid,
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule->Initialize(nn::psc::PmModuleId_Nfc, dependencies, sizeof(dependencies) / sizeof(dependencies[0]), nn::os::EventClearMode_ManualClear));
    nn::os::InitializeMultiWaitHolder(&m_PmModuleHolder, m_PmModule->GetEventPointer()->GetBase());
    nn::os::LinkMultiWaitHolder(&m_LinkMonitoringMultiWait, &m_PmModuleHolder);
#endif

    UpdateDevices();

    nn::os::CreateThread(&m_LinkMonitoringThread, LinkMonitoringThread, this, m_LinkMonitoringThreadStack->Get(), m_LinkMonitoringThreadStack->GetSize(), NN_SYSTEM_THREAD_PRIORITY(nfc, CoreServerManager));
    m_IsRunningThread = true;
    nn::os::SetThreadNamePointer(&m_LinkMonitoringThread, NN_SYSTEM_THREAD_NAME(nfc, CoreServerManager));
    nn::os::StartThread(&m_LinkMonitoringThread);
}

Manager::~Manager() NN_NOEXCEPT
{
    m_IsRunningThread = false;
    nn::os::SignalSystemEvent(&m_TerminateLinkMonitoringEvent);

    nn::os::WaitThread(&m_LinkMonitoringThread);

    nn::os::DestroyThread(&m_LinkMonitoringThread);

    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(m_DeviceList[i])
        {
            m_DeviceList[i].reset(nullptr);
        }
    }

    nn::os::UnlinkAllMultiWaitHolder(&m_LinkMonitoringMultiWait);
    nn::os::FinalizeMultiWait(&m_LinkMonitoringMultiWait);

    nn::os::FinalizeMultiWaitHolder(&m_LinkHolder);
    nn::os::DestroySystemEvent(&m_LinkEvent);

    nn::os::FinalizeMultiWaitHolder(&m_TerminateLinkMonitoringHolder);
    nn::os::DestroySystemEvent(&m_TerminateLinkMonitoringEvent);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::os::FinalizeMultiWaitHolder(&m_PmModuleHolder);
    m_PmModule->Finalize();
    m_PmModule.reset(nullptr);
#endif
}

/*
  この関数では指定されたサービス(ライブラリ)で NFC デバイスが利用可能かどうか判断します。

  最前面で複数の Initalize が行われている場合が複雑なのでコメントしておきます。

  複数の Initalize が行われる組み合わせは、
  (旧)NFC x NFP x MIFARE
  (旧)NFC x MIFARE
  NFP x MIFARE
  (旧)NFC x NFP
  (新)NFC x NFP
  の 5 通りです。

  このうち
  (旧)NFC x NFP x MIFARE
  (旧)NFC x MIFARE
  (旧)NFC x NFP
  の(旧)NFC(LibraryType_Nfc) は NFC デバイスを利用せず、他のライブラリとの関係を気にすることはない(m_Service[] に格納されない)ので、この関数内では実質
  NFP x MIFARE
  MIFARE 単独
  NFP 単独
  と同じ処理になります。

  よって、この関数において、複数の Initalize については、実質 2 通りの組み合わせについて考えればよいです。
  そのうちの一つの
  NFP x MIFARE
  の組み合わせでは、
  従来仕様との互換性を維持するため従来通り NFP ライブラリか MIFARE ライブラリのうちどちらか先に Initialize している方で NFC デバイスを利用できるようにしています。

  残りの
  (新)NFC x NFP
  の組み合わせでは、NFC ライブラリ、NFP ライブラリいずれも NFC デバイスを利用できます。
  従来仕様(NFC ライブラリ利用中でも NFP ライブラリで NFC デバイスを利用できるという仕様)との互換性を維持しつつ、
  NFP ライブラリ利用中でも NFC ライブラリで NFC デバイスを利用できるというシンプルな仕様に対応しています。
*/
nn::Result Manager::CheckServicePrivate(Service* service) NN_NOEXCEPT
{
    if(service == Service::NfcProcessService)
    {
        NN_RESULT_SUCCESS;
    }

    //指定されたサービスが最前面で Initialize されているかどうか
    bool isForeground = false;

    //最前面で nn::nfc::mifare::Initialize されているかどうか
    bool hasNfcMifareInForeground = false;

    //最前面で Initialize されているかライブラリの数
    int foregroundServiceCount = 0;

    //最前面で Initialize されているサービスのうち最初に Initialize されているサービス
    Service* firstServiceExceptNfcInForeground = nullptr;

    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        if(!m_Service[i])
        {
            continue;
        }

        if(m_Service[i]->GetAppletResourceUserId() == m_ForegroundApplet)
        {
            if(m_Service[i] == service)
            {
                isForeground = true;
            }
            foregroundServiceCount++;

            auto type = m_Service[i]->GetLibraryType();

            if(type == nn::nfc::server::core::LibraryType_NfcMifare)
            {
                hasNfcMifareInForeground = true;
            }

            if(firstServiceExceptNfcInForeground == nullptr)
            {
                firstServiceExceptNfcInForeground = m_Service[i];
            }
        }
    }

    if(!isForeground)
    {
        /*
          指定されたサービスが最前面でない
        */
        return nn::nfc::ResultNotForeground();
    }

    if(foregroundServiceCount == 1)
    {
        /*
          指定されたサービスが最前面で唯一のサービスなので無条件に利用可能
        */
        NN_RESULT_SUCCESS;
    }

    if(hasNfcMifareInForeground)
    {
        /*
          NFP x MIFARE
          の組み合わせ
          最初に Initialize しているサービスが利用可能
        */
        if(firstServiceExceptNfcInForeground == service)
        {
            NN_RESULT_SUCCESS;
        }
        else
        {
            return nn::nfc::ResultNotForeground();
        }
    }

    /*
      (新)NFC x NFP の組み合わせ
      どちらであっても利用可能
    */
    NN_RESULT_SUCCESS;
}

nn::Result Manager::CheckService(Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);
    return CheckServicePrivate(service);
}

nn::Result Manager::CheckNfcEnabledPrivate(Service* service) NN_NOEXCEPT
{
    NN_UNUSED(service);
    if(IsNfcEnabled())
    {
        NN_RESULT_SUCCESS;
    }

    return nn::nfc::ResultNfcDisabled();
}

nn::Result Manager::CheckNfcEnabled(Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);
    return CheckNfcEnabledPrivate(service);
}

nn::Result Manager::CheckAwakePrivate() NN_NOEXCEPT
{
    if(m_IsAwake)
    {
        NN_RESULT_SUCCESS;
    }

    return nn::nfc::ResultSleep();
}

nn::Result Manager::CheckAwake() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);
    return CheckAwakePrivate();
}

nn::Result Manager::NotifyForegroundApplet(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);

    if(m_ForegroundApplet != aruid)
    {
        //最前面に変更があれば、必ず再初期化
        ResetDevices(ResetReason_NotForeground, nullptr);

        m_ForegroundApplet = aruid;
        SignalLinkEvent();
    }

    NN_RESULT_SUCCESS;
}

bool Manager::IsInitialized() NN_NOEXCEPT
{
    return m_InitializedCount > 0;
}

void Manager::Initialize(Service* service) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);

    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        if(!m_Service[i])
        {
            m_Service[i] = service;
            break;
        }
    }

    if(!IsInitialized())
    {
        SaveData::GetInstance().Initialize();
        InitializeTerminalId();

        for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
        {
            if(m_DeviceList[i])
            {
                m_DeviceList[i]->Initialize();
            }
        }
    }

    ++m_InitializedCount;
}

void Manager::Finalize(Service* service) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);

    NN_ABORT_UNLESS(IsInitialized());
    --m_InitializedCount;

    if(!IsInitialized())
    {
        for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
        {
            if(m_DeviceList[i])
            {
                //Initialize に失敗している(コントローラ接続エラーなど)場合もあるのでチェックする
                if(m_DeviceList[i]->IsInitialized())
                {
                    m_DeviceList[i]->Finalize();
                }
            }
        }

        SaveData::GetInstance().Finalize();
    }
    else if(CheckServicePrivate(service).IsSuccess())
    {
        ResetDevices(ResetReason_NotForeground, service);
    }

    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        if(m_Service[i] == service)
        {
            for(auto j = i; j < ServiceCountMax - 1; ++j)
            {
                m_Service[j] = m_Service[j + 1];
            }
            m_Service[ServiceCountMax - 1] = nullptr;
            break;
        }
    }
}

nn::Result Manager::ListDevices(uint64_t *pOutDeviceUniqueId, nn::nfc::DeviceHandle* pOutBuffer, int* pOutCount, Service* service, int bufferCount) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_UNUSED(service);

    int count = 0;

    for(auto i = 0; i < nn::nfc::DeviceCountMax && count < bufferCount; ++i)
    {
        if(!m_DeviceList[i])
        {
            continue;
        }

        pOutDeviceUniqueId[count] = m_DeviceList[i]->GetUniqueId();
        pOutBuffer[count] = m_DeviceList[i]->GetHandle();
        ++count;
    }

    if(count == 0)
    {
        return nn::nfc::ResultNfcDeviceNotFound();
    }

    *pOutCount = count;

    NN_RESULT_SUCCESS;
}

nn::Result Manager::StartDiscovery(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    return StartDiscovery(service, deviceHandle, nn::nfc::NfcProtocol_All);
}

nn::Result Manager::StartDiscovery(Service* service, nn::nfc::DeviceHandle deviceHandle, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    NN_SDK_LOG("[NFP] %s\n", NN_CURRENT_FUNCTION_NAME);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    nn::nfc::server::core::DiscoveryParameter discoverParamater;
    discoverParamater.protocolFilter = protocolFilter;
    discoverParamater.activationTimeout = 0;
    discoverParamater.discoveryPeriod = 300;

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, StartDiscovery(service, discoverParamater, service->GetMcuVersionData()), service, xcdDeviceHandle);
}

nn::Result Manager::StopDiscovery(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, StopDiscovery(), service, xcdDeviceHandle);
}

nn::Result Manager::KeepSession(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, KeepSession(), service, xcdDeviceHandle);
}

nn::Result Manager::ReleaseSession(Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, ReleaseSession(), service, xcdDeviceHandle);
}

void Manager::SetEvent(nn::os::SystemEventType* pLinkEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);
    m_pLinkEventForNfcUser = pLinkEvent;
}

void Manager::SignalLinkEvent() NN_NOEXCEPT
{
    if(m_pLinkEventForNfcUser != nullptr)
    {
        nn::os::SignalSystemEvent(m_pLinkEventForNfcUser);
    }
}

nn::Result Manager::SetEvent(nn::os::SystemEventType* pEvent, nn::os::SystemEventType* pDetectEvent, nn::os::SystemEventType* pResetEvent, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, Service::NfcProcessService, deviceHandle, false));

    device->SetEvent(pEvent, pDetectEvent, pResetEvent);
    NN_RESULT_SUCCESS;
}

nn::Result Manager::GetInfo(Info* pInfo, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle, false));

    Info& info = device->GetInfo();
    std::memcpy(pInfo, &info, sizeof(Info));
    NN_RESULT_SUCCESS;
}

nn::Result Manager::GetDetectInfo(Info* pInfo, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle, false));

    Info& info = device->GetDetectInfo();
    std::memcpy(pInfo, &info, sizeof(Info));
    NN_RESULT_SUCCESS;
}

nn::Result Manager::GetResetInfo(ResetInfo* pResetInfo, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle, false));

    ResetInfo& resetInfo = device->GetResetInfo();
    std::memcpy(pResetInfo, &resetInfo, sizeof(ResetInfo));
    NN_RESULT_SUCCESS;
}

nn::Result Manager::StartNtagRead(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const NtagReadParameter& ntagReadParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, StartNtagRead(tagId, ntagReadParameter), service, xcdDeviceHandle);
}

nn::Result Manager::StartNtagWrite(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const NtagWriteParameter& ntagWriteParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, StartNtagWrite(tagId, ntagWriteParameter), service, xcdDeviceHandle);
}

nn::Result Manager::SendRawData(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const PassThruParameter& passThruParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, SendRawData(tagId, passThruParameter), service, xcdDeviceHandle);
}

nn::Result Manager::GetTagInfo(nn::nfc::TagInfo* tagInfo, Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, GetTagInfo(tagInfo, tagId), service, xcdDeviceHandle);
}

nn::Result Manager::GetNpadId(nn::hid::NpadIdType* pOutNpadId, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, GetNpadId(pOutNpadId), service, xcdDeviceHandle);
}

nn::Result Manager::GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutXcdDeviceHandle, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, GetXcdDeviceHandle(pOutXcdDeviceHandle), service, xcdDeviceHandle);
}

bool Manager::IsNfcEnabled() NN_NOEXCEPT
{
    return nn::settings::system::IsNfcEnabled();
}

nn::Result Manager::IsNfcEnabled(bool* pOutIsNfcEnabled, Service* service) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);
    NN_UNUSED(service);
    *pOutIsNfcEnabled = IsNfcEnabled();
    NN_RESULT_SUCCESS;
}

nn::Result Manager::SetNfcEnabled(Service* service, bool isNfcEnabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::ReaderWriterLock> lock(m_ReaderWriterLock);
    NN_UNUSED(service);
    bool old = IsNfcEnabled();
    if(old == isNfcEnabled)
    {
        NN_RESULT_SUCCESS;
    }

    nn::settings::system::SetNfcEnabled(isNfcEnabled);

    if(!isNfcEnabled)
    {
        //変更があれば、必ず再初期化
        ResetDevices(ResetReason_NfcDisabled, nullptr);
    }

    SignalLinkEvent();

    NN_RESULT_SUCCESS;
}

nn::Result Manager::GetDevice(Device** pOutDevice, Service* service, nn::nfc::DeviceHandle deviceHandle, bool check) NN_NOEXCEPT
{
    if(check)
    {
        NN_RESULT_DO(CheckAwakePrivate());
        NN_RESULT_DO(CheckNfcEnabledPrivate(service));
        NN_RESULT_DO(CheckServicePrivate(service));
    }

    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(!m_DeviceList[i])
        {
            continue;
        }

        if(nn::nfc::server::util::IsEqualDeviceHandle(m_DeviceList[i]->GetHandle(), deviceHandle))
        {
            *pOutDevice = m_DeviceList[i].get();
            NN_RESULT_SUCCESS;
        }
    }

    return nn::nfc::ResultNfcDeviceNotFound();
}

nn::Result Manager::GetDevice(Device** pOutDevice, Service* service, nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
{
    return GetDevice(pOutDevice, service, deviceHandle, true);
}

void Manager::InitializeTerminalId() NN_NOEXCEPT
{
    if(m_IsInitializedTerminalId)
    {
        return;
    }

    std::memset(&m_TerminalId, 0, sizeof(m_TerminalId));
    m_IsInitializedTerminalId = true;

    int fileHandle = -1;
    bool isSuccess = false;

    // バックアップデータが存在しない場合には作成します。
    auto result = nn::nfc::server::core::SaveData::GetInstance().Create(nn::nfc::server::core::SaveData::FilePathNfcTerminalId, sizeof(m_TerminalId));

    if(result.IsSuccess())
    {
        //新規作成時
        nn::Bit64 randValue = ((static_cast<nn::Bit64>(std::rand() & 0xFFFF) << 48)
                               | (static_cast<nn::Bit64>(std::rand() & 0xFFFF) << 32)
                               | (static_cast<nn::Bit64>(std::rand() & 0xFFFF) << 16)
                               | (static_cast<nn::Bit64>(std::rand() & 0xFFFF)));
        {
            if(nn::nfc::server::core::SaveData::GetInstance().Open(&fileHandle, nn::nfc::server::core::SaveData::FilePathNfcTerminalId, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write).IsFailure())
            {
                return;
            }
            NN_UTIL_SCOPE_EXIT
            {
                nn::nfc::server::core::SaveData::GetInstance().Close(fileHandle);
            };

            if(nn::nfc::server::core::SaveData::GetInstance().Write(fileHandle, &randValue, sizeof(randValue), false).IsFailure())
            {
                return;
            }

            if(nn::nfc::server::core::SaveData::GetInstance().Flush(fileHandle).IsFailure())
            {
                return;
            }
        }

        if(nn::nfc::server::core::SaveData::Commit().IsFailure())
        {
            return;
        }

        isSuccess = true;
    }
    else if(!nn::nfc::server::core::ResultFsPathAlreadyExists::Includes(result))
    {
        //既に保存済みでもない
        return;
    }

    if(nn::nfc::server::core::SaveData::GetInstance().Open(&fileHandle, nn::nfc::server::core::SaveData::FilePathNfcTerminalId, nn::fs::OpenMode_Read).IsFailure())
    {
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::nfc::server::core::SaveData::GetInstance().Close(fileHandle);
    };

    if(nn::nfc::server::core::SaveData::GetInstance().Read(&m_TerminalId, fileHandle, sizeof(m_TerminalId)).IsFailure())
    {
        return;
    }
}


nn::Bit64 Manager::GetTerminalId() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);
    return m_TerminalId;
}

void Manager::ResetDevices(ResetReason reason, Service* service) NN_NOEXCEPT
{
    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(m_DeviceList[i])
        {
            if(m_DeviceList[i]->IsInitialized())
            {
                m_DeviceList[i]->Reset(reason, service);
            }
        }
    }
}

nn::Result Manager::StartMifareRead(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const MifareReadParameter& mifareReadParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, StartMifareRead(tagId, mifareReadParameter), service, xcdDeviceHandle);
}

nn::Result Manager::StartMifareWrite(Service* service, nn::nfc::DeviceHandle deviceHandle, const nn::nfc::TagId& tagId, const MifareWriteParameter& mifareWriteParameter) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    Device* device;
    NN_RESULT_DO(GetDevice(&device, service, deviceHandle));
    nn::xcd::DeviceHandle xcdDeviceHandle;
    NN_RESULT_DO(device->GetXcdDeviceHandle(&xcdDeviceHandle));

    NN_NFC_SERVER_MANAGER_CHECK_RETURN(device, StartMifareWrite(tagId, mifareWriteParameter), service, xcdDeviceHandle);
}

void Manager::SignalAvailabilityChangeEvent() NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        if(m_Service[i])
        {
            m_Service[i]->SignalAvailabilityChangeEvent();
        }
    }
}

void Manager::SignalEvent(Service* service, nn::nfc::DeviceHandle deviceHandle, bool isActivate) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    if(service == nullptr)
    {
        return;
    }

    /////////////////////////////////////////////////////////////////////////////
    // 指定されたサービスで無条件にシグナル
    if(isActivate)
    {
        service->SignalActivateEvent(deviceHandle);
    }
    else
    {
        service->SignalDeactivateEvent(deviceHandle);
    }


    /////////////////////////////////////////////////////////////////////////////
    // (新)NFC x NFP の組み合わせの場合のみもう一方のサービスでシグナル
    Service* otherService = nullptr;
    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        if(!m_Service[i])
        {
            continue;
        }

        if(m_Service[i]->GetAppletResourceUserId() == service->GetAppletResourceUserId())
        {
            auto type = m_Service[i]->GetLibraryType();

            if(type == nn::nfc::server::core::LibraryType_NfcMifare)
            {
                /* 一方のサービスが MIFARE であれば (新)NFC x NFP ではない */
                return;
            }

            if(m_Service[i] != service)
            {
                otherService = m_Service[i];
            }
        }
    }

    if(otherService)
    {
        /* ここは (新)NFC x NFP でないと通らない */

        if(isActivate)
        {
            otherService->SignalActivateEvent(deviceHandle);
        }
        else
        {
            otherService->SignalDeactivateEvent(deviceHandle);
        }
    }
}

void Manager::LinkMonitoringThread(void* arg) NN_NOEXCEPT
{
    Manager* manager = reinterpret_cast<Manager*>(arg);
    bool retryFlag = false;

    while (NN_STATIC_CONDITION(true))
    {
        // デバイス情報の更新を待機
        nn::os::MultiWaitHolderType* pHolder;
        if(retryFlag)
        {
            pHolder = nn::os::TimedWaitAny(&manager->m_LinkMonitoringMultiWait, nn::TimeSpan::FromMilliSeconds(LinkMonitoringThreadRetryInterval));
        }
        else
        {
            pHolder = nn::os::WaitAny(&manager->m_LinkMonitoringMultiWait);
        }
        {
            if (manager->m_IsRunningThread == false || pHolder == &manager->m_TerminateLinkMonitoringHolder)
            {
                break;
            }
            else
            {
                if (pHolder == &manager->m_LinkHolder || pHolder == nullptr)
                {
                    if(pHolder == &manager->m_LinkHolder)
                    {
                        nn::os::ClearSystemEvent(&manager->m_LinkEvent);
                    }
                    if(!manager->m_ReaderWriterLock.try_lock())
                    {
                        retryFlag = true;
                        continue;
                    }
                    manager->UpdateDevices();
                    manager->SignalLinkEvent();
                    manager->m_ReaderWriterLock.unlock();
                    retryFlag = false;
                }
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
                else if(pHolder == &manager->m_PmModuleHolder)
                {
                    manager->m_PmModule->GetEventPointer()->Clear();
                    manager->PmHandler();
                }
#endif
                else
                {
                    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
                    {
                        if(manager->m_DeviceList[i])
                        {
                            manager->m_DeviceList[i]->LoopEventThreadFunction(pHolder);
                        }
                    }
                }
            }
        }
    }
}

void Manager::UpdateDevices() NN_NOEXCEPT
{
    nn::hid::NpadIdType npadId;

    struct DeviceList
    {
        int deviceCount;
        nn::hid::NpadIdType npadIdList[nn::nfc::DeviceCountMax];
    } list;

    list.deviceCount = nn::hid::system::GetNpadsWithNfc(list.npadIdList, nn::nfc::DeviceCountMax);

    //不要なデバイスを削除
    for(auto i = 0; i < nn::nfc::DeviceCountMax; ++i)
    {
        if(!m_DeviceList[i])
        {
            continue;
        }

        m_DeviceList[i]->GetNpadId(&npadId);

        bool find = false;
        for(auto j = 0; j < list.deviceCount; ++j)
        {
            if(list.npadIdList[j] == npadId)
            {
                find = true;
                break;
            }
        }

        if(!find)
        {
            if(m_DeviceList[i])
            {
                m_DeviceList[i]->SetState(Device::State_Unexpected);
                m_DeviceList[i].reset(nullptr);
            }
        }
    }

    for(auto i = 0; i < list.deviceCount; ++i)
    {
        //既に見つかっているか
        bool find = false;
        for(auto j = 0; j < nn::nfc::DeviceCountMax; ++j)
        {
            if(!m_DeviceList[j])
            {
                continue;
            }

            m_DeviceList[j]->GetNpadId(&npadId);

            if(list.npadIdList[i] == npadId)
            {
                find = true;
                break;
            }
        }

        //なければ追加
        if(!find)
        {
            for(auto j = 0; j < nn::nfc::DeviceCountMax; ++j)
            {
                //空いているところを探して
                if(!m_DeviceList[j])
                {
                    m_DeviceList[j].reset(new Device(m_DeviceUniqueId++, list.npadIdList[i], &m_LinkMonitoringMultiWait));
                    if(IsInitialized())
                    {
                        m_DeviceList[j]->Initialize();
                    }
                    break;
                }
            }
        }
    }
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
void Manager::PmHandler() NN_NOEXCEPT
{
    nn::psc::PmFlagSet   flags;
    nn::psc::PmState     state;
    nn::Result           result;

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule->GetRequest(&state, &flags));

    switch (state)
    {
    case nn::psc::PmState_FullAwake:
        {
            m_IsAwake = true;
            result = nn::ResultSuccess();
        }
        break;
    case nn::psc::PmState_MinimumAwake:
    case nn::psc::PmState_SleepReady:
        {
            ResetDevices(ResetReason_Sleep, nullptr);
            m_IsAwake = false;
            SignalLinkEvent();
            result = nn::ResultSuccess();
        }
        break;
    default:
        {
            // no-op, but still return ResultSuccess()
            // for unhandled states - new states may be
            // added in the future.
            result = nn::ResultSuccess();
        }
        break;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule->Acknowledge(state, result));
}
#endif

bool Manager::IsInitialized(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    nn::nfc::server::util::ScopedLockShared lock(m_ReaderWriterLock);

    for(auto i = 0; i < ServiceCountMax; ++i)
    {
        if(m_Service[i] )
        {
            if(m_Service[i]->GetAppletResourceUserId() == aruid)
            {
                return true;
            }
        }
    }

    return false;
}

}}}}  // namespace nn::nfc::server::core
