﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/core/nfc_CoreManager.h>
#include "nfc_MifareMifare.h"
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfc/server/core/nfc_CoreUtil.h>
#include <nn/nfc/server/util/nfc_ScopedMutexLock.h>

namespace
{
nn::Result ConvertToNfcMifareResultForReadWrite(nn::Result nfcResult) NN_NOEXCEPT
{
    NN_RESULT_TRY(nfcResult)
        NN_RESULT_CATCH(nn::nfc::ResultTimeOutError)
        {
            NN_RESULT_THROW(nn::nfc::ResultAccessTimeOutError());
        }
        NN_RESULT_CATCH(nn::nfc::ResultOperationFailed)
        {
            NN_RESULT_THROW(nn::nfc::ResultAccessOperationFailed());
        }
        NN_RESULT_CATCH(nn::nfc::ResultNeedRetry)
        {
            NN_RESULT_THROW(nn::nfc::ResultAccessError());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

nn::Result CheckReadData(const nn::nfc::server::core::MifareReadParameter& parameter, const nn::xcd::NfcMifareData& mifareData)
{
    if(parameter.blockCount != mifareData.dataCount)
    {
        return nn::nfc::ResultAccessError();
    }

    std::unique_ptr<bool[]> checked(new bool[mifareData.dataCount]);
    for(int32_t i = 0; i < mifareData.dataCount; i++)
    {
        checked[i] = false;
    }

    for(int32_t i = 0; i < parameter.blockCount; i++)
    {
        bool find = false;
        for(int32_t j = 0; j < mifareData.dataCount; j++)
        {
            if(!checked[j] && parameter.blocks[i].blockAddress == mifareData.readDataBlocks[j].blockAddress)
            {
                checked[j] = true;
                find = true;
                break;
            }
        }
        if(!find)
        {
            return nn::nfc::ResultAccessError();
        }
    }

    NN_RESULT_SUCCESS;
}

}

namespace nn { namespace nfc { namespace mifare { namespace server {

Mifare::Mifare(const nn::nfc::DeviceHandle& deviceHandle, const nn::nfc::TagId& id, nn::os::SystemEventType* accessFinishEvent, nn::os::SystemEventType* accessResetEvent) NN_NOEXCEPT : Tag(deviceHandle, id, nn::nfc::NfcProtocol_TypeA, nn::nfc::TagType_Mifare, accessFinishEvent, accessResetEvent)
{
}

Mifare::~Mifare() NN_NOEXCEPT
{
};


nn::Result Mifare::Read(nn::nfc::MifareReadBlockData* pOutBlockData, nn::nfc::server::core::Service* service, const nn::nfc::MifareReadBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pOutBlockData);
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pBlockParameter);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < blockCount);

    nn::Bit8 format = pBlockParameter[0].key.valueFormat;
    for(size_t i = 1; i < blockCount; ++i)
    {
        NN_NFC_SERVER_UTIL_REQUIRES_EQUAL(format, pBlockParameter[i].key.valueFormat);
    }

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    std::unique_ptr<nn::nfc::server::core::MifareReadParameter> parameter(new nn::nfc::server::core::MifareReadParameter());
    parameter->timeoutMsec = 2000;
    parameter->keyFormat = static_cast<nn::xcd::MifareKeyValueFormat>(format);
    parameter->tagId = m_Id;

    size_t loop = (blockCount + (nn::xcd::MifareReadBlockCountMax - 1)) / nn::xcd::MifareReadBlockCountMax;
    size_t lastBlockCount;

    if((blockCount % nn::xcd::MifareReadBlockCountMax) == 0)
    {
        lastBlockCount = nn::xcd::MifareReadBlockCountMax;
    }
    else
    {
        lastBlockCount = (blockCount % nn::xcd::MifareReadBlockCountMax);
    }

    size_t blockCountInLoop = nn::xcd::MifareReadBlockCountMax;
    size_t offset = 0;
    for(size_t i = 0; i < loop; ++i)
    {
        if(i == loop - 1)
        {
            blockCountInLoop = lastBlockCount;
        }

        for(size_t j = 0; j < blockCountInLoop; ++j)
        {
            parameter->blocks[j].key.type = static_cast<nn::xcd::MifareKeyType>(pBlockParameter[offset + j].key.type);
            std::memcpy(parameter->blocks[j].key._dummy, pBlockParameter[offset + j].key._dummy, sizeof(parameter->blocks[j].key._dummy));
            parameter->blocks[j].blockAddress = pBlockParameter[offset + j].blockAddr;
        }
        parameter->blockCount = static_cast<int32_t>(blockCountInLoop);

        ClearEvent();
        NN_RESULT_DO(ConvertToNfcMifareResultForReadWrite(nn::nfc::server::core::Manager::GetInstance().StartMifareRead(service, m_Handle, m_Id, *parameter)));

        std::unique_ptr<nn::nfc::server::core::Info> info(new nn::nfc::server::core::Info());
        NN_RESULT_DO(ConvertToNfcMifareResultForReadWrite(WaitInfo(info.get(), service, m_Handle, parameter->timeoutMsec + 1000)));
        switch(info->reason)
        {
        case nn::xcd::NfcEventReason_MifareResult:
            {
                NN_RESULT_DO(CheckReadData(*parameter, info->mifareData));
                for(int32_t j = 0; j < info->mifareData.dataCount; ++j)
                {
                    pOutBlockData[offset + j].blockAddr = info->mifareData.readDataBlocks[j].blockAddress;
                    std::memcpy(pOutBlockData[offset + j].buffer, info->mifareData.readDataBlocks[j].data, sizeof(pOutBlockData[j].buffer));
                }
                offset += blockCountInLoop;
            }
            break;
        case nn::xcd::NfcEventReason_Error:
            {
                return ConvertToNfcMifareResultForReadWrite(nn::nfc::server::core::ConvertToNfcResult(info->errorInfo.resultCode));
            }
        default:
            {
                return nn::nfc::ResultAccessError();
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result Mifare::Write(nn::nfc::server::core::Service* service, const nn::nfc::MifareWriteBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT
{
    NN_NFC_SERVER_UTIL_REQUIRES_NOT_NULL(pBlockParameter);
    NN_NFC_SERVER_UTIL_REQUIRES(0 < blockCount);

    nn::Bit8 format = pBlockParameter[0].key.valueFormat;
    for(size_t i = 1; i < blockCount; ++i)
    {
        NN_NFC_SERVER_UTIL_REQUIRES_EQUAL(format, pBlockParameter[i].key.valueFormat);
    }

    nn::nfc::server::util::ScopedMutexLock lock(m_Mutex);

    std::unique_ptr<nn::nfc::server::core::MifareWriteParameter> parameter(new nn::nfc::server::core::MifareWriteParameter());
    parameter->timeoutMsec = 2000;
    parameter->keyFormat = static_cast<nn::xcd::MifareKeyValueFormat>(format);
    parameter->tagId = m_Id;

    size_t loop = (blockCount + (nn::xcd::MifareWriteBlockCountMax - 1)) / nn::xcd::MifareWriteBlockCountMax;
    size_t lastBlockCount;

    if((blockCount % nn::xcd::MifareWriteBlockCountMax) == 0)
    {
        lastBlockCount = nn::xcd::MifareWriteBlockCountMax;
    }
    else
    {
        lastBlockCount = (blockCount % nn::xcd::MifareWriteBlockCountMax);
    }

    size_t blockCountInLoop = nn::xcd::MifareWriteBlockCountMax;
    size_t offset = 0;
    for(size_t i = 0; i < loop; ++i)
    {
        if(i == loop - 1)
        {
            blockCountInLoop = lastBlockCount;
        }

        for(size_t j = 0; j < blockCountInLoop; ++j)
        {
            parameter->blocks[j].key.type = static_cast<nn::xcd::MifareKeyType>(pBlockParameter[offset + j].key.type);
            std::memcpy(parameter->blocks[j].key._dummy, pBlockParameter[offset + j].key._dummy, sizeof(parameter->blocks[j].key._dummy));
            parameter->blocks[j].blockAddress = pBlockParameter[offset + j].blockAddr;
            std::memcpy(parameter->blocks[j].data, pBlockParameter[offset + j].data, sizeof(parameter->blocks[j].data));
        }
        parameter->blockCount = static_cast<int32_t>(blockCountInLoop);

        ClearEvent();
        NN_RESULT_DO(ConvertToNfcMifareResultForReadWrite(nn::nfc::server::core::Manager::GetInstance().StartMifareWrite(service, m_Handle, m_Id, *parameter)));

        std::unique_ptr<nn::nfc::server::core::Info> info(new nn::nfc::server::core::Info());
        NN_RESULT_DO(ConvertToNfcMifareResultForReadWrite(WaitInfo(info.get(), service, m_Handle, parameter->timeoutMsec + 1000)));
        switch(info->reason)
        {
        case nn::xcd::NfcEventReason_MifareResult:
            {
                offset += blockCountInLoop;
            }
            break;
        case nn::xcd::NfcEventReason_Error:
            {
                return ConvertToNfcMifareResultForReadWrite(nn::nfc::server::core::ConvertToNfcResult(info->errorInfo.resultCode));
            }
        default:
            {
                return nn::nfc::ResultAccessError();
            }
        }
    }

    NN_RESULT_SUCCESS;
}

bool Mifare::IsMifare(nn::nfc::NfcProtocol protocol, nn::nfc::TagType type) NN_NOEXCEPT
{
    return (protocol == nn::nfc::NfcProtocol_TypeA && type == nn::nfc::TagType_Mifare);
}

}}}}  // namespace nn::nfc::mifare::server
