﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/service/news_Server.h>
#include <nn/news/detail/ipc/news_ServiceConfig.h>
#include <nn/news/detail/ipc/news_ServiceName.h>
#include <nn/news/detail/service/news_Capability.h>
#include <nn/news/detail/service/news_Service.h>
#include <nn/news/detail/service/core/news_AccountCountryChecker.h>
#include <nn/news/detail/service/core/news_FileSystem.h>
#include <nn/news/detail/service/core/news_NewsTaskManager.h>
#include <nn/news/detail/service/core/news_NewsSubscriber.h>
#include <nn/news/detail/service/core/news_One2OneNotificationManager.h>
#include <nn/news/detail/service/core/news_DownloaderThread.h>
#include <nn/os.h>
#include <nn/npns.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/crypto/crypto_Md5Generator.h>

namespace nn { namespace news { namespace service {

namespace
{
    const int ParallelExecutionCountMax = 1;

    nn::os::ThreadType g_Threads[ParallelExecutionCountMax] = {};
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStacks[ParallelExecutionCountMax][32 * 1024];

    struct NewsServiceHipcSimpleAllInOneServerManagerOption
    {
        // これ以上のサイズのデータは、[BufferTransferMode(BufferTransferMode.MapAlias)] でマップ転送に切り替えること。
        static const size_t PointerTransferBufferSize = 1152;

        static const bool CanDeferInvokeRequest = false;

        static const int SubDomainCountMax = 64;
        static const int ObjectInSubDomainCountMax = 16;
    };

    class NewsServiceServerManager :
        public nn::sf::HipcSimpleAllInOneServerManager<64, detail::ipc::ServiceCount,
            NewsServiceHipcSimpleAllInOneServerManagerOption>
    {
    };

    NewsServiceServerManager* g_ServerManager = nullptr;
    std::aligned_storage<sizeof (NewsServiceServerManager)>::type g_ServerManagerStorage;
}

namespace
{
    void RegisterService(int index, const char* serviceName, Bit32 capability) NN_NOEXCEPT
    {
        detail::service::InitializeServiceCreator(index, serviceName, capability);

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager->RegisterObjectForPort(detail::service::GetServiceCreator(index),
            detail::ipc::SessionCountMax, serviceName));
    }

    void IpcDispatcher(void*) NN_NOEXCEPT
    {
        g_ServerManager->LoopAuto();
    }
}

void StartServer() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_INFO("[news] Start service...\n");

    g_ServerManager = new (&g_ServerManagerStorage) NewsServiceServerManager;

    // news:p
    RegisterService(0, detail::ipc::ServiceNameForPost,
        detail::service::Capability::Flag_Post);

    // news:v
    RegisterService(1, detail::ipc::ServiceNameForViewer,
        detail::service::Capability::Flag_View);

    // news:m
    RegisterService(2, detail::ipc::ServiceNameForManager,
        detail::service::Capability::Flag_Post | detail::service::Capability::Flag_View | detail::service::Capability::Flag_Management);

    // news:c
    RegisterService(3, detail::ipc::ServiceNameForConfiguration,
        detail::service::Capability::Flag_Config);

    // news:a
    RegisterService(4, detail::ipc::ServiceNameForAdministrator,
        detail::service::Capability::Flag_All);

    g_ServerManager->Start();

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::FileSystem::Mount(NN_DETAIL_NEWS_DATA_MOUNT_NAME, false));

#if NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS == 1
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::FileSystem::Mount(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME, false));
#endif

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::NewsTaskManager::GetInstance().Load());
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::One2OneNotificationManager::GetInstance().Load());

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i],
            IpcDispatcher, nullptr, g_ThreadStacks[i], sizeof (g_ThreadStacks[i]), NN_SYSTEM_THREAD_PRIORITY(news, IpcServer)));

        nn::os::SetThreadNamePointer(&g_Threads[i], NN_SYSTEM_THREAD_NAME(news, IpcServer));
        nn::os::StartThread(&g_Threads[i]);
    }

    detail::service::core::DownloaderThread::Start();

    NN_DETAIL_NEWS_INFO("[news] Start service... done!\n");
}

void StopServer() NN_NOEXCEPT
{
    g_ServerManager->RequestStop();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        nn::os::DestroyThread(&g_Threads[i]);
    }

    g_ServerManager->~NewsServiceServerManager();
    g_ServerManager = nullptr;

    detail::service::FinalizeServiceCreators();

    detail::service::core::DownloaderThread::Stop();

    detail::service::core::FileSystem::Unmount(NN_DETAIL_NEWS_DATA_MOUNT_NAME, false);

#if NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS == 1
    detail::service::core::FileSystem::Unmount(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME, false);
#endif

    NN_DETAIL_NEWS_INFO("[news] End service.\n");
}

bool GetHalfAwakeScheduleInterval(nn::bgtc::Interval* outSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSeconds);

    nn::TimeSpan interval;

    if (!detail::service::core::DownloaderThread::GetNextScheduleInterval(&interval))
    {
        return false;
    }

    // 分解能が秒になるので、+1 しておく。
    *outSeconds = static_cast<nn::bgtc::Interval>(interval.GetSeconds() + 1);

    return true;
}

void NotifySystemHalfAwaked() NN_NOEXCEPT
{
    detail::service::core::DownloaderThread::NotifySystemRunnableStateChanged(true);
}

void NotifySystemFullAwaked() NN_NOEXCEPT
{
    detail::service::core::NewsTaskManager::GetInstance().NotifySleepAwaked();

    detail::service::core::DownloaderThread::NotifySystemRunnableStateChanged(true);
}

void NotifySystemSleepReady(bool isFromFullAwake) NN_NOEXCEPT
{
    NN_UNUSED(isFromFullAwake);

    detail::service::core::DownloaderThread::NotifySystemRunnableStateChanged(false);
}

void NotifySystemShutdownReady() NN_NOEXCEPT
{
    detail::service::core::DownloaderThread::NotifySystemRunnableStateChanged(false);
}

void WaitForSleepReady() NN_NOEXCEPT
{
}

void WaitForShutdownReady() NN_NOEXCEPT
{
}

bool NotifyNotificationReceived(const nn::bcat::service::NotificationPayload& payload) NN_NOEXCEPT
{
    char formatType[8];
    size_t size;

    if (!payload.GetValue(&size, "type", formatType, sizeof (formatType)))
    {
        return false;
    }
    if (nn::util::Strncmp(formatType, "news", sizeof (formatType)) != 0)
    {
        return false;
    }

    TopicId topicId;

    if (!payload.GetValue(&size, "topic_id", topicId.value, sizeof (topicId.value)))
    {
        NN_DETAIL_NEWS_INFO("[news] 'topic_id' is not included in the payload.\n");
        return false;
    }
    if (!detail::service::core::StringVerifier::VerifyTopicId(topicId.value))
    {
        NN_DETAIL_NEWS_INFO("[news] The topic id is invalid.\n");
        return false;
    }

    int64_t waitRange = 0;

    payload.GetValue(&waitRange, "wait_range");

    if (waitRange < 0 || waitRange > INT16_MAX)
    {
        NN_DETAIL_NEWS_INFO("[news] The range of wait time is invalid.\n");
        return false;
    }

    int16_t waitTime = static_cast<int16_t>(detail::service::util::GetRandom(0, waitRange + 1));

    int64_t one2one = 0;

    payload.GetValue(&one2one, "one2one");

    if (one2one)
    {
        SubscriptionStatus status = detail::service::core::NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId);

        if (status == SubscriptionStatus_Unsubscribed || status == SubscriptionStatus_Unconfigured)
        {
            NN_DETAIL_NEWS_INFO("[news] '%s' is not subscribed.\n", topicId.value);
            return true;
        }

        detail::Url url;

        if (!payload.GetValue(&size, "url", url.value, sizeof (url.value)))
        {
            NN_DETAIL_NEWS_INFO("[news] 'url' is not included in the payload.\n");
            return false;
        }

        NN_DETAIL_NEWS_INFO("[news] The notification was received. (%s, 0-%lld)\n", url.value, waitRange);

        detail::service::core::One2OneNotificationManager::GetInstance().NotifyNotificationReceived(url, waitTime);
    }
    else
    {
        char revision[64] = {};

        if (!payload.GetValue(&size, "revision", revision, sizeof (revision)))
        {
            NN_DETAIL_NEWS_INFO("[news] 'revision' is not included in the payload.\n");
            return false;
        }

        int64_t pseudo = 0;

        payload.GetValue(&pseudo, "pseudo");

        Bit8 hash[16];
        nn::crypto::GenerateMd5Hash(hash, 16, revision, size);

        detail::RevisionHash revisionHash;
        std::memcpy(revisionHash.value, hash, sizeof (revisionHash.value));

        NN_DETAIL_NEWS_INFO("[news] The notification was received. (%s%s, %s, 0-%lld)\n", topicId.value,
            (pseudo != 0) ? " (pseudo)" : "", revision, waitRange);

        detail::service::core::NewsTaskManager::GetInstance().NotifyNotificationReceived(topicId,
            (pseudo != 0), revisionHash, waitTime);
    }

    return true;
}

void NotifyNpnsStateChanged() NN_NOEXCEPT
{
    nn::npns::State state = nn::npns::GetState();

    if (state == nn::npns::State_Connected || state == nn::npns::State_ConnectedOnHalfAwake)
    {
        detail::service::core::DownloaderThread::NotifyNpnsConnected();
    }
}

void NotifyNetworkServiceAccountAvailabilityChanged() NN_NOEXCEPT
{
    if (detail::service::core::Account::IsAvailableNintendoAccountLinked())
    {
        detail::service::core::NewsTaskManager::GetInstance().NotifyNintendoAccountLinked();

        if (detail::service::core::AccountCountryChecker::GetInstance().UpdateQueryString())
        {
            detail::service::core::NewsTaskManager::GetInstance().NotifyAccountCountryListUpdated();
            detail::service::core::NewsSubscriber::NotifyAccountCountryListUpdated();
        }
    }
}

void NotifyNetworkConnected() NN_NOEXCEPT
{
    detail::service::core::DownloaderThread::NotifyNetworkConnected();
    detail::service::core::NewsSubscriber::NotifyNetworkConnected();
}

}}}
