﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_One2OneNotificationManager.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* FilePath = "news-sys:/one2one.bin";

    const nn::os::Tick RunnableTickMax = nn::os::Tick(INT64_MAX);
}

One2OneNotificationManager::One2OneNotificationManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_Event(nn::os::EventClearMode_ManualClear),
    m_Count(0),
    m_IsNetworkConnectionAvailable(false)
{
    std::memset(m_Records, 0, sizeof (m_Records));
    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));
}

nn::Result One2OneNotificationManager::Load() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_TRY(LoadImpl())
        NN_RESULT_CATCH_ALL
        {
            std::memset(m_Records, 0, sizeof (m_Records));
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));

    for (int i = 0; i < m_Count; i++)
    {
        SetWait(i, 0);
    }

    NN_RESULT_SUCCESS;
}

nn::Result One2OneNotificationManager::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    std::memset(m_Records, 0, sizeof (m_Records));
    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));
    m_Count = 0;

    NN_RESULT_SUCCESS;
}

nn::Result One2OneNotificationManager::GetCurrentNotification(Record* outRecord) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outRecord);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsNetworkConnectionAvailable, ResultNotFound());

    int index = -1;

    nn::os::Tick currentTick = nn::os::GetSystemTick();

    for (int i = 0; i < m_Count; i++)
    {
        if (m_VolatileRecords[i].runnableTick <= currentTick)
        {
            index = i;
            break;
        }
    }

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFound());

    *outRecord = m_Records[index];

    NN_RESULT_SUCCESS;
}

bool One2OneNotificationManager::GetNextScheduleInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInterval);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!GetScheduleMinInterval(outInterval))
    {
        return false;
    }

    if (*outInterval <= 0)
    {
        return false;
    }

    return true;
}

nn::os::TimerEvent& One2OneNotificationManager::GetEvent() NN_NOEXCEPT
{
    return m_Event;
}

nn::Result One2OneNotificationManager::NotifyNotificationReceived(const Url& url, int32_t waitTime) NN_NOEXCEPT
{
    if (!VerifyUrl(url))
    {
        NN_DETAIL_NEWS_WARN("[news] The url is invalid.\n");
        NN_RESULT_THROW(ResultInvalidArgument());
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(url);

    if (index != -1)
    {
        NN_DETAIL_NEWS_WARN("[news] The url has already been included in the notification list.\n");
        NN_RESULT_SUCCESS;
    }

    AddRecord(url);
    SetWait(m_Count - 1, waitTime);

    SetScheduleTimer();

    Save();

    NN_RESULT_SUCCESS;
}

nn::Result One2OneNotificationManager::NotifyDone(const Record& record, nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(record.url);

    if (index == -1)
    {
        NN_RESULT_SUCCESS;
    }

    if (result.IsSuccess() || nn::bcat::ResultServerError404::Includes(result))
    {
        RemoveRecord(index);
        Save();
    }
    else if (IsServerTemporaryError(result))
    {
        // 通信エラーは 5 分後に再試行する。
        SetWait(index, 5 * 60);
    }
    else if (IsServerFailureError(result))
    {
        // サーバー障害は 12 時間後に再試行する。
        SetWait(index, 12 * 3600);
    }
    else if (ResultInternetRequestNotAccepted::Includes(result))
    {
        m_IsNetworkConnectionAvailable = false;
        SetWait(index, 0);
    }
    else if (ResultNintendoAccountNotLinked::Includes(result))
    {
        // 通知を受信した後にニンテンドーアカウントの紐付けを解除した場合、通知を破棄する。
        RemoveRecord(index);
        Save();
    }
    else
    {
        // それ以外のエラーは再起動まで再試行しない。
        SetWait(index, -1);
    }

    SetScheduleTimer();

    NN_RESULT_SUCCESS;
}

void One2OneNotificationManager::NotifyNetworkConnected() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_IsNetworkConnectionAvailable)
    {
        m_IsNetworkConnectionAvailable = true;
        m_Event.Signal();
    }
}

nn::Result One2OneNotificationManager::LoadImpl() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (m_Records)));

    m_Count = 0;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Records); i++)
    {
        if (m_Records[i].url.value[0] == '\0')
        {
            break;
        }

        m_Count++;
    }

    if (!Verify())
    {
        NN_DETAIL_NEWS_WARN("[news] %s is corrupted. (verification failed)\n", FilePath);
        NN_RESULT_THROW(ResultSaveVerificationFailed());
    }

    NN_RESULT_SUCCESS;
}

nn::Result One2OneNotificationManager::Save() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(FileSystem::CreateFile(FilePath, sizeof (m_Records)));

    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (m_Records),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME));

    NN_RESULT_SUCCESS;
}

bool One2OneNotificationManager::Verify() const NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (!VerifyUrl(m_Records[i].url))
        {
            return false;
        }
    }

    return true;
}

int One2OneNotificationManager::SearchRecord(const Url& url) const NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (nn::util::Strncmp(url.value, m_Records[i].url.value, sizeof (url.value)) == 0)
        {
            return i;
        }
    }

    return -1;
}

void One2OneNotificationManager::AddRecord(const Url& url) NN_NOEXCEPT
{
    if (m_Count == NN_ARRAY_SIZE(m_Records))
    {
        RemoveOldestRecord();
    }

    Record record = {};
    VolatileRecord volatileRecord = {nn::os::Tick()};

    if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&record.receivedTime).IsFailure())
    {
        record.receivedTime.value = 0;
    }

    record.url = url;

    m_Records[m_Count] = record;
    m_VolatileRecords[m_Count] = volatileRecord;

    m_Count++;
}

void One2OneNotificationManager::RemoveRecord(int index) NN_NOEXCEPT
{
    int moveCount = m_Count - 1 - index;

    if (moveCount > 0)
    {
        std::memmove(&m_Records[index], &m_Records[index + 1], sizeof (Record) * moveCount);
        std::memmove(&m_VolatileRecords[index], &m_VolatileRecords[index + 1], sizeof (VolatileRecord) * moveCount);
    }

    // ゴミデータが残らないよう、末尾をゼロクリアする。
    std::memset(&m_Records[--m_Count], 0, sizeof (Record));
}

void One2OneNotificationManager::RemoveOldestRecord() NN_NOEXCEPT
{
    if (m_Count == 0)
    {
        return;
    }

    int index = 0;

    for (int i = 1; i < m_Count; i++)
    {
        if (m_Records[i].receivedTime < m_Records[index].receivedTime)
        {
            index = i;
        }
    }

    RemoveRecord(index);
}

void One2OneNotificationManager::SetWait(int index, int32_t waitTime) NN_NOEXCEPT
{
    if (waitTime >= 0)
    {
        m_VolatileRecords[index].runnableTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(waitTime));

        if (waitTime != 0)
        {
            NN_DETAIL_NEWS_INFO("[news] Wait for %d seconds ...\n", waitTime);
        }

        SetScheduleTimer();
    }
    else
    {
        m_VolatileRecords[index].runnableTick = RunnableTickMax;
    }
}

void One2OneNotificationManager::SetScheduleTimer() NN_NOEXCEPT
{
    nn::TimeSpan interval;

    if (GetScheduleMinInterval(&interval))
    {
        if (interval > 0)
        {
            m_Event.StartOneShot(interval);
        }
        else
        {
            m_Event.Signal();
        }
    }
}

bool One2OneNotificationManager::GetScheduleMinInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT
{
    nn::os::Tick nextRunnableTick = RunnableTickMax;
    bool hasRunnable = false;

    for (int i = 0; i < m_Count; i++)
    {
        if (m_VolatileRecords[i].runnableTick < nextRunnableTick)
        {
            nextRunnableTick = m_VolatileRecords[i].runnableTick;
            hasRunnable = true;
        }
    }

    if (!hasRunnable)
    {
        return false;
    }

    *outInterval = (nextRunnableTick - nn::os::GetSystemTick()).ToTimeSpan();

    return true;
}

bool One2OneNotificationManager::IsServerTemporaryError(nn::Result result) NN_NOEXCEPT
{
    if (nn::bcat::ResultHttpError::Includes(result))
    {
        return true;
    }

    return false;
}

bool One2OneNotificationManager::IsServerFailureError(nn::Result result) NN_NOEXCEPT
{
    // CDN の帯域制限の場合も 403 エラーが返る可能性がある。
    if (nn::bcat::ResultServerError403::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError500::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError502::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError503::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError504::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError509::Includes(result))
    {
        return true;
    }

    return false;
}

bool One2OneNotificationManager::VerifyUrl(const Url& url) NN_NOEXCEPT
{
    // https 以外は入っていないはず。
    if (nn::util::Strncmp(url.value, "https://", sizeof ("https://") - 1) != 0)
    {
        return false;
    }

    // MEMO: 必要なら、FQDN が *.nintendo.net に該当するかどうかも確認する。

    return true;
}

}}}}}
