﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsTaskManager.h>
#include <nn/news/detail/service/core/news_IncrementalId.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* FilePath = "news-sys:/task.bin";

    const nn::os::Tick RunnableTickMax = nn::os::Tick(INT64_MAX);
}

NN_STATIC_ASSERT(NewsTaskManager::TaskCountMax == TopicCountMax);

NewsTaskManager::NewsTaskManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_ScheduleEvent(nn::os::EventClearMode_ManualClear),
    m_SubscriptionEvent(nn::os::EventClearMode_ManualClear),
    m_SubscriptionCount(0),
    m_IsNetworkConnectionAvailable(false)
{
    std::memset(m_Records, 0, sizeof (m_Records));
    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));
}

nn::Result NewsTaskManager::Load() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_TRY(LoadImpl())
        NN_RESULT_CATCH_ALL
        {
            std::memset(m_Records, 0, sizeof (m_Records));
            m_SubscriptionCount = 0;
        }
    NN_RESULT_END_TRY;

    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    detail::service::core::FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    std::memset(m_Records, 0, sizeof (m_Records));
    std::memset(m_VolatileRecords, 0, sizeof (m_VolatileRecords));

    m_SubscriptionCount = 0;

    NN_RESULT_SUCCESS;
}

void NewsTaskManager::Schedule() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::os::Tick currentTick = nn::os::GetSystemTick();

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].status == TaskStatus_Wait)
        {
            if (m_VolatileRecords[i].runnableTick <= currentTick)
            {
                SetRunnableWithoutSignal(i);
            }
        }
    }

    SetScheduleTimer();
}

SubscriptionStatus NewsTaskManager::GetSubscriptionStatus(const TopicId& topicId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(topicId);

    if (index == -1)
    {
        return SubscriptionStatus_Unconfigured;
    }

    switch (m_Records[index].subscription)
    {
    case InternalSubscriptionStatus_Unsubscribed:
        {
            return SubscriptionStatus_Unsubscribed;
        }
    case InternalSubscriptionStatus_Subscribed:
        {
            return m_Records[index].isAuto ? SubscriptionStatus_AutoSubscribed : SubscriptionStatus_Subscribed;
        }
    default:
        {
            return SubscriptionStatus_Unconfigured;
        }
    }
}

nn::Result NewsTaskManager::SetSubscriptionStatus(const TopicId& topicId, SubscriptionStatus status) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(topicId);

    if (status == SubscriptionStatus_Unconfigured)
    {
        NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

        if (m_Records[index].subscription == InternalSubscriptionStatus_Subscribed)
        {
            m_SubscriptionCount--;
            NN_SDK_ASSERT(m_SubscriptionCount >= 0);
        }

        // NPNS トピックが購読されている場合、サーバーに解除を通知するまでトピック ID を残しておく。
        if (m_Records[index].isNpnsRegistered)
        {
            m_Records[index].subscription = InternalSubscriptionStatus_Deleted;

            m_Records[index].isNpnsPerformed = false;
            m_SubscriptionEvent.Signal();
        }
        else
        {
            std::memset(&m_Records[index], 0, sizeof (Record));
            std::memset(&m_VolatileRecords[index], 0, sizeof (VolatileRecord));
        }
    }
    else
    {
        if (index == -1)
        {
            NN_RESULT_DO(AddRecord(topicId, status));
        }
        else
        {
            if (status == SubscriptionStatus_Unsubscribed)
            {
                if (m_Records[index].subscription == InternalSubscriptionStatus_Subscribed)
                {
                    m_SubscriptionCount--;
                    NN_SDK_ASSERT(m_SubscriptionCount >= 0);
                }
                if (m_Records[index].subscription != InternalSubscriptionStatus_Unsubscribed)
                {
                    uint64_t id;
                    NN_RESULT_DO(IncrementalId::GetInstance().Issue(&id));

                    m_Records[index].subscription = InternalSubscriptionStatus_Unsubscribed;
                    m_Records[index].unsubscribedIndex = id;

                    m_Records[index].isNpnsPerformed = false;
                    m_SubscriptionEvent.Signal();
                }
            }
            else if (status == SubscriptionStatus_Subscribed || status == SubscriptionStatus_AutoSubscribed)
            {
                if (m_Records[index].subscription != InternalSubscriptionStatus_Subscribed)
                {
                    NN_RESULT_THROW_UNLESS(m_SubscriptionCount < SubscriptionCountMax, ResultSubscriptionLimitReached());

                    m_Records[index].subscription = InternalSubscriptionStatus_Subscribed;

                    m_SubscriptionCount++;

                    m_Records[index].isNpnsPerformed = false;
                    m_SubscriptionEvent.Signal();
                }

                m_Records[index].isAuto = (status == SubscriptionStatus_AutoSubscribed);
            }
        }
    }

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::GetCurrentTask(Record* outRecord) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outRecord);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsNetworkConnectionAvailable, ResultNotFound());

    int index = -1;

    for (int i = 0; i < TaskCountMax; i++)
    {
        const Record& record = m_Records[i];

        if (record.status == TaskStatus_Running && record.subscription == InternalSubscriptionStatus_Subscribed)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        for (int i = 0; i < TaskCountMax; i++)
        {
            const Record& record = m_Records[i];

            if (record.status == TaskStatus_Runnable && record.subscription == InternalSubscriptionStatus_Subscribed)
            {
                index = i;
                break;
            }
        }
    }

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    *outRecord = m_Records[index];

    NN_RESULT_SUCCESS;
}

void NewsTaskManager::GetTopicList(int* outCount, TopicId* outTopicIds, int count, SubscriptionStatusFilter filter) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outTopicIds);
    NN_SDK_REQUIRES_GREATER(count, 0);

    FilterFunction filterFunction = nullptr;

    switch (filter)
    {
    case SubscriptionStatusFilter_None:
        filterFunction = FilterWithNone;
        break;
    case SubscriptionStatusFilter_Unsubscribed:
        filterFunction = FilterWithUnsubscribed;
        break;
    case SubscriptionStatusFilter_Subscribed:
        filterFunction = FilterWithSubscribed;
        break;
    default:
        *outCount = 0;
        return;
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int actualCount = 0;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (filterFunction(m_Records[i].subscription))
        {
            if (count-- == 0)
            {
                break;
            }

            outTopicIds[actualCount++] = m_Records[i].topicId;
        }
    }

    *outCount = actualCount;
}

bool NewsTaskManager::GetNextScheduleInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInterval);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!GetScheduleMinInterval(outInterval))
    {
        return false;
    }

    if (*outInterval <= 0)
    {
        return false;
    }

    return true;
}

bool NewsTaskManager::IsNpnsSubscriptionProcessingRequired() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (!m_Records[i].isNpnsRegistered && !m_Records[i].isNpnsPerformed &&
            m_Records[i].subscription == InternalSubscriptionStatus_Subscribed)
        {
            return true;
        }
        if (m_Records[i].isNpnsRegistered && !m_Records[i].isNpnsPerformed &&
            (m_Records[i].subscription == InternalSubscriptionStatus_Deleted || m_Records[i].subscription == InternalSubscriptionStatus_Unsubscribed))
        {
            return true;
        }
    }

    return false;
}

bool NewsTaskManager::GetNpnsTopicToBeSubscribed(TopicId* outTopicId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTopicId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (!m_Records[i].isNpnsRegistered && !m_Records[i].isNpnsPerformed &&
            m_Records[i].subscription == InternalSubscriptionStatus_Subscribed)
        {
            *outTopicId = m_Records[i].topicId;
            return true;
        }
    }

    return false;
}

bool NewsTaskManager::GetNpnsTopicToBeUnsubscribed(TopicId* outTopicId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTopicId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].isNpnsRegistered && !m_Records[i].isNpnsPerformed &&
            (m_Records[i].subscription == InternalSubscriptionStatus_Deleted || m_Records[i].subscription == InternalSubscriptionStatus_Unsubscribed))
        {
            nn::util::Strlcpy(outTopicId->value, m_Records[i].topicId.value, sizeof (outTopicId->value));
            return true;
        }
    }

    return false;
}

nn::os::TimerEvent& NewsTaskManager::GetScheduleEvent() NN_NOEXCEPT
{
    return m_ScheduleEvent;
}

nn::os::Event& NewsTaskManager::GetSubscriptionEvent() NN_NOEXCEPT
{
    return m_SubscriptionEvent;
}

nn::Result NewsTaskManager::Run(const TopicId& topicId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(topicId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    if (m_Records[index].status == TaskStatus_Running)
    {
        NN_RESULT_SUCCESS;
    }

    SetRunnable(index);

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::RunImmediately(const TopicId& topicId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int runningIndex = -1;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].status == TaskStatus_Running)
        {
            runningIndex = i;
            break;
        }
    }

    int index = SearchRecord(topicId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    // 現在実行中のタスクを中断する。
    if (runningIndex != -1)
    {
        SetRunnable(index);
    }

    m_Records[index].status = TaskStatus_Running;
    m_ScheduleEvent.Signal();

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::Wait(const TopicId& topicId, int32_t waitTime) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(topicId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    if (m_Records[index].status == TaskStatus_Wait ||
        m_Records[index].status == TaskStatus_Runnable || m_Records[index].status == TaskStatus_Running)
    {
        SetWait(index, waitTime);
    }
    else
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::UpdateETag(const TopicId& topicId, const ETag& eTag) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(topicId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    m_Records[index].eTag = eTag;

    // NotifyDone 時の Save に任せる。
    // NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifyNotificationReceived(const TopicId& topicId, bool isPseudo, const RevisionHash& revisionHash, int32_t waitTime) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(topicId);

    if (index == -1)
    {
        if (isPseudo)
        {
            NN_DETAIL_NEWS_INFO("[news] '%s' is unmanaged pseudo topic id. Do nothing.\n", topicId.value);
            NN_RESULT_SUCCESS;
        }

        Record record = {};

        // NPNS トピックを購読解除するためのタスクを作成する。
        record.topicId = topicId;
        record.subscription = InternalSubscriptionStatus_Deleted;
        record.isNpnsRegistered = true;

        for (int i = 0; i < TaskCountMax; i++)
        {
            if (m_Records[i].status == TaskStatus_Empty)
            {
                std::memcpy(&m_Records[i], &record, sizeof (Record));
                break;
            }
        }

        NN_DETAIL_NEWS_INFO("[news] '%s' is unmanaged topic id.\n", topicId.value);
        NN_RESULT_DO(Save());

        m_SubscriptionEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    if (m_Records[index].subscription == InternalSubscriptionStatus_Deleted || m_Records[index].subscription == InternalSubscriptionStatus_Unsubscribed)
    {
        // 未購読なら何もしない。
        NN_DETAIL_NEWS_INFO("[news] The topic is not subscribed. Do nothing.\n");
        NN_RESULT_SUCCESS;
    }
    if (std::memcmp(revisionHash.value, m_Records[index].revisionHash.value, sizeof (revisionHash.value)) == 0)
    {
        // リビジョンのハッシュ値が一致したら何もしない。
        NN_DETAIL_NEWS_INFO("[news] The revision hash is matched. Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    m_Records[index].revisionHash = revisionHash;

    // TORIAEZU: いきなり Runnable 状態にならないようにする。
    //           NotifyComplete に RevisionHash チェックを入れたら削除する。
    if (waitTime == 0)
    {
        waitTime = 1;
    }

    SetWait(index, waitTime);

    // TODO: NotifyComplete に RevisionHash チェックを入れたら削除する。
    NN_RESULT_DO(Save());

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifyDone(const Record& record, nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchRecord(record.topicId);

    NN_RESULT_THROW_UNLESS(index >= 0, ResultNotFound());

    if (result.IsSuccess() || ResultNoLongerSubscribed::Includes(result))
    {
        // TODO: RevisionHash を引数にとって比較する。
        if (m_Records[index].status == TaskStatus_Runnable || m_Records[index].status == TaskStatus_Running)
        {
            m_Records[index].status = TaskStatus_Done;
            NN_RESULT_DO(Save());
        }
    }
    else
    {
        if (IsServerTemporaryError(result))
        {
            // 通信エラーは 5 分後に再試行する。
            SetWait(index, 5 * 60);
        }
        else if (IsServerFailureError(result))
        {
            // サーバー障害は 12 時間後に再試行する。
            SetWait(index, 12 * 3600);
        }
        else if (ResultInternetRequestNotAccepted::Includes(result))
        {
            m_IsNetworkConnectionAvailable = false;

            // TORIAEZU: いきなり Runnable 状態にならないように 0 ではなく 1 を指定する。
            //           NotifyComplete に RevisionHash チェックを入れたら削除する。
            SetWait(index, 1);
        }
        else if (ResultSystemUpdateRequired::Includes(result))
        {
            m_Records[index].status = TaskStatus_SystemUpdateRequired;
            NN_RESULT_DO(Save());
        }
        else if (ResultNintendoAccountNotLinked::Includes(result))
        {
            m_Records[index].status = TaskStatus_NintendoAccountRequired;
            NN_RESULT_DO(Save());
        }
        else if (ResultServiceExpired::Includes(result))
        {
            m_Records[index].status = TaskStatus_Expired;
            NN_RESULT_DO(Save());
        }
        else
        {
            m_Records[index].status = TaskStatus_Error;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifyNpnsTopicSubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (nn::util::Strncmp(topicId.value, m_Records[i].topicId.value, sizeof (topicId.value)) == 0 &&
            m_Records[i].subscription == InternalSubscriptionStatus_Subscribed)
        {
            m_Records[i].isNpnsPerformed = true;

            if (result.IsSuccess())
            {
                m_Records[i].isNpnsRegistered = true;
                NN_RESULT_DO(Save());
            }
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifyNpnsTopicUnsubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (nn::util::Strncmp(topicId.value, m_Records[i].topicId.value, sizeof (topicId.value)) == 0 &&
            (m_Records[i].subscription == InternalSubscriptionStatus_Deleted || m_Records[i].subscription == InternalSubscriptionStatus_Unsubscribed))
        {
            m_Records[i].isNpnsPerformed = true;

            if (result.IsSuccess())
            {
                if (m_Records[i].subscription == InternalSubscriptionStatus_Deleted)
                {
                    std::memset(&m_Records[i], 0, sizeof (Record));
                    std::memset(&m_VolatileRecords[i], 0, sizeof (VolatileRecord));
                }
                else
                {
                    m_Records[i].isNpnsRegistered = false;
                }
                NN_RESULT_DO(Save());
            }
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifyNintendoAccountLinked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    bool isDirty = false;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].status == TaskStatus_NintendoAccountRequired)
        {
            SetRunnable(i);
            isDirty = true;
        }
    }

    if (isDirty)
    {
        NN_RESULT_DO(Save());
    }

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifyAccountCountryListUpdated() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    bool isDirty = false;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].status == TaskStatus_Done)
        {
            SetRunnable(i);
            isDirty = true;
        }
    }

    if (isDirty)
    {
        NN_RESULT_DO(Save());
    }

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::NotifySystemUpdated() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    bool isDirty = false;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].status == TaskStatus_SystemUpdateRequired)
        {
            SetRunnable(i);
            isDirty = true;
        }
    }

    if (isDirty)
    {
        NN_RESULT_DO(Save());
    }

    NN_RESULT_SUCCESS;
}

void NewsTaskManager::NotifySleepAwaked() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < TaskCountMax; i++)
    {
        // HTTP status 404 等のエラーが解消されているかもしれないので、スリープ復帰時に再試行する。
        if (m_Records[i].status == TaskStatus_Error)
        {
            SetRunnable(i);
        }

        if (!m_Records[i].isNpnsRegistered && m_Records[i].isNpnsPerformed &&
            m_Records[i].subscription == InternalSubscriptionStatus_Subscribed)
        {
            m_Records[i].isNpnsPerformed = false;
            m_SubscriptionEvent.Signal();
        }
        else if (m_Records[i].isNpnsRegistered && m_Records[i].isNpnsPerformed &&
            (m_Records[i].subscription == InternalSubscriptionStatus_Deleted || m_Records[i].subscription == InternalSubscriptionStatus_Unsubscribed))
        {
            m_Records[i].isNpnsPerformed = false;
            m_SubscriptionEvent.Signal();
        }
    }
}

void NewsTaskManager::NotifyNetworkConnected() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_IsNetworkConnectionAvailable)
    {
        m_IsNetworkConnectionAvailable = true;
        m_ScheduleEvent.Signal();
    }
}

nn::Result NewsTaskManager::LoadImpl() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (m_Records)));

    if (!Verify())
    {
        NN_DETAIL_NEWS_WARN("[news] %s is corrupted. (verification failed)\n", FilePath);
        NN_RESULT_THROW(ResultSaveVerificationFailed());
    }

    m_SubscriptionCount = 0;

    for (int i = 0; i < TaskCountMax; i++)
    {
        m_Records[i].isNpnsPerformed = false;

        if (m_Records[i].status == TaskStatus_Runnable)
        {
            m_ScheduleEvent.Signal();
        }
        else if (m_Records[i].status == TaskStatus_Wait || m_Records[i].status == TaskStatus_Error)
        {
            SetRunnable(i);
        }

        if (m_Records[i].subscription == InternalSubscriptionStatus_Subscribed)
        {
            m_SubscriptionCount++;
        }
    }

    if (IsNpnsSubscriptionProcessingRequired())
    {
        m_SubscriptionEvent.Signal();
    }

    NN_RESULT_SUCCESS;
}

nn::Result NewsTaskManager::Save() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

    NN_RESULT_TRY(nn::fs::DeleteFile(FilePath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(FileSystem::CreateFile(FilePath, sizeof (m_Records)));

    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (m_Records),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

bool NewsTaskManager::Verify() const NN_NOEXCEPT
{
    int runningCount = 0;

    for (int i = 0; i < TaskCountMax; i++)
    {
        const Record& record = m_Records[i];

        if (record.status != TaskStatus_Empty &&
            record.status != TaskStatus_Runnable &&
            record.status != TaskStatus_Running &&
            record.status != TaskStatus_SystemUpdateRequired &&
            record.status != TaskStatus_NintendoAccountRequired &&
            record.status != TaskStatus_Wait &&
            record.status != TaskStatus_Done &&
            record.status != TaskStatus_Error &&
            record.status != TaskStatus_Expired)
        {
            return false;
        }
        if (record.subscription != InternalSubscriptionStatus_Empty &&
            record.subscription != InternalSubscriptionStatus_Deleted &&
            record.subscription != InternalSubscriptionStatus_Unsubscribed &&
            record.subscription != InternalSubscriptionStatus_Subscribed)
        {
            return false;
        }
        if (record.topicId.value[0] != '\0' && !StringVerifier::VerifyTopicId(record.topicId.value))
        {
            return false;
        }
        if (record.eTag.value[0] != '\0' && !StringVerifier::VerifyETag(record.eTag.value))
        {
            return false;
        }
    }

    return (runningCount <= 1);
}

int NewsTaskManager::SearchRecord(const TopicId& topicId) const NN_NOEXCEPT
{
    for (int i = 0; i < TaskCountMax; i++)
    {
        const Record& record = m_Records[i];

        if ((record.subscription == InternalSubscriptionStatus_Unsubscribed || record.subscription == InternalSubscriptionStatus_Subscribed) &&
            nn::util::Strnicmp(topicId.value, record.topicId.value, sizeof (record.topicId.value)) == 0)
        {
            return i;
        }
    }

    return -1;
}

nn::Result NewsTaskManager::AddRecord(const TopicId& topicId, SubscriptionStatus status) NN_NOEXCEPT
{
    int index = -1;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].subscription == InternalSubscriptionStatus_Empty)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        // 空きがなかったら削除マーク付きのレコードに上書きする。
        for (int i = 0; i < TaskCountMax; i++)
        {
            if (m_Records[i].subscription == InternalSubscriptionStatus_Deleted)
            {
                index = i;
                break;
            }
        }
    }

    if (index == -1)
    {
        uint64_t minUnsubscribedIndex = UINT64_MAX;

        // 削除マーク付きのレコードもない場合、unsubscribedIndex が一番小さいレコードを選択する。
        for (int i = 0; i < TaskCountMax; i++)
        {
            if (m_Records[i].subscription == InternalSubscriptionStatus_Unsubscribed &&
                minUnsubscribedIndex > m_Records[i].unsubscribedIndex)
            {
                minUnsubscribedIndex = m_Records[i].unsubscribedIndex;
                index = i;
            }
        }

        NN_ABORT_UNLESS(index != -1);
    }

    Record record = {};

    bool isRunnable = false;

    record.topicId = topicId;

    if (status == SubscriptionStatus_Unsubscribed)
    {
        uint64_t id;
        NN_RESULT_DO(IncrementalId::GetInstance().Issue(&id));

        record.status = TaskStatus_Done;
        record.subscription = InternalSubscriptionStatus_Unsubscribed;
        record.unsubscribedIndex = id;
    }
    else
    {
        NN_RESULT_THROW_UNLESS(m_SubscriptionCount < SubscriptionCountMax, ResultSubscriptionLimitReached());

        record.subscription = InternalSubscriptionStatus_Subscribed;
        record.isAuto = (status == SubscriptionStatus_AutoSubscribed);

        m_SubscriptionCount++;

        m_SubscriptionEvent.Signal();

        isRunnable = true;
    }

    m_Records[index] = record;

    std::memset(&m_VolatileRecords[index], 0, sizeof (VolatileRecord));

    if (isRunnable)
    {
        SetRunnable(index);
    }

    NN_RESULT_SUCCESS;
}

void NewsTaskManager::SetRunnable(int index) NN_NOEXCEPT
{
    m_Records[index].status = TaskStatus_Runnable;

    m_ScheduleEvent.Signal();
}

void NewsTaskManager::SetRunnableWithoutSignal(int index) NN_NOEXCEPT
{
    m_Records[index].status = TaskStatus_Runnable;
}

void NewsTaskManager::SetWait(int index, int32_t waitTime) NN_NOEXCEPT
{
    if (waitTime == 0)
    {
        SetRunnable(index);
    }
    else
    {
        if (m_Records[index].status == TaskStatus_Wait)
        {
            nn::TimeSpan remainWaitTime = (m_VolatileRecords[index].runnableTick - nn::os::GetSystemTick()).ToTimeSpan();

            // 待機時間が短くならない場合、無視する。
            if (static_cast<int32_t>(remainWaitTime.GetSeconds()) <= waitTime)
            {
                return;
            }
        }
        else
        {
            m_Records[index].status = TaskStatus_Wait;
        }

        if (waitTime > 0)
        {
            m_VolatileRecords[index].runnableTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(waitTime));

            NN_DETAIL_NEWS_INFO("[news] Wait for %d seconds ...\n", waitTime);

            SetScheduleTimer();
        }
        else
        {
            m_VolatileRecords[index].runnableTick = RunnableTickMax;
        }
    }
}

void NewsTaskManager::SetScheduleTimer() NN_NOEXCEPT
{
    nn::TimeSpan interval;

    if (GetScheduleMinInterval(&interval))
    {
        if (interval > 0)
        {
            m_ScheduleEvent.StartOneShot(interval);
        }
        else
        {
            m_ScheduleEvent.Signal();
        }
    }
}

bool NewsTaskManager::GetScheduleMinInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT
{
    nn::os::Tick nextRunnableTick = RunnableTickMax;
    bool hasWaitTask = false;

    for (int i = 0; i < TaskCountMax; i++)
    {
        if (m_Records[i].status == TaskStatus_Wait && m_VolatileRecords[i].runnableTick < nextRunnableTick)
        {
            nextRunnableTick = m_VolatileRecords[i].runnableTick;
            hasWaitTask = true;
        }
    }

    if (!hasWaitTask)
    {
        return false;
    }

    *outInterval = (nextRunnableTick - nn::os::GetSystemTick()).ToTimeSpan();

    return true;
}

bool NewsTaskManager::FilterWithNone(InternalSubscriptionStatus status) NN_NOEXCEPT
{
    return (status == InternalSubscriptionStatus_Unsubscribed || status == InternalSubscriptionStatus_Subscribed);
}

bool NewsTaskManager::FilterWithUnsubscribed(InternalSubscriptionStatus status) NN_NOEXCEPT
{
    return (status == InternalSubscriptionStatus_Unsubscribed);
}

bool NewsTaskManager::FilterWithSubscribed(InternalSubscriptionStatus status) NN_NOEXCEPT
{
    return (status == InternalSubscriptionStatus_Subscribed);
}

bool NewsTaskManager::IsServerTemporaryError(nn::Result result) NN_NOEXCEPT
{
    if (nn::bcat::ResultHttpError::Includes(result))
    {
        return true;
    }

    return false;
}

bool NewsTaskManager::IsServerFailureError(nn::Result result) NN_NOEXCEPT
{
    // CDN の帯域制限の場合も 403 エラーが返る可能性がある。
    if (nn::bcat::ResultServerError403::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError500::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError502::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError503::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError504::Includes(result))
    {
        return true;
    }
    if (nn::bcat::ResultServerError509::Includes(result))
    {
        return true;
    }

    return false;
}

}}}}}
