﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_IncrementalId.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* g_FilePath = "news-sys:/id.bin";
}

IncrementalId::IncrementalId() NN_NOEXCEPT :
    m_Mutex(true),
    m_Base(0),
    m_Index(ReserveCount)
{
}

nn::Result IncrementalId::Issue(uint64_t* outId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outId);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_Index == ReserveCount)
    {
        NN_RESULT_TRY(ReserveRange())
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(ResultIdIssueFailed());
            }
        NN_RESULT_END_TRY;
    }

    *outId = m_Base + m_Index;
    m_Index++;

    NN_RESULT_SUCCESS;
}

nn::Result IncrementalId::ReserveRange() NN_NOEXCEPT
{
    NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();
    {
        nn::fs::FileHandle handle = {};
        bool isReadable = true;

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(FileSystem::CreateFile(g_FilePath, 8));
                NN_RESULT_DO(nn::fs::OpenFile(&handle, g_FilePath, nn::fs::OpenMode_Write));

                isReadable = false;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        uint64_t currentBase = 0;

        if (isReadable)
        {
            NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &currentBase, 8));
        }

        uint64_t nextBase = currentBase + ReserveCount;

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &nextBase, 8, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        m_Base = currentBase;
        m_Index = 0;
    }
    FileSystem::Commit(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

    NN_RESULT_SUCCESS;
}

}}}}}
