﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/ipc/news_IpcService.h>
#include <nn/news/detail/ipc/news_ServiceConfig.h>
#include <nn/news/detail/ipc/news_ServiceName.h>
#include <nn/news/detail/ipc/news_IServiceCreator.sfdl.h>
#include <nn/news/news_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sm/sm_Result.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <mutex>

#if defined (NN_BUILD_CONFIG_OS_WIN)
#include <nn/news/detail/service/news_ServiceCreator.generic.h>
#endif

namespace nn { namespace news { namespace detail { namespace ipc {

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

#if defined (NN_BUILD_CONFIG_OS_WIN)

    detail::service::ServiceCreatorForGeneric g_Creator;

#else

    nn::sf::ShimLibraryObjectHolder<IServiceCreator>
        g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::sf::SimpleAllInOneHipcSubDomainClientManager<ObjectCountPerProcess>
        g_Domain = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;

#endif
}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

#if !defined (NN_BUILD_CONFIG_OS_WIN)

    const char* serviceNames[] =
    {
        ServiceNameForPost,
        ServiceNameForManager,
        ServiceNameForViewer,
        ServiceNameForConfiguration,
        ServiceNameForAdministrator
    };

    bool isServiceFound = false;

    // 1 アプリに割り当てる権限は 1 つのみ。
    // 1 アプリが複数の権限を所持していた場合、最初に発見したもののみを利用する。
    for (auto name : serviceNames)
    {
        nn::Result result = g_Domain.InitializeShimLibraryHolder(&g_Holder, name);

        if (result.IsSuccess())
        {
            isServiceFound = true;
            break;
        }
        else if (!nn::sm::ResultNotPermitted::Includes(result))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    NN_ABORT_UNLESS(isServiceFound, "[news] Needs API permission.");

    g_Domain.GetClientSessionManager().SetSessionCount(SessionCountPerProcess);

#endif
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

#if !defined (NN_BUILD_CONFIG_OS_WIN)

    g_Holder.FinalizeHolder();

#endif
}

nn::Result CreateNewsService(INewsService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<INewsService> service = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateNewsService(&service));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateNewsService(&service));

#endif

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result CreateNewlyArrivedEventHolder(detail::ipc::INewlyArrivedEventHolder** outHolder) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<INewlyArrivedEventHolder> holder = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateNewlyArrivedEventHolder(&holder));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateNewlyArrivedEventHolder(&holder));

#endif

    *outHolder = holder.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result CreateOverwriteEventHolder(detail::ipc::IOverwriteEventHolder** outHolder) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<IOverwriteEventHolder> holder = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateOverwriteEventHolder(&holder));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateOverwriteEventHolder(&holder));

#endif

    *outHolder = holder.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result CreateNewsDataService(detail::ipc::INewsDataService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<INewsDataService> service = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateNewsDataService(&service));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateNewsDataService(&service));

#endif

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result CreateNewsDatabaseService(detail::ipc::INewsDatabaseService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::sf::SharedPointer<INewsDatabaseService> service = nullptr;

#if defined (NN_BUILD_CONFIG_OS_WIN)

    NN_RESULT_DO(g_Creator.CreateNewsDatabaseService(&service));

#else

    NN_RESULT_DO(g_Holder.GetObject()->CreateNewsDatabaseService(&service));

#endif

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

}}}}
