﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <atomic>

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/socket.h>
#include <curl/curl.h>
#include <nn/ssl/ssl_Context.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/netdiag/netdiag_Result.h>
#include <nn/netdiag/netdiag_ResultPrivate.h>
#include <nn/netdiag/netdiag_GlobalIpAddressTypes.h>
#include <nn/netdiag/detail/netdiag_CurlMulti.h>
#include <nn/netdiag/detail/netdiag_Log.h>

#include <cctype>

#include "netdiag_Util.h"

namespace nn { namespace netdiag { namespace detail {

namespace
{
    struct CurlWriteFunctionUserData
    {
        uint8_t* pWriteBuffer;
        size_t writePosition;
        size_t buffreSize;
    };

    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
    {
        CurlWriteFunctionUserData* userData = static_cast<CurlWriteFunctionUserData*>(userdata);

        auto count = blobsize * blobcount;
        if(userData->writePosition + count <= userData->buffreSize)
        {
            memcpy(&userData->pWriteBuffer[userData->writePosition], pData, count);
            userData->writePosition += count;
        }
        else
        {
            NN_DETAIL_NETDIAG_INFO("[netdiag] At CurlWriteFunction, download buffer was overflowed. bufferSize:%zu\n", userData->buffreSize);
            return 0; // 処理を中断します
        }

        return count;
    }

    size_t CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pUserData);

        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        if (context.Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
        {
            return static_cast<size_t>(- 1);
        }
        return 0;
    }
}

//----------------------------------------------------------------
// サーバからデータをダウンロード
// (サーバからは時刻データとグローバルIPアドレスを取得)
nn::Result DownloadFromServer( void* pDownloadBuffer, size_t downloadBufferSize ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pDownloadBuffer );
    NN_ABORT_UNLESS_GREATER_EQUAL( static_cast<int>(downloadBufferSize), 128 );

    const char *url = "https://aauth-%.ndas.srv.nintendo.net/v1/time";

    CurlWriteFunctionUserData userData = { static_cast<uint8_t*>(pDownloadBuffer), 0, downloadBufferSize };

    // curl セッションインスタンスの生成
    CURL* curl;
    NN_RESULT_DO(AcquireCurlHandle(&curl));
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);
    };

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, &userData);
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);

    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0L);
    curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);

    // TODO:タイムアウト時間要調整
    const long LowSpeedLimit = 1L; // 転送ビットレートは 1bps を下限とする
    const long LowSpeedTime = 30L; // LowSpeedLimit が 30 秒つづくと通信を中断する
    const long ConnectTimeout = 30L; // サーバとの接続(CONNECT)のタイムアウト (30秒)
    const long TimeoutSeconds = 60L; // 通信全体でのタイムアウト (1分)
    curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, LowSpeedLimit);
    curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, LowSpeedTime);
    curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, ConnectTimeout);
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, TimeoutSeconds);

    CURLcode curlCode;
    NN_RESULT_DO( PerformCurlMulti( curl, &curlCode ) );
    NN_RESULT_DO(HandleCurlError(curlCode));

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ダウンロードバッファから展開
nn::Result ExtractIpAddress( nn::netdiag::GlobalIpAddress* pGlobalIpAddress, uint8_t* pDownloadBuffer, size_t downloadBufferSize ) NN_NOEXCEPT
{
    // pDownloadBuffer には 時間情報 + 改行 + グローバルIPアドレス が格納されているはず。その後は終端文字
    uint8_t* p = pDownloadBuffer;
    for( ; isdigit(*p) && static_cast<size_t>(p - pDownloadBuffer) < downloadBufferSize; p++ ) // 数字をスキップ。文字列オーバーでも打ち切り
    {
    }
    // オーバーしている
    NN_RESULT_THROW_UNLESS( static_cast<size_t>(p - pDownloadBuffer) < downloadBufferSize, nn::netdiag::ResultInvalidDownloadFormat() );
    // 改行コードのはず
    NN_RESULT_THROW_UNLESS( *p == '\n', nn::netdiag::ResultInvalidDownloadFormat() );

    p++;
    uint8_t* pIpAddress = p;
    // IPアドレスを構成する文字のチェック。文字列オーバーでも打ち切り
    for( ; (isdigit(*p) || *p == '.') && static_cast<size_t>(p - pDownloadBuffer) < downloadBufferSize; p++ )
    {
    }
    // オーバーしている
    NN_RESULT_THROW_UNLESS( static_cast<size_t>(p - pDownloadBuffer) < downloadBufferSize, nn::netdiag::ResultInvalidDownloadFormat() );
    // 終端文字のはず
    NN_RESULT_THROW_UNLESS( *p == '\0', nn::netdiag::ResultInvalidDownloadFormat() );

    // IPアドレス部分の長さチェック
    size_t len = p - pIpAddress;
    NN_RESULT_THROW_UNLESS( 1<=len && len<=pGlobalIpAddress->Size - 1, nn::netdiag::ResultInvalidDownloadFormat() );

    // IPアドレス文字列をコピー
    nn::util::Strlcpy( reinterpret_cast<char*>(&pGlobalIpAddress->value), reinterpret_cast<char*>(pIpAddress), pGlobalIpAddress->Size );

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// グローバルIPアドレス取得
nn::Result GetGlobalIpAddress( GlobalIpAddress* pGlobalIpAddress ) NN_NOEXCEPT
{
    ClearInterruptCurlMulti(); // 中断フラグのクリア

    NN_ABORT_UNLESS_NOT_NULL( pGlobalIpAddress );
    uint8_t downloadBuffer[128]; // ダウンロードバッファ
    std::memset( downloadBuffer, 0, sizeof(downloadBuffer) );

    // ダウンロード
    NN_RESULT_DO( DownloadFromServer( downloadBuffer, sizeof(downloadBuffer) ) );

    // IP アドレス文字列だけを取得
    NN_RESULT_DO( ExtractIpAddress( pGlobalIpAddress, downloadBuffer, sizeof(downloadBuffer) ) );

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// グローバルIPアドレス取得の中断指示を出す
//
void InterruptGetGlobalIpAddress() NN_NOEXCEPT
{
    InterruptCurlMulti();
}

}}} // nn::netdiag::detail
