﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "mmnv_RequestManager.h"
#include "mmnv_Log.h"

#include <nn/fgm/fgm_Types.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <new>


namespace nn     {
namespace mmnv   {
namespace server {

nn::Result RequestManager::AcquireRequestId(std::int32_t* pRequestId, nn::mmnv::Module moduleId, nn::os::EventClearMode eventClearMode, Owner* owner)
NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    for (int i = 0; i < MaxNumRequestSet; i++)
    {
        if (m_RequestSet[i].IsFree())
        {
            result = m_RequestSet[i].FgmRequest().Initialize(
                                static_cast<nn::fgm::Module>(moduleId),
                                nn::fgm::Priority_Lowest,
                                eventClearMode);

            if (result.IsSuccess())
            {
                m_RequestSet[i].SetOwner(owner);
                *pRequestId = i;
            }
            return result;
        }
    }

    return nn::os::ResultInternalError();
}

nn::Result RequestManager::ReleaseRequestId(std::int32_t requestId)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(requestId >= 0 && requestId < MaxNumRequestSet, "requestId %d\n", requestId);
    NN_ABORT_UNLESS(!m_RequestSet[requestId].IsFree());
    m_RequestSet[requestId].FgmRequest().Finalize();
    m_RequestSet[requestId].Clear();
    return ResultSuccess();
}

nn::Result RequestManager::ReleaseOwnRequestIds(Owner* owner)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(owner != 0);

    for (int i = 0; i < MaxNumRequestSet; i++)
    {
        if (m_RequestSet[i].IsFree())
        {
            continue;
        }

        if (m_RequestSet[i].GetOwner() == owner)
        {
            (void)ReleaseRequestId(i);
        }
    }
    return ResultSuccess();
}

nn::Result RequestManager::SetAndWait(std::int32_t requestId, std::uint32_t min, std::uint32_t max)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(requestId >= 0 && requestId < MaxNumRequestSet, "requestId %d\n", requestId);
    NN_ABORT_UNLESS(!m_RequestSet[requestId].IsFree());
    return m_RequestSet[requestId].FgmRequest().SetAndWait(min, max);
}

nn::Result RequestManager::Get(std::uint32_t* pSetting, std::int32_t requestId)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(requestId >= 0 && requestId < MaxNumRequestSet, "requestId %d\n", requestId);
    NN_ABORT_UNLESS(!m_RequestSet[requestId].IsFree());
    return m_RequestSet[requestId].FgmRequest().Get(pSetting);
}

RequestManager::RequestManager()
NN_NOEXCEPT
{
    for (int i = 0; i < MaxNumRequestSet; i++)
    {
        new (&m_RequestSet[i]) RequestSet();
    }

    for (int i = 0; i < MaxNumRequsetSetForCompatibility; i++)
    {
        new (&m_RequestSetForCompatibility[i]) RequestSetOld();
    }
}

// for compatibility with NX Addon 0.12.x
nn::Result RequestManager::InitializeRequest(nn::mmnv::Module moduleId, nn::os::EventClearMode eventClearMode, Owner* owner)
NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    for (int i = 0; i < MaxNumRequsetSetForCompatibility; i++)
    {
        if (m_RequestSetForCompatibility[i].IsFree())
        {
            result = m_RequestSetForCompatibility[i].FgmRequest().Initialize(
                                static_cast<nn::fgm::Module>(moduleId),
                                nn::fgm::Priority_Lowest,
                                eventClearMode);

            if (result.IsSuccess())
            {
                m_RequestSetForCompatibility[i].SetOwner(owner);
                m_RequestSetForCompatibility[i].SetModuleId(moduleId);
            }

            return result;
        }
    }

    return nn::os::ResultInternalError();
}

nn::Result RequestManager::FinalizeRequest(nn::mmnv::Module moduleId)
NN_NOEXCEPT
{
    for (int i = 0; i < MaxNumRequsetSetForCompatibility; i++)
    {
        if (m_RequestSetForCompatibility[i].IsFree() || !m_RequestSetForCompatibility[i].IsSameModule(moduleId))
        {
            continue;
        }
        m_RequestSetForCompatibility[i].FgmRequest().Finalize();
        m_RequestSetForCompatibility[i].Clear();
        // 同一 moduleId のものを一括で解放してしまう
        //（ロンチではこのままリリースされたため、あえて修正せずにそのままにしている）
    }

    return ResultSuccess();
}

nn::Result RequestManager::ReleaseOwnRequestIdsForCompatibility(Owner* owner)
NN_NOEXCEPT
{
    for (int i = 0; i < MaxNumRequsetSetForCompatibility; i++)
    {
        if (m_RequestSetForCompatibility[i].IsFree())
        {
            continue;
        }

        if (m_RequestSetForCompatibility[i].GetOwner() == owner)
        {
            m_RequestSetForCompatibility[i].FgmRequest().Finalize();
            m_RequestSetForCompatibility[i].Clear();
        }
    }

    return ResultSuccess();
}

nn::Result RequestManager::SetAndWait(nn::mmnv::Module moduleId, std::uint32_t min, std::uint32_t max)
NN_NOEXCEPT
{
    for (int i = 0; i < MaxNumRequsetSetForCompatibility; i++)
    {
        if (m_RequestSetForCompatibility[i].IsFree() || !m_RequestSetForCompatibility[i].IsSameModule(moduleId))
        {
            continue;
        }

        return m_RequestSetForCompatibility[i].FgmRequest().SetAndWait(min, max);
    }
    return nn::os::ResultInternalError();
}

nn::Result RequestManager::Get(std::uint32_t* pSetting, nn::mmnv::Module moduleId)
NN_NOEXCEPT
{
    for (int i = 0; i < MaxNumRequsetSetForCompatibility; i++)
    {
        if (m_RequestSetForCompatibility[i].IsFree() || !m_RequestSetForCompatibility[i].IsSameModule(moduleId))
        {
            continue;
        }

        return m_RequestSetForCompatibility[i].FgmRequest().Get(pSetting);
    }
    return nn::os::ResultInternalError();
}

}}}
