﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mii/detail/service/mii_ImageDatabaseServiceInterface.h>

#include <nn/mii/detail/mii_ImageDatabaseImpl.h>
#include <nn/mii/detail/service/mii_ServiceProvider.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/mii_PrivateResult.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn
{
namespace mii
{
namespace detail
{

nn::Result ImageDatabaseServiceInterface::CheckDatabaseTestModeEnabled() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_HORIZON)
    bool isEnabled = false;
    auto outBytes = nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "mii", "is_img_db_test_mode_enabled");
    NN_SDK_ASSERT_EQUAL(outBytes, sizeof(isEnabled));
    NN_UNUSED(outBytes);
    NN_RESULT_THROW_UNLESS(isEnabled, ResultDevelopmentOnly());
#else
    // HORIZON 以外では常にテストモード有効
#endif

    NN_RESULT_SUCCESS;
}

ImageDatabaseServiceInterface::ImageDatabaseServiceInterface(ServiceProvider* pServiceProvider) NN_NOEXCEPT
    : m_pImageDatabaseImpl(&pServiceProvider->m_ImageDatabaseImpl)
{
    NN_SDK_ASSERT_NOT_NULL(pServiceProvider);
}

ImageDatabaseServiceInterface::~ImageDatabaseServiceInterface() NN_NOEXCEPT
{
}

nn::Result ImageDatabaseServiceInterface::Initialize(nn::sf::Out<int8_t> outDatabaseResult, bool enableFormat) NN_NOEXCEPT
{
    if (!m_pImageDatabaseImpl->IsInitialized())
    {
        nn::mii::DatabaseResult databaseResult;
        const nn::Result result = m_pImageDatabaseImpl->Initialize(
            &databaseResult, enableFormat);
        outDatabaseResult.Set(static_cast<int8_t>(databaseResult));

        return result;
    }

    outDatabaseResult.Set(static_cast<int8_t>(nn::mii::DatabaseResult_Success));
    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::Reload() NN_NOEXCEPT
{
    return m_pImageDatabaseImpl->Reload();
}

nn::Result ImageDatabaseServiceInterface::GetCount(nn::sf::Out<int32_t> outValue) NN_NOEXCEPT
{
    outValue.Set(m_pImageDatabaseImpl->GetCount());

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::IsEmpty(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    outValue.Set(m_pImageDatabaseImpl->IsEmpty());

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::IsFull(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    outValue.Set(m_pImageDatabaseImpl->IsFull());

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::GetAttribute(nn::sf::Out<nn::mii::detail::ImageAttributeImpl> outAttribute, int32_t index) NN_NOEXCEPT
{
    int32_t count = 0;
    GetCount(&count);
    NN_RESULT_THROW_UNLESS(0 <= index && index < count, nn::mii::ResultInvalid());
    NN_UNUSED(count);
    m_pImageDatabaseImpl->GetAttribute(outAttribute.GetPointer(), index);

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::LoadImage(
    nn::sf::OutBuffer outBuffer, const nn::util::Uuid& uuid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uuid != nn::util::InvalidUuid, nn::mii::ResultInvalid());
    return m_pImageDatabaseImpl->LoadImage(
        outBuffer.GetPointerUnsafe(), outBuffer.GetSize(), uuid);
}

nn::Result ImageDatabaseServiceInterface::AddOrUpdateImage(
    nn::sf::Out<int32_t> outIndex,
    int8_t expression, // enum を int8_t で
    int8_t pose,
    int8_t fontRegion,
    nn::mii::Nickname nickname,
    nn::mii::CreateId createId,
    nn::util::Uuid usedUuid, // nullptrだと InvalidUuid を入れる
    nn::sf::InArray<nn::util::Uuid> validUuidArray,
    nn::sf::InBuffer srcBuffer
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(0 <= expression && expression < nn::mii::Expression_End, nn::mii::ResultInvalid());
    NN_RESULT_THROW_UNLESS(0 <= pose && pose < nn::mii::ImagePoseType_End, nn::mii::ResultInvalid());
    NN_RESULT_THROW_UNLESS(0 <= fontRegion && fontRegion < FontRegion_End, nn::mii::ResultInvalid());
    NN_RESULT_THROW_UNLESS(nickname.IsValid(), nn::mii::ResultInvalid());
    NN_RESULT_THROW_UNLESS(createId.IsValid(), nn::mii::ResultInvalid());

    return m_pImageDatabaseImpl->AddOrUpdateImage(
        outIndex.GetPointer(),
        static_cast<nn::mii::Expression>(expression),
        static_cast<nn::mii::ImagePoseType>(pose),
        static_cast<nn::mii::FontRegion>(fontRegion),
        nickname, createId, usedUuid,
        validUuidArray.GetData(),
        static_cast<int>(validUuidArray.GetLength()),
        srcBuffer.GetPointerUnsafe(),
        static_cast<int>(srcBuffer.GetSize())
        );
}

nn::Result ImageDatabaseServiceInterface::DeleteImages(nn::sf::InArray<nn::util::Uuid> uuids) NN_NOEXCEPT
{
    return m_pImageDatabaseImpl->DeleteImages(
        uuids.GetData(), static_cast<int>(uuids.GetLength()));
}

nn::Result ImageDatabaseServiceInterface::DeleteFile() NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    nn::mii::detail::ImageDatabaseImpl::DeleteFile();

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::DestroyFile() NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    nn::mii::detail::ImageDatabaseImpl::DestroyFile();

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::ImportFile(nn::sf::InBuffer inBuffer) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    nn::mii::detail::ImageDatabaseImpl::ImportFile(
        inBuffer.GetPointerUnsafe(), inBuffer.GetSize());

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::ExportFile(
    nn::sf::Out<int64_t> outSize, nn::sf::OutBuffer outBuffer) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    outSize.Set(nn::mii::detail::ImageDatabaseImpl::ExportFile(
            outBuffer.GetPointerUnsafe(), outBuffer.GetSize()));

    NN_RESULT_SUCCESS;
}

nn::Result ImageDatabaseServiceInterface::ForceInitialize(nn::sf::Out<int8_t> outDatabaseResult, bool enableFormat) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckDatabaseTestModeEnabled());

    // 強制初期化用の関数を呼ぶ
    nn::mii::DatabaseResult databaseResult;
    const nn::Result result = m_pImageDatabaseImpl->ForceInitialize(
        &databaseResult, enableFormat);
    outDatabaseResult.Set(static_cast<int8_t>(databaseResult));

    return result;
}

}
}
}
