﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <utility>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/time.h>
#include "lm_LogBuffer.h"
#include "lm_LogServerProxy.h"
#include "lm_SdCardLogger.h"
#include "lm_LogPacketParser.h"

namespace nn { namespace lm { namespace impl {

namespace {
    void UpdateUserSystemClock(const uint8_t* message, size_t messageSize) NN_NOEXCEPT
    {
        time::PosixTime currentTime;

        if (time::StandardUserSystemClock::GetCurrentTime(&currentTime).IsFailure())
        {
            NN_SDK_ASSERT(false);
            return;
        }
        auto baseTime = currentTime.value - nn::os::GetSystemTick().ToTimeSpan().GetSeconds();
        LogPacketParser::ParsePacket(message, messageSize,
            [](const detail::LogPacketHeader& header, const void* pPayload, size_t payloadSize, void* argument) -> bool
            {
                auto pBaseTime = reinterpret_cast<int64_t*>(argument);
                if (!header.IsHead())
                {
                    // 先頭パケット以外は、データチャンクが分割されていて、
                    // パースできない可能性があるため、データチャンクのパースは行わずに読み飛ばす。
                    return true;
                }
                return LogPacketParser::ParseDataChunk(pPayload, payloadSize,
                    [](detail::LogDataChunkKey key, const void* pChunk, size_t chunkSize, void* argument) -> bool
                    {
                        auto pBaseTime = reinterpret_cast<int64_t*>(argument);
                        if (key == detail::LogDataChunkKey_UserSystemClock)
                        {
                            int64_t time;
                            NN_SDK_ASSERT_EQUAL(chunkSize, sizeof(time));
                            std::memcpy(&time, pChunk, chunkSize);
                            time += *pBaseTime;
                            std::memcpy(const_cast<void*>(pChunk), &time, sizeof(time));
                        }
                        return true;
                    }, pBaseTime);
            }, &baseTime);
    }

    bool DefaultFlushFunction(const uint8_t* message, size_t messageSize) NN_NOEXCEPT
    {
        // フラッシュ関数が失敗したときは、バッファの中身が残る。
        // 同じ中身に対して、UserSystemClock を繰り返し更新しないようにする。
        // LogBuffer::Flush が排他されているため、s_IsUserSystemClockUpdated のアクセスは排他不要。
        NN_FUNCTION_LOCAL_STATIC(bool, s_IsUserSystemClockUpdated, = false);

        if (!s_IsUserSystemClockUpdated)
        {
            UpdateUserSystemClock(message, messageSize);
            s_IsUserSystemClockUpdated = true;
        }

#if defined(NN_DETAIL_LM_WITHOUT_TMA)
        const auto isSuccess = SdCardLogger::GetInstance().Write(message, messageSize);
#else
        const auto isHostLogSuccess = LogServerProxy::GetInstance().Send(message, messageSize);
        const auto isSdLogSuccess = SdCardLogger::GetInstance().Write(message, messageSize);
        const auto isSuccess = isHostLogSuccess || isSdLogSuccess;
#endif

        if (isSuccess)
        {
            s_IsUserSystemClockUpdated = false;
        }

        return isSuccess;
    }
}

LogBuffer::LogBuffer(void* buffer, size_t bufferSize, FlushFunction function) NN_NOEXCEPT :
    m_pPushBuffer(&m_Buffers[0]),
    m_pFlushBuffer(&m_Buffers[1]),
    m_BufferSize(bufferSize / 2),
    m_FlushFunction(function),
    m_PushBufferMutex(false),
    m_FlushBufferMutex(false),
    m_PushCanceled(false),
    m_PushReadyWaitCount(0u)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(m_BufferSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(function);

    m_Buffers[0].m_pHead = reinterpret_cast<uint8_t*>(buffer);
    m_Buffers[1].m_pHead = reinterpret_cast<uint8_t*>(buffer) + (bufferSize / 2);
}

LogBuffer& LogBuffer::GetDefaultInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(std::aligned_storage<(128 * 1024) * 2>::type, storage);
    NN_FUNCTION_LOCAL_STATIC(LogBuffer, instance, (&storage, sizeof(storage), DefaultFlushFunction));
    return instance;
}

// @brief       バッファにログを格納します。
// @detail      バッファに空きがない場合、空きができて格納できるまでブロックします。
//              @ref CancelPush によって中断されたときは、false を返します。
bool LogBuffer::Push(const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    return PushImpl(pSrc, srcSize, true);
}

// @brief       バッファにログを格納します。
// @detail      格納に成功した場合は true を返します。
//              バッファに空きがない場合は失敗して false を返します。
bool LogBuffer::TryPush(const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    return PushImpl(pSrc, srcSize, false);
}

bool LogBuffer::PushImpl(const void* pSrc, size_t srcSize, bool blocking) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(srcSize, m_BufferSize);
    NN_SDK_REQUIRES(pSrc || srcSize == 0u);

    if (srcSize == 0u)
    {
        return true;
    }

    uint8_t* pDst;
    {
        std::lock_guard<os::Mutex> pushLock(m_PushBufferMutex);

        while (!(srcSize <= m_BufferSize - m_pPushBuffer->m_StoredSize))
        {
            if (!blocking)
            {
                return false;
            }

            m_PushReadyWaitCount++;

            m_ConditionPushReady.Wait(m_PushBufferMutex);

            m_PushReadyWaitCount--;

            if (m_PushCanceled)
            {
                if (m_PushReadyWaitCount == 0u)
                {
                    m_PushCanceled = false;
                }

                return false;
            }
        }

        pDst = m_pPushBuffer->m_pHead + m_pPushBuffer->m_StoredSize;

        m_pPushBuffer->m_StoredSize += srcSize;
        m_pPushBuffer->m_ReferenceCount++;
    }

    std::memcpy(pDst, pSrc, srcSize);

    {
        std::lock_guard<os::Mutex> pushLock(m_PushBufferMutex);

        m_pPushBuffer->m_ReferenceCount--;

        if (m_pPushBuffer->m_ReferenceCount == 0u)
        {
            m_ConditionFlushReady.Signal();
        }
    }

    return true;
}

// @brief       バッファの空きを待っている @ref Push を中断させます。
void LogBuffer::CancelPush() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> pushLock(m_PushBufferMutex);

    if (m_PushReadyWaitCount > 0u)
    {
        m_PushCanceled = true;
        m_ConditionPushReady.Broadcast();
    }
}

// @brief       バッファリングされたログをフラッシュします。
// @detail      フラッシュに成功した場合は true を返します。
//              バッファにログがない場合、ログが書き込まれてフラッシュが完了するまでブロックします。
bool LogBuffer::Flush() NN_NOEXCEPT
{
    return FlushImpl(true);
}

// @brief       バッファリングされたログをフラッシュします。
// @detail      フラッシュに成功した場合は true を返します。
//              バッファにログがないか、ログの書き込み中でロックが取れない場合は、
//              失敗して false を返します。
bool LogBuffer::TryFlush() NN_NOEXCEPT
{
    return FlushImpl(false);
}

bool LogBuffer::FlushImpl(bool blocking) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> flushLock(m_FlushBufferMutex);

    if (!(m_pFlushBuffer->m_StoredSize > 0u))
    {
        std::lock_guard<os::Mutex> pushLock(m_PushBufferMutex);

        while (!(m_pPushBuffer->m_StoredSize > 0u && m_pPushBuffer->m_ReferenceCount == 0u))
        {
            if (!blocking)
            {
                return false;
            }

            m_ConditionFlushReady.Wait(m_PushBufferMutex);
        }

        std::swap(m_pPushBuffer, m_pFlushBuffer);

        m_ConditionPushReady.Broadcast();
    }

    if (!m_FlushFunction(m_pFlushBuffer->m_pHead, m_pFlushBuffer->m_StoredSize))
    {
        return false;
    }

    m_pFlushBuffer->m_StoredSize = 0u;

    return true;
}

}}} // nn::lm::impl
