﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Thread.h>
#include <nn/TargetConfigs/build_Base.h>
#include "lm_LogPacketTransmitterBase.h"

namespace nn { namespace lm { namespace detail {

bool LogPacketTransmitterBase::Flush(bool isTail) NN_NOEXCEPT
{
    if (m_pCurrent == m_pPayload)
    {
        return true;
    }

    m_pHeader->SetTail(isTail);
    m_pHeader->SetPayloadSize(static_cast<uint32_t>(m_pCurrent - m_pPayload));
    auto flushResult = m_FlushFunction(m_pBegin, static_cast<size_t>(m_pCurrent - m_pBegin));
    m_pHeader->SetHead(false);
    m_pCurrent = m_pPayload;

    return flushResult;
}

/**
*   @brief      ログパケットに整数を ULEB128 形式で格納します。
*
*   @details    ULEB128（Unsigned Little Endian Base 128） は、可変長整数のデータ形式で、
*               小さな整数を少ないバイト数で表すことができます。
*               構造化ログでは、データチャンクのキーとサイズをエンコードする際に利用しています。
*
*               ULEB 128 は、各バイトの最上位ビットをバイト幅を延長するかどうかのフラグとして利用し、
*               下位 7 bit に実際の値を格納します。
*               符号無し整数を、下位桁から順に格納し、128（7 bit）の乗数ごとにバイト幅を延長することが名前の由来です。
*
*               詳しくは WEB を検索してください。
*/
void LogPacketTransmitterBase::PushUleb128(unsigned int value) NN_NOEXCEPT
{
    const unsigned int valueBitMask = 0x7F;
    const unsigned int valueBitCount = 7;

    do {
        if (m_pCurrent == m_pEnd)
        {
            Flush(false);
        }
        const uint8_t msbMask = (value & (~valueBitMask)) != 0 ? 0x80 : 0x00;
        *m_pCurrent = static_cast<uint8_t>(value & valueBitMask) | msbMask;
        value >>= valueBitCount;
        m_pCurrent++;
    } while (value > 0);
}

void LogPacketTransmitterBase::Push(const void* data, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);

    auto pSrc = reinterpret_cast<const uint8_t*>(data);
    auto rest = size;
    do {
        if (m_pCurrent == m_pEnd)
        {
            Flush(false);
        }
        auto copySize = std::min(rest, static_cast<size_t>(m_pEnd - m_pCurrent));
        std::memcpy(m_pCurrent, pSrc, copySize);
        rest -= copySize;
        pSrc += copySize;
        m_pCurrent += copySize;
    } while (rest > 0);
}

void LogPacketTransmitterBase::PushDataChunk(
    LogDataChunkKey key, const void* data, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);

    PushUleb128(key);
    PushUleb128(static_cast<unsigned int>(size));
    Push(data, size);
}

LogPacketTransmitterBase::LogPacketTransmitterBase(
    void*           buffer,
    size_t          bufferSize,
    FlushFunction   function,
    uint8_t         severity,
    uint8_t         verbosity,
    uint64_t        processId,
    bool            isHead,
    bool            isTail) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_ASSERT_ALIGNED(buffer, NN_ALIGNOF(LogPacketHeader));
    NN_SDK_REQUIRES_GREATER(bufferSize, LogPacketHeaderSize);
    NN_SDK_REQUIRES_NOT_NULL(function);

    m_pHeader = new (buffer) LogPacketHeader;

    m_pBegin    = static_cast<uint8_t*>(buffer);
    m_pEnd      = m_pBegin + bufferSize;
    m_pPayload  = m_pBegin + LogPacketHeaderSize;
    m_pCurrent  = m_pPayload;

    m_IsTail = isTail;

    m_FlushFunction = function;

    m_pHeader->SetProcessId(processId);
    m_pHeader->SetThreadId(os::GetThreadId(os::GetCurrentThread()));
    m_pHeader->SetHead(isHead);
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    m_pHeader->SetLittleEndian(true);
#else
    m_pHeader->SetLittleEndian(false);
#endif
    m_pHeader->SetSeverity(severity);
    m_pHeader->SetVerbosity(verbosity);
}

LogPacketTransmitterBase::~LogPacketTransmitterBase() NN_NOEXCEPT
{
    Flush(m_IsTail);
}

}}} // nn::lm::detail
