﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/lbl/lbl_Type.h>

#include "lbl_Fader.h"

namespace nn { namespace lbl { namespace impl { namespace detail {

Fader::Fader() NN_NOEXCEPT
{
}

void Fader::StartFade(bool isOn, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    if ( 0 == fadeTime )
    {
        m_State = isOn ?
            nn::lbl::BacklightSwitchStatus_OnWithStability :
            nn::lbl::BacklightSwitchStatus_OffWithStability;
        m_LastRawBrightnessFadeRatio = isOn ? 1.0f : 0.0f;
    }
    else
    {
        m_LastUpdatedTime = nn::os::GetSystemTick().ToTimeSpan();
        m_AssignedFadeTime = fadeTime;
        m_State = isOn ?
            nn::lbl::BacklightSwitchStatus_OffInTransitionToOn :
            nn::lbl::BacklightSwitchStatus_OnInTransitionToOff;
    }
}

float Fader::UpdateActiveBrightnessRatio() NN_NOEXCEPT
{
    if ( nn::lbl::BacklightSwitchStatus_OnWithStability == m_State )
    {
        return 1.0f;
    }
    else if ( nn::lbl::BacklightSwitchStatus_OffWithStability == m_State )
    {
        return 0.0f;
    }

    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    auto now = nn::os::GetSystemTick().ToTimeSpan();
    auto duration = now - m_LastUpdatedTime;
    m_LastUpdatedTime = now;
    auto variation = static_cast<float>(duration.GetMicroSeconds()) / static_cast<float>(m_AssignedFadeTime.GetMicroSeconds());
    if ( nn::lbl::BacklightSwitchStatus_OffInTransitionToOn == m_State )
    {
        m_LastRawBrightnessFadeRatio += variation;
        if ( 1.0f <= m_LastRawBrightnessFadeRatio )
        {
            // 点灯へ状態遷移が完了。
            m_State = nn::lbl::BacklightSwitchStatus_OnWithStability;
            m_LastRawBrightnessFadeRatio = 1.0f;
        }
    }
    else if ( nn::lbl::BacklightSwitchStatus_OnInTransitionToOff == m_State )
    {
        m_LastRawBrightnessFadeRatio -= variation;
        if ( 0.0f >= m_LastRawBrightnessFadeRatio )
        {
            // 消灯へ状態遷移が完了。
            m_State = nn::lbl::BacklightSwitchStatus_OffWithStability;
            m_LastRawBrightnessFadeRatio = 0.0f;
        }
    }
    return m_LastRawBrightnessFadeRatio;
}

}}}} // namespace nn::lbl::impl::detail
