﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cmath>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/lbl/lbl_Type.h>

#include "lbl_BrightnessController.h"

namespace nn { namespace lbl { namespace impl { namespace detail {

BrightnessController::BrightnessController(const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
    : m_pSettingsHolder(pSettingsHolder)
{
}

void BrightnessController::Initialize() NN_NOEXCEPT
{
    m_DeviceAccessorBacklight.Initialize();

    // 初期化時は消灯の状態にする。
    m_Fader.StartFade(false, 0);
}

void BrightnessController::Finalize() NN_NOEXCEPT
{
    m_Fader.StartFade(false, 0);
    m_DeviceAccessorBacklight.Finalize();
}

void BrightnessController::StartFade(bool isOn, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    if ( isOn )
    {
        // 点灯開始
        if ( nn::lbl::BacklightSwitchStatus_OnWithStability != m_Fader.GetState() )
        {
            m_Fader.StartFade(true, fadeTime);
        }
    }
    else
    {
        // 消灯開始
        if ( nn::lbl::BacklightSwitchStatus_OffWithStability != m_Fader.GetState() )
        {
            m_Fader.StartFade(false, fadeTime);
        }
    }
}

double BrightnessController::CalcVrModeBrightness() const NN_NOEXCEPT
{
    const double baseBrightness
        = static_cast<double>(m_pSettingsHolder->GetBrightnessForVrMode())
        * static_cast<double>(m_pSettingsHolder->GetVrBrightnessScale());
    return std::sqrt( baseBrightness * baseBrightness + m_pSettingsHolder->GetVrLowerBrightness() );
}

double BrightnessController::UpdateBrightnessTransitionRatio() NN_NOEXCEPT
{
    // 1フレームあたりの明るさの変化量
    const double ratioChange = 1.0 / m_pSettingsHolder->GetVrBrightnessTransitionFrame();

    const auto nextFrameBlendRatio
        = m_IsInVrMode ? m_VrBrightnessTransitionRatio + ratioChange
                       : m_VrBrightnessTransitionRatio - ratioChange;

    // 0.0 ~ 1.0 の範囲への丸め
    m_VrBrightnessTransitionRatio = nextFrameBlendRatio < 0.0 ? 0.0
                                  : nextFrameBlendRatio > 1.0 ? 1.0
                                  : nextFrameBlendRatio;
    return m_VrBrightnessTransitionRatio;
}

void BrightnessController::UpdateAndApplyBrightness(float illuminanceLux) NN_NOEXCEPT
{
    float normalModeBrightness;
    if ( IsAutoAdjustEnabled() )
    {
        normalModeBrightness = m_AutoBrightnessAdjuster.UpdateForAutoBrightness(illuminanceLux, m_pSettingsHolder->GetBrightness(), m_ApplyImmediatelyNext);
        m_ApplyImmediatelyNext = false;
    }
    else
    {
        normalModeBrightness = m_AutoBrightnessAdjuster.UpdateForManualBrightness(m_pSettingsHolder->GetBrightness());
    }

    const double vrModeBrightness  = CalcVrModeBrightness();
    const double vrBrightnessRatio = UpdateBrightnessTransitionRatio();
    m_AppliedBrightness = static_cast<float>(
        vrBrightnessRatio * vrModeBrightness
        + ( 1.0 - vrBrightnessRatio ) * normalModeBrightness );

    // 低輝度時、既に低輝度時の輝度を下回っていた場合には輝度を丸めない。
    if ( m_IsDimmingEnabled ) {
        auto dimmingBrightness = m_pSettingsHolder->GetDimmingBrightness();
        if (m_AppliedBrightness > dimmingBrightness) {
            m_AppliedBrightness = dimmingBrightness;
        }
    }

    auto latestActiveRatio = m_Fader.UpdateActiveBrightnessRatio();
    if ( nn::lbl::BacklightSwitchStatus_OffWithStability == m_Fader.GetState() )
    {
        m_DeviceAccessorBacklight.SetDuty(0);
        m_DeviceAccessorBacklight.SetPower(false);
    }
    else
    {
        m_DeviceAccessorBacklight.SetPower(true);
        m_DeviceAccessorBacklight.SetDuty(TransformBrightnessToDuty(m_AppliedBrightness, latestActiveRatio));
    }
}

/**
* @brief   一度だけ UpdateAutoLight の輝度更新で以前に算出した輝度値を考慮せず、
*          即座に照度センサー値と現在の輝度設定値が反映されるようにします。
*/
void BrightnessController::ApplyImmediatelyNext() NN_NOEXCEPT
{
    m_ApplyImmediatelyNext = true;
}

/**
* @brief   輝度設定値をPWMのduty比に変換する写像。
*          歴代の輝度変換で使われている式(2次関数)を流用。
*/
int BrightnessController::TransformBrightnessToDuty(double rawBrightness, double activeRatio) NN_NOEXCEPT
{
    // 飽和演算のために double で値を扱う。
    auto brightness = rawBrightness * activeRatio;
    const auto MaxDuty = DeviceAccessorBacklight::MaxDuty;
    int duty = static_cast<int>(static_cast<double>(MaxDuty) * (
        m_pSettingsHolder->GetBrightnessCoefficients().v[0] * brightness * brightness +
        m_pSettingsHolder->GetBrightnessCoefficients().v[1] * brightness +
        m_pSettingsHolder->GetBrightnessCoefficients().v[2] * activeRatio) + 0.5f);
    // Pwmライブラリが受け付ける [0..255] の範囲に収める。
    if ( duty < 0 )
    {
        return 0;
    }
    else if ( MaxDuty < duty )
    {
        return MaxDuty;
    }
    return duty;
}

bool BrightnessController::IsAutoAdjustEnabled() const NN_NOEXCEPT
{
    return m_pSettingsHolder->IsAlsEnabled() && m_pSettingsHolder->IsAutoBrightnessEnabled();
}

}}}} // namespace nn::lbl::impl::detail
