﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "jitsrv_DllPlugin.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/jit/jit_Result.h>

#include <nn/ro.h>
#include <nn/os.h>
#include <memory>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>

namespace nn { namespace jitsrv {

Result DllPlugin::Initialize(const void* nrrData, size_t nrrSize, const void* nroData, size_t nroSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_State == State::Initial);
    NN_SDK_REQUIRES_ALIGNED(buffer, os::MemoryPageSize);
    NN_SDK_REQUIRES_ALIGNED(bufferSize, os::MemoryPageSize);

    auto success = false;

    auto offset = static_cast<size_t>(0);
    auto rest = [&]()
    {
        return bufferSize - offset;
    };
    auto allocateBuffer = [&](size_t size)
    {
        NN_SDK_REQUIRES_ALIGNED(size, os::MemoryPageSize);
        NN_SDK_ASSERT(size <= rest());
        auto ret = static_cast<char*>(buffer) + offset;
        offset += size;
        NN_SDK_ASSERT(offset <= bufferSize);
        return ret;
    };

    auto pageAlignedNrrSize = util::align_up(nrrSize, os::MemoryPageSize);
    NN_RESULT_THROW_UNLESS(pageAlignedNrrSize <= rest(), jit::ResultOutOfMemoryForDllLoad());

    auto nrrBuffer = allocateBuffer(pageAlignedNrrSize);
    std::memcpy(nrrBuffer, nrrData, nrrSize);

    NN_RESULT_DO(ro::RegisterModuleInfo(&m_RegistrationInfo, nrrBuffer));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            ro::UnregisterModuleInfo(&m_RegistrationInfo);
        }
    };

    auto pageAlignedNroSize = util::align_up(nroSize, os::MemoryPageSize);
    NN_RESULT_THROW_UNLESS(pageAlignedNroSize <= rest(), jit::ResultOutOfMemoryForDllLoad());

    size_t pageAlignedBssSize;
    NN_RESULT_DO(ro::GetBufferSize(&pageAlignedBssSize, nroData));
    NN_SDK_ASSERT_ALIGNED(pageAlignedBssSize, os::MemoryPageSize);
    NN_RESULT_THROW_UNLESS(pageAlignedBssSize <= rest(), jit::ResultOutOfMemoryForDllLoad());

    auto nroBuffer = allocateBuffer(pageAlignedNroSize);
    std::memcpy(nroBuffer, nroData, nroSize);
    auto bssBuffer = allocateBuffer(pageAlignedBssSize);

    success = true;
    this->m_State = State::NotLoaded;
    this->m_NroBuffer = nroBuffer;
    this->m_BssBuffer = bssBuffer;
    this->m_PageAlignedBssSize = pageAlignedBssSize;
    NN_RESULT_SUCCESS;
}

Result DllPlugin::Load() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_State == State::NotLoaded, jit::ResultInvalidState());

    auto success = false;

    NN_RESULT_DO(ro::LoadModule(&m_Module, m_NroBuffer, m_BssBuffer, m_PageAlignedBssSize, ro::BindFlag_Now));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            ro::UnloadModule(&m_Module);
        }
    };

    success = true;
    this->m_State = State::Loaded;
    NN_RESULT_SUCCESS;
}

void DllPlugin::Unload() NN_NOEXCEPT
{
    switch (m_State)
    {
        case State::Loaded:
            ro::UnloadModule(&m_Module);
            NN_FALL_THROUGH;
        case State::NotLoaded:
            ro::UnregisterModuleInfo(&m_RegistrationInfo);
            NN_FALL_THROUGH;
        default:
            this->m_State = State::Initial;
    }
}

DllPlugin::~DllPlugin() NN_NOEXCEPT
{
    Unload();
}

Result DllPlugin::GetSymbol(uintptr_t* pOut, const char* name, bool allowNotFound) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_State == State::Loaded, jit::ResultInvalidState());
    uintptr_t ret;
    NN_RESULT_TRY(ro::LookupModuleSymbol(&ret, &m_Module, name))
        NN_RESULT_CATCH(ro::ResultNotFound)
        {
            if (!allowNotFound)
            {
                NN_RESULT_RETHROW;
            }
            ret = 0;
        }
    NN_RESULT_END_TRY
    *pOut = ret;
    NN_RESULT_SUCCESS;
}

}}
