﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/idle/detail/idle_Log.h>
#include "idle_HandlerImplSleeper.h"
#include "idle_IHandler.h"

namespace nn { namespace idle { namespace server {

namespace {

    // 無効化する場合は 0 を返す
    nn::TimeSpan GetTimeToTriggerBody(const PolicyParam& policyParam)
    {
        // fwdbg の指定が全てをオーバーライド
        if ( policyParam.isAutoSleepDisabledForDebug )
        {
            return 0;
        }
        if ( policyParam.overrideAutoSleepTimeForDebugInSeconds > 0 )
        {
            return nn::TimeSpan::FromSeconds(policyParam.overrideAutoSleepTimeForDebugInSeconds);
        }

        // コンテキスト依存で無効化されていれば優先
        if ( policyParam.handlingContext.isAutoSleepDisabled )
        {
            return 0;
        }

        // 「メディア再生中に自動スリープする」設定がオフで、今メディア再生中なら、寝ない
        if ( !policyParam.isAutoSleepInMediaPlaybackEnabled && policyParam.handlingContext.isInMediaPlayback )
        {
            return 0;
        }

        switch (policyParam.handlingContext.operationMode)
        {
            case nn::omm::OperationMode::Handheld:
            {
                // コンテキスト依存の一時的な設定上書きがあれば優先
                if ( policyParam.handlingContext.overrideAutoSleepTimeInHandheldInSeconds > 0 )
                {
                    return nn::TimeSpan::FromSeconds(policyParam.handlingContext.overrideAutoSleepTimeInHandheldInSeconds);
                }

                if ( !policyParam.handlingContext.isInMediaPlayback )
                {
                    // 通常時ポリシーを適用
                    switch ( policyParam.handheldSleepPlan )
                    {
                        case nn::settings::system::HandheldSleepPlan_1Min: return nn::TimeSpan::FromMinutes(1);
                        case nn::settings::system::HandheldSleepPlan_3Min: return nn::TimeSpan::FromMinutes(3);
                        case nn::settings::system::HandheldSleepPlan_5Min: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::HandheldSleepPlan_10Min: return nn::TimeSpan::FromMinutes(10);
                        case nn::settings::system::HandheldSleepPlan_30Min: return nn::TimeSpan::FromMinutes(30);
                        case nn::settings::system::HandheldSleepPlan_Never: return 0; // 無効化
                    } // NOLINT(style/switch_default)
                }
                else
                {
                    // メディア再生中ポリシー
                    switch ( policyParam.handheldSleepPlan )
                    {
                        case nn::settings::system::HandheldSleepPlan_1Min: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::HandheldSleepPlan_3Min: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::HandheldSleepPlan_5Min: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::HandheldSleepPlan_10Min: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::HandheldSleepPlan_30Min: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::HandheldSleepPlan_Never: return 0; // 無効化
                    } // NOLINT(style/switch_default)
                }
            }
            case nn::omm::OperationMode::Console:
            {
                // コンテキスト依存の一時的な設定上書きがあれば優先
                if ( policyParam.handlingContext.overrideAutoSleepTimeInConsoleInSeconds > 0 )
                {
                    return nn::TimeSpan::FromSeconds(policyParam.handlingContext.overrideAutoSleepTimeInConsoleInSeconds);
                }

                if ( !policyParam.handlingContext.isInMediaPlayback )
                {
                    // 通常時ポリシーを適用
                    switch ( policyParam.consoleSleepPlan )
                    {
                        case nn::settings::system::ConsoleSleepPlan_1Hour: return nn::TimeSpan::FromHours(1);
                        case nn::settings::system::ConsoleSleepPlan_2Hour: return nn::TimeSpan::FromHours(2);
                        case nn::settings::system::ConsoleSleepPlan_3Hour: return nn::TimeSpan::FromHours(3);
                        case nn::settings::system::ConsoleSleepPlan_6Hour: return nn::TimeSpan::FromHours(6);
                        case nn::settings::system::ConsoleSleepPlan_12Hour: return nn::TimeSpan::FromHours(12);
                        case nn::settings::system::ConsoleSleepPlan_Never: return 0; // 無効化
                    } // NOLINT(style/switch_default)
                }
                else
                {
                    // メディア再生中ポリシー
                    switch ( policyParam.consoleSleepPlan )
                    {
                        case nn::settings::system::ConsoleSleepPlan_1Hour: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::ConsoleSleepPlan_2Hour: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::ConsoleSleepPlan_3Hour: return nn::TimeSpan::FromHours(4);
                        case nn::settings::system::ConsoleSleepPlan_6Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_12Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_Never: return 0; // 無効化
                    } // NOLINT(style/switch_default)
                }
            }
        } // NOLINT(style/switch_default)

        NN_SDK_ASSERT(false, "NEVER REACHED HERE");
        return 0;
    }

    nn::TimeSpan DelayTriggerTimeIfRequested(const nn::TimeSpan& newTime, const PolicyParam& policyParam)
    {
        switch ( policyParam.handlingContext.idleTimeDetectionDelayMode )
        {
            case IdleTimeDetectionExtension_Enabled:
            {
                const auto MaxDelayTime = nn::TimeSpan::FromMinutes(30);
                if ( 0 < newTime && newTime < MaxDelayTime )
                {
                    return MaxDelayTime;
                }
                break;
            }
            case IdleTimeDetectionExtension_EnabledUnsafe:
            {
                const auto MaxDelayTime = nn::TimeSpan::FromMinutes(60);
                if ( 0 < newTime && newTime < MaxDelayTime )
                {
                    return MaxDelayTime;
                }
            }
            default:
                break;
        }
        return newTime;
    }
}

// 無効化する場合は 0 を返す
nn::TimeSpan HandlerImplSleeper::GetTimeToTrigger(const PolicyParam& policyParam) NN_NOEXCEPT
{
    auto newTime = GetTimeToTriggerBody(policyParam);
    newTime = DelayTriggerTimeIfRequested(newTime, policyParam);
    NN_DETAIL_IDLE_INFO("Time to auto sleep: %d sec\n", newTime.GetSeconds());
    return newTime;
}

HandlerImplSleeper::HandlerImplSleeper() NN_NOEXCEPT :
    m_AutoPowerDownEvent(nn::os::EventClearMode_AutoClear, true)
{
}

void HandlerImplSleeper::HandleActiveState() NN_NOEXCEPT
{
}

void HandlerImplSleeper::HandleIdleState() NN_NOEXCEPT
{
    NN_DETAIL_IDLE_INFO("Signaling auto sleep.\n");
    m_AutoPowerDownEvent.Signal();
}

nn::os::SystemEventType* HandlerImplSleeper::GetAutoPowerDownEvent() NN_NOEXCEPT
{
    return m_AutoPowerDownEvent.GetBase();
}

}}} // namespace nn::idle::server

