﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/idle/idle_SystemTypes.h>
#include "idle_InactivityCounter.h"

namespace nn { namespace idle { namespace server {

    // ハンドリングコンテキストに依存するポリシーをカプセル化したクラス
    class ContextDependentGoverner
    {
    public:
        ContextDependentGoverner() NN_NOEXCEPT :
            m_pInactivityCounter(nullptr),
            m_IsInMediaPlaybackAfterglow(false)
        {}

        void Initialize(InactivityCounter* pInactivityCounter) NN_NOEXCEPT
        {
            m_pInactivityCounter = pInactivityCounter;
            m_IsInMediaPlaybackAfterglow = false;
        }

        // 新旧のコンテキストから独自ポリシーでフィルタした最終的なハンドリングコンテキストを生成する
        void InputHandlingContextChange(HandlingContext* pOutNew, const nn::TimeSpan& nowTime, const HandlingContext& prev, const HandlingContext& curr) NN_NOEXCEPT
        {
            *pOutNew = curr; // 基本は新しいコンテキストの値をコピー

            if ( prev.operationMode != curr.operationMode )
            {
                // 無条件で再カウントかけてよい
                m_pInactivityCounter->InputLastActiveTime(nowTime);
                m_IsInMediaPlaybackAfterglow = false;
            }

            if ( prev.aruid == curr.aruid ) // フォアグラウンドが切り替わっていない
            {
                if ( !prev.isInMediaPlayback && curr.isInMediaPlayback )
                {
                    NN_SDK_ASSERT(!m_IsInMediaPlaybackAfterglow);

                    // メディア再生開始
                }
                else if ( m_IsInMediaPlaybackAfterglow && curr.isInMediaPlayback )
                {
                    // メディア再生中ポリシー解除保留期間中のメディア再生開始
                    // このときはメディア再生しつづけている扱いなので無操作状態を解除しない
                    m_IsInMediaPlaybackAfterglow = false;
                }
                else if ( prev.isInMediaPlayback && !curr.isInMediaPlayback )
                {
                    // メディア再生終了
                    // メディア再生中ポリシー解除保留期間の開始
                    // 定期ポーリングされる Update の中で期間満了をチェックし、満了時に無操作状態の解除をする
                    m_IsInMediaPlaybackAfterglow = true;
                    pOutNew->isInMediaPlayback = true; // ポリシー上もうしばらく再生中に見せかける
                }
            }
            else // prev.aruid != curr.aruid : フォアグラウンドが切り替わった
            {
                // メディア再生中ポリシー解除保留期間は無効
                m_IsInMediaPlaybackAfterglow = false;
            }
        }

        // pHandlingContext を変更したとき true を返す
        bool UpdateContext(HandlingContext* pHandlingContext, const nn::TimeSpan& nowTime, bool wasNewInputDetected) NN_NOEXCEPT
        {
            bool hasChanged = false;
            if ( m_IsInMediaPlaybackAfterglow )
            {
                if ( wasNewInputDetected ||
                     ContiguousMediaPlaybackThreshold <= nowTime - nn::os::ConvertToTimeSpan(pHandlingContext->timeMediaPlaybackFinished) )
                {
                    // メディア再生中ポリシー解除保留期間の満了
                    m_IsInMediaPlaybackAfterglow = false;
                    pHandlingContext->isInMediaPlayback = false; // 再生中に見せかけていたのをやめる
                    hasChanged = true;
                }
            }
            return hasChanged;
        }

    private:
        const nn::TimeSpan ContiguousMediaPlaybackThreshold = nn::TimeSpan::FromMinutes(5);

    private:
        InactivityCounter* m_pInactivityCounter;

        // メディア再生終了後の余韻区間 = メディア再生中ポリシー解除保留期間（この区間内に次のメディア再生が始まった場合に連続再生とみなす）
        bool m_IsInMediaPlaybackAfterglow;
    };

}}} // namespace nn::idle::server

