﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/i2c/driver/detail/i2c_I2cSessionContext.h>
#include <nn/i2c/driver/i2c_Bus.h>
#include <nn/i2c/driver/i2c_BusDev.h>
#include <nn/i2c/driver/i2c_I2cDeviceProperty.h>
#include <nn/i2c/driver/i2c_Lib.h>
#include <nn/i2c/i2c_DeviceName.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../detail/i2c_CommandListFormat.h"
#include "detail/i2c_Core.h"
#include "i2c_Gen1DeviceNameConvertTable.h"

namespace {

const int DefaultRetryCount = 3;
const nn::TimeSpan DefaultRetryInterval = nn::TimeSpan::FromMilliSeconds(5);

nn::Result OpenSessionBody(nn::i2c::driver::I2cSession* pOutSession, nn::i2c::driver::I2cDeviceProperty* pDevice) NN_NOEXCEPT
{
    bool needsRollback = true;

    auto pSession = new (&nn::i2c::driver::detail::GetI2cSessionContext(*pOutSession))
        nn::i2c::driver::detail::I2cSessionContext(DefaultRetryCount, DefaultRetryInterval);

    NN_UTIL_SCOPE_EXIT
    {
        if ( needsRollback )
        {
            pSession->~I2cSessionContext();
        }
    };

    // TORIAEZU: 現在の API 体系ではアクセスモードは RW のみ
    NN_RESULT_DO(pSession->Open(pDevice, nn::ddsf::AccessMode_ReadAndWrite));

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

}

namespace nn { namespace i2c { namespace driver {

void Initialize() NN_NOEXCEPT
{
    detail::InitializeDrivers();
}

void Finalize() NN_NOEXCEPT
{
}

// [Gen2]
nn::Result OpenSession(I2cSession* pOutSession, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    I2cDeviceProperty* pDevice = nullptr;
    NN_RESULT_DO(detail::FindDevice(&pDevice, deviceCode));
    NN_SDK_ASSERT_NOT_NULL(pDevice);

    NN_RESULT_DO(OpenSessionBody(pOutSession, pDevice));

    NN_RESULT_SUCCESS;
}

// [Gen1] TODO: Deprecate
void OpenSession(I2cSession* pOutSession, I2cDevice device) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    nn::DeviceCode deviceCode = nn::DeviceCode::GetInvalidCode();
    NN_ABORT_UNLESS(FindCompatDeviceCodeEntryForGen1(&deviceCode, device));

    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenSession(pOutSession, deviceCode));
}

// [Gen1] TODO: Deprecate
void OpenSessionForDev(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    NN_UNUSED(addressingMode);
    NN_UNUSED(speedMode);

    I2cDeviceProperty* pDevice = nullptr;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::FindDeviceByBusIndexAndAddress(&pDevice, busIdx, slaveAddress));
    NN_SDK_ASSERT_NOT_NULL(pDevice);

    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenSessionBody(pOutSession, pDevice));
}

Result Send(I2cSession& session, const void* pInData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInData);
    NN_SDK_REQUIRES_GREATER(dataBytes, 0);

    NN_ABORT_UNLESS(dataBytes > 0);

    return detail::GetI2cSessionContextWithOpenCheck(session).Send(pInData, dataBytes, inOption);
}

Result Receive(void* pOutData, I2cSession& session, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutData);
    NN_SDK_REQUIRES_GREATER(dataBytes, 0);

    NN_ABORT_UNLESS(dataBytes > 0);

    return detail::GetI2cSessionContextWithOpenCheck(session).Receive(pOutData, dataBytes, inOption);
}

Result ExecuteCommandList(void* pReceiveBuffer, size_t receiveBufferSize, I2cSession& session, const void* pCommandList, size_t commandListLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReceiveBuffer);
    NN_SDK_REQUIRES_GREATER(receiveBufferSize, 0);
    NN_SDK_REQUIRES_NOT_NULL(pCommandList);
    NN_SDK_REQUIRES_GREATER(commandListLength, 0);

    NN_ABORT_UNLESS(receiveBufferSize > 0);
    NN_ABORT_UNLESS(commandListLength > 0);

    return detail::GetI2cSessionContextWithOpenCheck(session).ExecuteCommandList(pReceiveBuffer, receiveBufferSize, pCommandList, commandListLength);
}

nn::Result SetRetryPolicy(I2cSession& session, int maxRetryCount, int retryIntervalMicroSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(maxRetryCount, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(retryIntervalMicroSeconds, 0);

    return detail::GetI2cSessionContextWithOpenCheck(session).SetRetryPolicy(maxRetryCount, retryIntervalMicroSeconds);
}

void CloseSession(I2cSession& session) NN_NOEXCEPT
{
    auto& I2cSessionContext = detail::GetI2cSessionContext(session);
    I2cSessionContext.~I2cSessionContext();
}

void SuspendBuses() NN_NOEXCEPT
{
    for ( auto&& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<II2cDriver>().SuspendBus();
    }
}

void SuspendPowerBuses() NN_NOEXCEPT
{
    for ( auto&& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<II2cDriver>().SuspendPowerBus();
    }
}

void ResumeBuses() NN_NOEXCEPT
{
    for ( auto&& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<II2cDriver>().ResumeBus();
    }
}

void ResumePowerBuses() NN_NOEXCEPT
{
    for ( auto&& driver : detail::GetDriverList() )
    {
        driver.SafeCastTo<II2cDriver>().ResumePowerBus();
    }
}

}}} // nn::i2c::driver
