﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/dt.h>

#include <nn/dt/dt_DebugApi.h> // For GetNodePath()
#include <nn/i2c/detail/i2c_Log.h>
#include <nn/i2c/driver/detail/dt/i2c_DeviceCodeNodeParser.h>
#include <nn/i2c/i2c_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringView.h>

namespace nn { namespace i2c { namespace driver { namespace detail { namespace dt {

bool IDeviceCodeNodeParser::IsCompatible(const char* valueInProperty) const NN_NOEXCEPT
{
    for ( int i = 0; i < m_CompatibleNameCount; ++i )
    {
        const nn::util::string_view compatible(m_ppCompatibleNames[i]);

        if ( !compatible.compare(valueInProperty) )
        {
            return true;
        }
    }
    return false;
}

nn::Result DeviceCodeNodeRootParser::ParseAll() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ParserListLock)> lock(m_ParserListLock);

    nn::dt::Node dtDeviceCodeRootNode;
    NN_RESULT_TRY(nn::dt::FindNodeByPath(&dtDeviceCodeRootNode, "/chosen/nintendo,devicecodes"));
    NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
    {
        NN_DETAIL_I2C_WARN("/chosen/nintendo,devicecodes node not exist on device tree. Canceling parsing\n");
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_END_TRY

    // /chosen/nintendo,devicecodes 下のノードをひとつずつ探索する
    nn::dt::Node deviceCodeNode;
    NN_RESULT_TRY(nn::dt::GetFirstChildNode(&deviceCodeNode, &dtDeviceCodeRootNode));
    NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
    {
        NN_DETAIL_I2C_WARN("/chosen/nintendo,devicecodes did not have any child node. Canceling parsing\n");
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_END_TRY

    bool isLast = false;
    do
    {
        bool isTarget = false;
        NN_RESULT_DO(CheckSubsystemTag(&isTarget, deviceCodeNode));
        if ( isTarget )
        {
            NN_RESULT_DO(ParseSingleNode(deviceCodeNode));
        }

        NN_RESULT_TRY(nn::dt::GetNextSiblingNode(&deviceCodeNode, &deviceCodeNode));
        NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
        {
            isLast = true;
        }
        NN_RESULT_END_TRY
    } while( !isLast );


    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeNodeRootParser::CheckSubsystemTag(bool *pOutIsTagMatched, nn::dt::Node node) NN_NOEXCEPT
{
    *pOutIsTagMatched = false;

    bool hasDeviceCodesProperty = false;
    NN_RESULT_DO(nn::dt::IsPropertyExist(&hasDeviceCodesProperty, &node, "device-codes"));
    if ( !hasDeviceCodesProperty )
    {
        // device-codes プロパティが無かった
        char nodePath[64] = "";
        auto result = nn::dt::GetNodePath(nodePath, sizeof(nodePath), &node);
        if ( result.IsSuccess() )
        {
            NN_DETAIL_I2C_ERROR("node (%s) did not have device-codes property!\n", nodePath);
        }
        else
        {
            NN_DETAIL_I2C_ERROR("Node (offset %d) did not have device-codes property! and could not get path on node ( result:0x%x ) ...\n", node.GetOffset(), result.GetInnerValueForDebug());
        }
        NN_RESULT_SUCCESS;
    }

    bool isTargetSubsystemNode = false;
    NN_RESULT_DO(nn::dt::IsPropertyExist(&isTargetSubsystemNode, &node, m_SubsystemTag));
    if ( !isTargetSubsystemNode )
    {
        NN_RESULT_SUCCESS;
    }

    *pOutIsTagMatched = true;
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeNodeRootParser::FindCompatibleParser(IDeviceCodeNodeParser** ppOutParser, nn::dt::Node node) NN_NOEXCEPT
{
    *ppOutParser = nullptr;

    char compatibleNamesBuffer[MaxCompatibleBytes];
    size_t propertySize;
    NN_RESULT_TRY(nn::dt::GetPropertyString(&propertySize, compatibleNamesBuffer, sizeof(compatibleNamesBuffer), &node, "compatible"));
    NN_RESULT_CATCH(nn::dt::ResultNodeNotExist)
    {
        // compatible プロパティが無かった
        char nodePath[64] = "";
        auto result = nn::dt::GetNodePath(nodePath, sizeof(nodePath), &node);
        if ( result.IsSuccess() )
        {
            NN_DETAIL_I2C_ERROR("Node (%s) did not have compatible property!\n", nodePath);
        }
        else
        {
            NN_DETAIL_I2C_ERROR("Node (offset %d) did not have compatible property! and could not get path on node ( result:0x%x ) ...\n", node.GetOffset(), result.GetInnerValueForDebug());
        }
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_END_TRY

    nn::dt::StringForwardList compatibleNames(compatibleNamesBuffer, compatibleNamesBuffer + propertySize);
    for ( const auto& compatible : compatibleNames )
    {
        // compatible プロパティは先頭の記載を優先するため、先頭から順に判定して見つかったら即 return で良い
        for ( auto&& parser : m_ParserList )
        {
            if ( parser.IsCompatible(compatible) )
            {
                // TORIAEZU: 一つの compatible 文字列に複数のドライバパーサがヒットする状況はありうるが、最初にヒットしたパーサで即 return する
                *ppOutParser = &parser;
                NN_RESULT_SUCCESS;
            }
        }
    }

    // このノードの compatible をサポートするドライバが誰もいなかったので、諸情報をログにダンプしておく
    {
        char nodePath[64] = "";
        auto result = nn::dt::GetNodePath(nodePath, sizeof(nodePath), &node);
        if ( result.IsSuccess() )
        {
            NN_DETAIL_I2C_ERROR("Node (%s) is not supported by any available driver.\n", nodePath);
        }
        else
        {
            NN_DETAIL_I2C_ERROR("Node (offset %d) is not supported by any available driver, and could not get path on node ( result:0x%x ) ...\n", node.GetOffset(), result.GetInnerValueForDebug());
        }

        NN_DETAIL_I2C_ERROR("compatible property on this node were:\n");
        for ( const auto& compatible : compatibleNames )
        {
            NN_UNUSED(compatible); // Release ビルドでの警告対策
            NN_DETAIL_I2C_ERROR("  %s\n", compatible);
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeNodeRootParser::ParseSingleNode(nn::dt::Node node) NN_NOEXCEPT
{
    IDeviceCodeNodeParser* pParser = nullptr;
    NN_RESULT_DO(FindCompatibleParser(&pParser, node));
    if ( !pParser )
    {
        // サポートするパーサが不在。 FindCompatibleParser 内でエラーログのダンプは済んでいるのでそのまま return
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(pParser->OnCompatibleNodeBegin(node));
    NN_UTIL_SCOPE_EXIT
    {
        pParser->OnCompatibleNodeEnd(node);
    };

    int propertyDeviceCodeIndex = 0; // "device-codes"
    for ( ; ; )
    {
        Bit32 deviceCode;
        NN_RESULT_TRY(nn::dt::GetProperty(&deviceCode, &node, "device-codes", propertyDeviceCodeIndex))
        NN_RESULT_CATCH(nn::dt::ResultPropertyNotExist)
        {
            // 終端に到達
            break;
        }
        NN_RESULT_END_TRY

        ++propertyDeviceCodeIndex;

        // これが失敗を返すとパースは中断する
        NN_RESULT_DO(pParser->OnDeviceCodeFound(deviceCode, node));
    }

    NN_RESULT_SUCCESS;
}

}}}}} // nn::i2c::driver::detail::dt

