﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>

#include <nn/i2c/detail/i2c_Log.h>
#include <nn/i2c/driver/i2c.h>
#include <nn/i2c/driver/i2c_Suspend.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/i2c/i2c_CommandListFormatter.h>
#include "detail/i2c_TargetSpec.h"
#include "detail/i2c_ResourceManager.h"
#include "detail/i2c_Util.h"
#include "../detail/i2c_CommandListFormat.h"


namespace {

void AssertLibraryInitialized()
{
    NN_SDK_REQUIRES(nn::i2c::driver::detail::ResourceManager::GetInstance().IsInitialized(),
        "I2C Error: Library is not initialized\n");
}

inline nn::i2c::driver::detail::ResourceManager& GetResourceManager()
{
    return nn::i2c::driver::detail::ResourceManager::GetInstance();
}


/**
 * @brief   コマンドのパースと実行を行う関数の型宣言
 */
typedef nn::Result (*CommandHandlerFunction)(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer, nn::i2c::driver::I2cSession& session);

nn::Result ReceiveHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer, nn::i2c::driver::I2cSession& session) NN_NOEXCEPT
{
    // byte1
    const nn::i2c::TransactionOption Option = static_cast<nn::i2c::TransactionOption>(
        ( ( **pCurrentCmd & nn::i2c::detail::ReceiveCommandFormat::StartCondition::Mask ) ? nn::i2c::TransactionOption_StartCondition : 0 ) |
        ( ( **pCurrentCmd & nn::i2c::detail::ReceiveCommandFormat::StopCondition::Mask  ) ? nn::i2c::TransactionOption_StopCondition  : 0 ) );
    (*pCurrentCmd)++;

    // byte2
    const size_t DataBytes = ( **pCurrentCmd & nn::i2c::detail::ReceiveCommandFormat::DataBytes::Mask ) >> nn::i2c::detail::ReceiveCommandFormat::DataBytes::Pos;
    (*pCurrentCmd)++;

    // コマンド発行
    NN_RESULT_DO( Receive( *pCurrentReceiveBuffer, session, DataBytes, Option ) );
    (*pCurrentReceiveBuffer) += DataBytes;

    NN_RESULT_SUCCESS;
}

nn::Result SendHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer, nn::i2c::driver::I2cSession& session) NN_NOEXCEPT
{
    // byte1
    const nn::i2c::TransactionOption Option = static_cast<nn::i2c::TransactionOption>(
        ( ( **pCurrentCmd & nn::i2c::detail::SendCommandFormat::StartCondition::Mask ) ? nn::i2c::TransactionOption_StartCondition : 0 ) |
        ( ( **pCurrentCmd & nn::i2c::detail::SendCommandFormat::StopCondition::Mask  ) ? nn::i2c::TransactionOption_StopCondition  : 0 ) );
    (*pCurrentCmd)++;

    // byte2
    const size_t DataBytes = ( **pCurrentCmd & nn::i2c::detail::SendCommandFormat::DataBytes::Mask ) >> nn::i2c::detail::SendCommandFormat::DataBytes::Pos;
    (*pCurrentCmd)++;

    // byte3 -
    // コマンド発行
    NN_RESULT_DO( Send( session, *pCurrentCmd, DataBytes, Option ) );
    (*pCurrentCmd) += DataBytes;

    NN_RESULT_SUCCESS;
}

nn::Result ExtensionHandler(const uint8_t** pCurrentCmd, uint8_t** pCurrentReceiveBuffer, nn::i2c::driver::I2cSession& session) NN_NOEXCEPT
{
    // byte1
    const uint8_t SubCommand = ( **pCurrentCmd & nn::i2c::detail::CommonCommandFormat::SubCommandHeader::Mask ) >> nn::i2c::detail::CommonCommandFormat::SubCommandHeader::Pos;
    (*pCurrentCmd)++;

    switch (SubCommand)
    {
    case nn::i2c::detail::SubCommandId_Sleep:
        {
            // byte2
            const size_t DataBytes = ( **pCurrentCmd & nn::i2c::detail::SleepCommandFormat::SleepMicroSeconds::Mask ) >> nn::i2c::detail::SleepCommandFormat::SleepMicroSeconds::Pos;
            (*pCurrentCmd)++;
            // コマンド発行
            nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(DataBytes) );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

const CommandHandlerFunction CommandHandlerFunctionTable[nn::i2c::detail::CommandId_CountMax] =
{
    SendHandler      ,
    ReceiveHandler   ,
    ExtensionHandler ,
};
} // namespace

namespace nn {
namespace i2c {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    GetResourceManager().Initialize();
}

void Finalize() NN_NOEXCEPT
{
    GetResourceManager().Finalize();
}

// [Gen2] NOT SUPPORTED
nn::Result OpenSession(I2cSession* pOutSession, nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    NN_UNUSED(pOutSession);
    NN_UNUSED(deviceCode);
    NN_DETAIL_I2C_FATAL("Driver generation mismatch: Gen2 API called for Gen1 server.\n");
    NN_ABORT();
    return nn::i2c::ResultDeviceNotFound();
}

void OpenSession(I2cSession* pOutSession, I2cDevice device) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    NN_ABORT_UNLESS( detail::IsSupported(device), "The device isn't supported. Please call ForDev APIs instead.\n");
    GetResourceManager().OpenSession(
        pOutSession, detail::GetBusIdx(device), detail::GetSlaveAddress(device), detail::GetAddressingMode(device),
                     detail::GetSpeedMode(device), detail::GetRetryConunt(device), detail::GetRetryInterval(device) );
}

void OpenSessionForDev(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode ) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_REQUIRES(busIdx < nn::i2c::driver::detail::MaxBuses);

    // ForDev では Retry は設定なし
    GetResourceManager().OpenSession(
        pOutSession, busIdx, slaveAddress, addressingMode, speedMode, 0, 0);
}

Result Send(I2cSession& session, const void* pInData, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pInData);
    NN_ABORT_UNLESS(dataBytes > 0);

    std::lock_guard<detail::StaticMutex> lock(*GetResourceManager().GetTransactionOrderMutex(session._busIdx));

    return GetResourceManager().GetSession(session._sessionId).ExecuteTransactionWithRetry(
        nullptr, pInData, dataBytes, inOption, nn::i2c::driver::detail::Command_Send );
}

Result Receive(void* pOutData, I2cSession& session, size_t dataBytes, TransactionOption inOption) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutData);
    NN_ABORT_UNLESS(dataBytes > 0);
    AssertLibraryInitialized();

    std::lock_guard<detail::StaticMutex> lock(*GetResourceManager().GetTransactionOrderMutex(session._busIdx));

    return GetResourceManager().GetSession(session._sessionId).ExecuteTransactionWithRetry(
        pOutData, nullptr, dataBytes, inOption, nn::i2c::driver::detail::Command_Receive );
}

Result ExecuteCommandList(void* pReceiveBuffer, size_t receiveBufferSize, I2cSession& session, const void* pCommandList, size_t commandListLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReceiveBuffer);
    NN_ABORT_UNLESS(receiveBufferSize > 0);
    NN_SDK_REQUIRES_NOT_NULL(pCommandList);
    NN_ABORT_UNLESS(commandListLength > 0);
    AssertLibraryInitialized();
    std::lock_guard<detail::StaticMutex> lock(*GetResourceManager().GetTransactionOrderMutex(session._busIdx));

    const uint8_t* pCurrentCommand    = static_cast<const uint8_t*>( pCommandList ); // 未実行コマンドの先頭アドレス
    const uint8_t* const pCommandLast = pCurrentCommand + commandListLength;         // コマンドリストの終端
    uint8_t* pCurrentReceiveBuffer    = static_cast<uint8_t*>(pReceiveBuffer);       // 受信前バッファの書き込み先先頭アドレス

    while( pCurrentCommand < pCommandLast )
    {
        // byte1 を見てコマンドを判定
        const uint8_t Cmd = ( *pCurrentCommand & nn::i2c::detail::CommonCommandFormat::CommandHeader::Mask ) >> nn::i2c::detail::CommonCommandFormat::CommandHeader::Pos;

        NN_ABORT_UNLESS( Cmd == nn::i2c::detail::CommandId_Send    ||
                         Cmd == nn::i2c::detail::CommandId_Receive ||
                         Cmd == nn::i2c::detail::CommandId_Extension );

        // コマンド別に byte1 以降のコマンドリストをパースして実行
        NN_RESULT_DO( CommandHandlerFunctionTable[Cmd](&pCurrentCommand, &pCurrentReceiveBuffer, session) );
    }

    NN_RESULT_SUCCESS;
}

// [Gen2] NOT SUPPORTED
nn::Result SetRetryPolicy(I2cSession& session, int maxRetryCount, int retryIntervalMicroSeconds) NN_NOEXCEPT
{
    NN_UNUSED(session);
    NN_UNUSED(maxRetryCount);
    NN_UNUSED(retryIntervalMicroSeconds);
    NN_DETAIL_I2C_FATAL("Driver generation mismatch: Gen2 API called for Gen1 server.\n");
    NN_ABORT();
    return nn::i2c::ResultUnknown();
}

void CloseSession(I2cSession& session) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    GetResourceManager().CloseSession( session );
}

void SuspendBuses() NN_NOEXCEPT
{
    GetResourceManager().SuspendBuses();
}

void SuspendPowerBuses() NN_NOEXCEPT
{
    GetResourceManager().SuspendPowerBuses();
}


void ResumeBuses() NN_NOEXCEPT
{
    GetResourceManager().ResumeBuses();
}

void ResumePowerBuses() NN_NOEXCEPT
{
    GetResourceManager().ResumePowerBuses();
}


} // driver
} // i2c
} // nn
