﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/i2c/driver/i2c.h>
#include <nn/i2c/detail/i2c_Log.h>
#include <nn/i2c/i2c_ResultForPrivate.h>

#include "i2c_TargetSpec.h"

#include "i2c_Session.h"
#include "i2c_Command.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

void Session::Open( int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode, BusAccessor* pBusAccessor, int retryCount, nn::TimeSpan retryInterval) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );
    if( !m_IsOpen )
    {
        m_pBusAccessor   = pBusAccessor;
        m_BusIdx         = busIdx;
        m_SlaveAddress   = slaveAddress;
        m_AddressingMode = addressingMode;
        m_RetryCount     = retryCount;
        m_RetryInterval  = retryInterval;
        m_pBusAccessor->Open(busIdx, speedMode);
        m_IsOpen = true; // m_IsOpen へのアクセスは ResourceManager でシリアライズされている
    }
}

void Session::Start()  NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );
    if( m_IsOpen )
    {
        // Open 時に UserCount は 1 になっているので、初回のみ InitialConfig を実行
        if(m_pBusAccessor->GetUserCount() == 1)
        {
            m_pBusAccessor->ExecuteInitialConfig();
        }
    }

}

void Session::Close() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );
    if( m_IsOpen )
    {
        m_pBusAccessor->Close();
        m_pBusAccessor = nullptr;
        m_IsOpen = false; // m_IsOpen へのアクセスは ResourceManager でシリアライズされている
    }
}

bool Session::IsOpened() const NN_NOEXCEPT
{
    return m_IsOpen;
}

Result  Session::ExecuteTransactionWithRetry(void* pReceiveData, const void* pSendData, size_t dataBytes, TransactionOption inOption, Command command) NN_NOEXCEPT
{
    Result    result;

    int retryCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        result = ExecuteTransaction(pReceiveData, pSendData, dataBytes, inOption, command);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(ResultTimeout)
            {
                ++retryCount;
                if (retryCount <= m_RetryCount)
                {
                    NN_DETAIL_I2C_ERROR("timeout occur retry (%d)\n", retryCount);
                    nn::os::SleepThread(m_RetryInterval);
                    continue;
                }
                NN_RESULT_THROW(ResultBusBusy());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }
    return result;
}

Result  Session::ExecuteTransaction( void* pReceiveData, const void* pSendData, size_t dataBytes, TransactionOption inOption, Command command ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );

    Result    result;

    if (m_pBusAccessor->CheckBusy())
    {
        return nn::i2c::ResultBusBusy();
    }

    m_pBusAccessor->StartCondition();

    if ((result = m_pBusAccessor->StartTransaction(m_SlaveAddress, m_AddressingMode, command)).IsSuccess()) // スレーブアドレスの送信
    {
        switch (command)
        {
        case Command_Send:
            result = m_pBusAccessor->Send(reinterpret_cast<const Bit8*>(pSendData), dataBytes, inOption, m_SlaveAddress, m_AddressingMode);
            break;

        case Command_Receive:
            result = m_pBusAccessor->Receive(reinterpret_cast<Bit8*>(pReceiveData), dataBytes, inOption, m_SlaveAddress, m_AddressingMode);
            break;

        default:
            NN_ABORT("Invalid command");
            break;
        }
    }

    m_pBusAccessor->StopCondition();

    return result;
}

void Session::Dump() NN_NOEXCEPT
{
    if ( m_IsOpen )
    {
        NN_DETAIL_I2C_INFO("Opened session, BusIdx:%d, SlaveAddress:0x%x.\n", m_BusIdx, m_SlaveAddress);
    }
    else
    {
        NN_DETAIL_I2C_INFO("Closed session.\n");
    }
}

} // detail
} // driver
} // i2c
} // nn
