﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   レジスタ操作等のデバイスアクセス系ユーティリティ関数
 */

#pragma once

#include <nn/nn_Abort.h>
#include <nn/dd.h>

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

// address に指定したレジスタの読み出しを実行します。
// TegraX1 では レジスタの書き込み完了に使用します。
inline void DummyRead(volatile Bit32* address)
{
    (void)(*address);
}

NN_FORCEINLINE void Write32(volatile uint32_t* reg, uint32_t val) NN_NOEXCEPT
{
    *reg = val;
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOn32(volatile uint32_t* reg, uint32_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp |= flags;
    *reg = temp;
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOff32(volatile uint32_t* reg, uint32_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp &= ~flags;
    *reg = temp;
}

// Overwrite partly on specified register
// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void Modify32(volatile uint32_t* reg, uint32_t value, uint32_t mask) NN_NOEXCEPT
{
    auto temp = *reg;
    temp &= ~mask;
    temp |= value;
    *reg = temp;
}

NN_FORCEINLINE uint32_t Read32(volatile uint32_t* reg) NN_NOEXCEPT
{
    return *reg;
}

NN_FORCEINLINE uintptr_t GetVirtualAddress(nn::dd::PhysicalAddress physicalAddress, size_t addressSize) NN_NOEXCEPT
{
    uintptr_t virtualAddress = nn::dd::QueryIoMappingAddress(physicalAddress, addressSize);
    NN_ABORT_UNLESS(virtualAddress != 0, "I/O registers for 0x%llx are not mapped. "
        "Make sure the capability setting is properly set for this process.\n", physicalAddress);
    return virtualAddress;
}

} // detail
} // driver
} // i2c
} // nn
