﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http/stream/http_CurlInputStream.h>

#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/http/http_Result.h>
#include <nn/http/detail/http_Log.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_ScopeExit.h>

#include "http_StreamUtility.h"

namespace nn { namespace http { namespace stream {

/* --------------------------------------------------------------------------------------------
    Callback
 */
size_t CurlInputStreamBase::WriteFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT
{
    const auto InputBytes = unitBytes * count;
    auto& obj = *reinterpret_cast<CurlInputStreamBase*>(pContext);

    // Obj が受信可能であることの検査
    if (true
        && obj.m_BufferInfo.empty.load()
        && obj.m_BufferInfo.total > 0)
    {
        // 受信したデータをバッファにコピー
        auto copyHead = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(ptr) + obj.m_InputOffset);
        auto copyBytes = std::min(InputBytes - obj.m_InputOffset, obj.m_BufferInfo.total);
        std::memcpy(obj.m_BufferInfo.address, copyHead, copyBytes);
        obj.m_BufferInfo.used = copyBytes;
        obj.m_BufferInfo.empty.store(false);

        // 末尾かどうか
        if (obj.m_InputOffset + copyBytes >= InputBytes)
        {
#if 0
            NN_DETAIL_HTTP_TRACE("BODY ------------------------\n");
            NN_DETAIL_HTTP_TRACE("%.*s\n", InputBytes, ptr);
#endif
            NN_SDK_ASSERT(obj.m_InputOffset + copyBytes == InputBytes);
            obj.m_InputOffset = 0;
            return InputBytes;
        }
        obj.m_InputOffset += copyBytes;
    }
    return CURL_WRITEFUNC_PAUSE;
}

/* --------------------------------------------------------------------------------------------
    libcURL の設定
*/
Result CurlInputStreamBase::FinalizeImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurlMulti != nullptr);

    auto mr = curl_multi_cleanup(m_CurlMulti);
    NN_RESULT_DO(ConvertCurlMultiCodeToResult(mr));
    m_CurlMulti = nullptr;
    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------------------------------------------
    InputStream 実装
 */
CurlInputStreamBase::CurlInputStreamBase(CURL* curlHandle, const util::Cancelable* pCancelable) NN_NOEXCEPT
    : WebApiAccessorBase(curlHandle, pCancelable)
{
}

CurlInputStreamBase::~CurlInputStreamBase() NN_NOEXCEPT
{
    if (m_Opened)
    {
        Close();
    }
    if (m_CurlMulti != nullptr)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(FinalizeImpl());
        WebApiAccessorBase::Reset();
    }
}

Result CurlInputStreamBase::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurlMulti == nullptr);

    NN_RESULT_DO(WebApiAccessorBase::SetDefault());
    NN_RESULT_DO(WebApiAccessorBase::SetWriteFunction(WriteFunction, this));

    auto* curlm = curl_multi_init();
    if (!curlm)
    {
        NN_DETAIL_HTTP_ERROR("Failed to create multi handle.\n");
        return ResultCurlMultiInitFailed();
    }
    m_CurlMulti = curlm;
    NN_RESULT_SUCCESS;
}

size_t CurlInputStreamBase::Receive(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
#define NN_HTTP_IO_FAIL_WITH_RESULT_UNLESS(condition, result) \
    do \
    { \
        if (!(condition)) \
        { \
            m_IoResult = (result); \
            return 0u; \
        } \
    } while (NN_STATIC_CONDITION(false))

    if (!(!m_BufferInfo.empty.load() && m_IoResult.IsSuccess()))
    {
        // 次のすべての条件を満たさない場合は即時返る
        // 1. 最後の読み込みが 0 バイトでない
        // 2. 以前に IO でエラーがない
        return 0u;
    }

    m_BufferInfo.address = buffer;
    m_BufferInfo.total = bufferSize;
    m_BufferInfo.used = 0u;
    m_BufferInfo.empty.store(true); // コールバックでのバッファの上書き OK

    int numRunning = 0;
    do
    {
        int numfds;
        // timeout = 1000msec を指定しているが、 libcURL 内部で 1msec に置き換えられる
        auto mr = curl_multi_wait(m_CurlMulti, NULL, 0, 1000, &numfds);
        NN_RESULT_DO(ConvertCurlMultiCodeToResult(mr));

        // キャンセルされているかの検査
        NN_HTTP_IO_FAIL_WITH_RESULT_UNLESS(!IsCanceled(), ResultCanceled());

        // 一時停止の解除
        auto r = WebApiAccessorBase::Pause(CURLPAUSE_CONT);
        NN_HTTP_IO_FAIL_WITH_RESULT_UNLESS(r.IsSuccess(), r);

        if (numfds <= 0)
        {
            // 通信が始まっていない場合は待つ。
            os::SleepThread(TimeSpan::FromMilliSeconds(10));
        }

        // Read の継続
        mr = curl_multi_perform(m_CurlMulti, &numRunning);
        NN_RESULT_DO(ConvertCurlMultiCodeToResult(mr));
        if (numRunning <= 0)
        {
            NN_SDK_ASSERT(numRunning == 0);
            m_IoResult = ExtractMultiPerformError(m_CurlMulti);
            break;
        }
    } while (m_BufferInfo.empty.load());
    return m_BufferInfo.used;

#undef NN_HTTP_IO_FAIL_WITH_RESULT_UNLESS
}

Result CurlInputStreamBase::Open() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurlMulti != nullptr);
    NN_SDK_REQUIRES(!m_Opened);

    // multi-interface に登録
    NN_RESULT_DO(WebApiAccessorBase::Attach(m_CurlMulti));

    // いったんここで Open 済みにする
    m_Opened = true;
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            Close();
        }
    };

    // Fillbuffer で即時 wait 解除されるように pause しておく
    NN_RESULT_DO(WebApiAccessorBase::Pause(CURLPAUSE_ALL));

    // 通信の開始 (この時点で m_BufferInfo にバッファは設定されていない)
    int numRunning;
    auto mr = curl_multi_perform(m_CurlMulti, &numRunning);
    NN_RESULT_DO(ConvertCurlMultiCodeToResult(mr));

    if (numRunning <= 0)
    {
        NN_SDK_ASSERT(numRunning == 0);
        NN_RESULT_DO(ExtractMultiPerformError(m_CurlMulti));
    }
    success = true;
    NN_RESULT_SUCCESS;
}

void CurlInputStreamBase::Close() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Opened);
    NN_ABORT_UNLESS_RESULT_SUCCESS(WebApiAccessorBase::Detach(m_CurlMulti));
    m_Opened = false;
}

Result CurlInputStreamBase::ExtractMultiPerformError(CURLM* curlMulti) NN_NOEXCEPT
{
    CURLcode r = CURLE_OK;
    int numMessages;
    do
    {
        auto* pInfo = curl_multi_info_read(curlMulti, &numMessages);
        if (true
            && r == CURLE_OK
            && pInfo != nullptr
            && pInfo->msg == CURLMSG_DONE
            && pInfo->data.result != CURLE_OK)
        {
            r = pInfo->data.result;
        }
    } while (numMessages > 0);
    NN_RESULT_DO(WebApiAccessorBase::HandleCurlErrorWithContext(r));
    NN_RESULT_DO(WebApiAccessorBase::GetResult());
    NN_RESULT_SUCCESS;
}

Result CurlInputStreamBase::GetResult() const NN_NOEXCEPT
{
    NN_RESULT_DO(m_IoResult);
    NN_RESULT_DO(WebApiAccessorBase::GetResult());
    NN_RESULT_SUCCESS;
}
int32_t CurlInputStreamBase::GetHttpCode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_Opened, "[nn::http] CurlInputStreamBase::GetHttpCode requires itself closed.\n");
    return WebApiAccessorBase::GetHttpCode();
}

CurlInputStream::CurlInputStream(CURL* curlHandle, const util::Cancelable* pCancelable) NN_NOEXCEPT
    : CurlInputStreamBase(curlHandle, pCancelable)
{
}

}}} // ~namespace nn::http::stream
