﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>
#include <nn/nifm.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/mem.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <curl/curl.h>

#include <cstdlib>
#include <atomic>

#include "http_Utility.h"

namespace nn { namespace http {

namespace {
    bool                   s_bUseAllocator = false;
    mem::StandardAllocator s_Allocator;
    std::atomic_int        s_InitializeCount = { 0 };

    void* AllocateFromAllocator(size_t size)
    {
#if 1
        NN_SDK_ASSERT(s_bUseAllocator);
        return s_Allocator.Allocate(size);
#else
        void* p = s_Allocator.Allocate(size);
        NN_SDK_LOG("Allocate(%d) => %p\n", size, p);
        return p;
#endif
    }
    void FreeToAllocator(void* p)
    {
        NN_SDK_ASSERT(s_bUseAllocator);
        s_Allocator.Free(p);
    }
    void* ReAllocateFromAllocator(void* p, size_t size)
    {
        NN_SDK_ASSERT(s_bUseAllocator);
        return s_Allocator.Reallocate(p, size);
    }

    Result InitializeImpl(AllocateFunction allocfun, FreeFunction freefun, ReAllocateFunction reallocfun)
    {
        Result result;

        if (s_InitializeCount++ == 0)
        {
            result = nn::nifm::Initialize();
            NN_HTTP_RESULT_THROW_IF_FAILURE(result);

            CURLcode code = curl_global_init_mem(CURL_GLOBAL_DEFAULT, allocfun, freefun, reallocfun, DuplicateString, AllocateZeroInitializedArray);
            if (code != CURLE_OK)
            {
                //nn::nifm::Finalize();
                return ConvertCurlCodeToResult(code);
            }

            SetAllocateFunctions(allocfun, freefun, reallocfun);
        }

        return ResultSuccess();
    }
}

Result Initialize(AllocateFunction allocfun, FreeFunction freefun, ReAllocateFunction reallocfun)
{
    NN_SDK_REQUIRES(s_InitializeCount == 0, "Initialize() with allocator must be called at first.");
    return InitializeImpl(allocfun, freefun, reallocfun);
}

Result Initialize(void* p, size_t size)
{
    NN_SDK_REQUIRES(s_InitializeCount == 0, "Initialize() with allocator must be called at first.");
    s_Allocator.Initialize(p, size);
    s_bUseAllocator = true;

    Result result = InitializeImpl(AllocateFromAllocator, FreeToAllocator, ReAllocateFromAllocator);
    if (result.IsFailure())
    {
        s_Allocator.Finalize();
        s_bUseAllocator = false;
        return result;
    }

    return ResultSuccess();
}

Result Initialize()
{
    return InitializeImpl(std::malloc, std::free, std::realloc);
}

void Finalize()
{
    if (--s_InitializeCount == 0)
    {
        curl_global_cleanup();

        //nn::nifm::Finalize();

        if (s_bUseAllocator)
        {
#ifdef NN_DETAIL_ENABLE_SDK_ASSERT
            mem::StandardAllocator::AllocatorHash hash = s_Allocator.Hash();
            if (hash.allocCount > 0)
            {
                s_Allocator.Dump();
            }
            NN_SDK_REQUIRES(hash.allocCount == 0, "nn::http related objects must be freed.");
#endif
            s_Allocator.Finalize();
            s_bUseAllocator = false;
        }

        SetAllocateFunctions(nullptr, nullptr, nullptr);
    }
}

bool IsInitialized()
{
    return s_InitializeCount > 0;
}

}} // ~namespace nn::http
