﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/socket.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include <curl/curl.h>
#include <mutex>

#include "http_Utility.h"

#define NN_HTTP_CURL_SYNCHRONIZED std::lock_guard<os::Mutex> synchronized(m_mutexCurl)

namespace {

static void Dump(const char *text, unsigned char *ptr, size_t size)
{
    size_t i;
    size_t c;
    unsigned int width = 0x10;

    NN_HTTP_TRACE_ALWAYS("%s, %10.10ld bytes (0x%8.8lx)\n", text, (long)size, (long)size);

    for (i = 0; i < size; i += width)
    {
        NN_HTTP_TRACE_ALWAYS("%4.4lx: ", (long)i);

        // hex 表示
        for (c = 0; c < width; c++)
        {
            if (i + c < size)
            {
                NN_HTTP_TRACE_ALWAYS("%02x ", ptr[i + c]);
            }
            else
            {
                NN_HTTP_TRACE_ALWAYS("   ");
            }
        }

        // ascii 表示
        for (c = 0; (c < width) && (i + c < size); ++c)
        {
            NN_HTTP_TRACE_ALWAYS("%c", (ptr[i + c] >= 0x20 && ptr[i + c] < 0x80) ? ptr[i + c] : '.');
        }

        NN_HTTP_TRACE_ALWAYS("\n");
    }
}

}

namespace nn {
namespace http {

ConnectionBroker::ConnectionBroker()
    : m_CurlMultiHandle(nullptr)
    , m_mutexCurl(true)
    , m_mutexPesudoThread(true)
    , m_eventPesudoThreadLeave(os::EventClearMode_AutoClear)
{
}

ConnectionBroker::~ConnectionBroker()
{
    Finalize();
}

Result ConnectionBroker::Initialize()
{
    NN_SDK_REQUIRES(IsInitialized(), "nn::http::Initialize() was not called.");
    NN_HTTP_CURL_SYNCHRONIZED;

    NN_SDK_REQUIRES_EQUAL(m_CurlMultiHandle, nullptr);

    m_CurlMultiHandle = curl_multi_init();
    if (!m_CurlMultiHandle)
    {
        NN_HTTP_ERROR("Failed to create multi handle.\n");
        return ResultCurlMultiInitFailed();
    }
    m_pNoProxyHosts = nullptr;
    std::memset(m_ProxyHostname, 0, sizeof(m_ProxyHostname));
    std::memset(m_ProxyUsername, 0, sizeof(m_ProxyUsername));
    std::memset(m_ProxyPassword, 0, sizeof(m_ProxyPassword));

    SetMaxTotalConnections(MaxConnects);
    m_bVerbose = false;
    m_bEnableClientCert = false;
    m_bEnableManualProxy = false;
    m_bSkipSslVerificationForDebug = false;

    m_nBufferSizeReceive = DefaultSocketBufferSize;
    m_nBufferSizeSend    = DefaultSocketBufferSize;

    return ResultSuccess();
}

void ConnectionBroker::Finalize()
{
    NN_HTTP_CURL_SYNCHRONIZED;

    if (m_CurlMultiHandle)
    {
        CURLMcode mcode = curl_multi_cleanup(m_CurlMultiHandle);
        NN_SDK_ASSERT_EQUAL(mcode, CURLM_OK); NN_UNUSED(mcode);
        m_CurlMultiHandle = nullptr;
    }
}

void ConnectionBroker::SetMaxTotalConnections(int maxConnections)
{
    NN_HTTP_CURL_SYNCHRONIZED;

    NN_SDK_REQUIRES(m_CurlMultiHandle != nullptr, "nn::http::ConnectionBroker::Initialize() was not called.");

    // 同時にオープンになる最大数
    CURLMcode mcode = curl_multi_setopt(m_CurlMultiHandle, CURLMOPT_MAXCONNECTS, maxConnections);
    NN_SDK_ASSERT_EQUAL(mcode, CURLM_OK); NN_UNUSED(mcode);

    // 同時アクティブになる最大数、つまり maxConnections を使い果たしている間は直近に接続したホストしか Keep-Alive されない
    mcode = curl_multi_setopt(m_CurlMultiHandle, CURLMOPT_MAX_TOTAL_CONNECTIONS, maxConnections);
    NN_SDK_ASSERT_EQUAL(mcode, CURLM_OK); NN_UNUSED(mcode);
}

void ConnectionBroker::SetEnableClientCert(bool bEnable)
{
    m_bEnableClientCert = bEnable;
}

void ConnectionBroker::SetProxy(const char * pHostname, uint16_t port)
{
    std::strncpy(m_ProxyHostname, pHostname, sizeof(m_ProxyHostname) - 1);
    m_ProxyPort = port;
    m_bEnableManualProxy = true;
}

void ConnectionBroker::SetProxyAuthentication(const char * pUsername, const char * pPassword)
{
    util::Strlcpy<char>(m_ProxyUsername, pUsername ? pUsername : "", sizeof(m_ProxyUsername));
    util::Strlcpy<char>(m_ProxyPassword, pPassword ? pPassword : "", sizeof(m_ProxyPassword));
}

void ConnectionBroker::SetNoProxyHostsPointer(const char * pHostnames)
{
    m_pNoProxyHosts = pHostnames;
}

void ConnectionBroker::SetVerbose(bool bVerbose)
{
    m_bVerbose = bVerbose;
}

void ConnectionBroker::SetSkipSslVerificationForDebug(bool bSkip)
{
    m_bSkipSslVerificationForDebug = bSkip;
}

void ConnectionBroker::SetSocketBufferSize(size_t sizeSend, size_t sizeReceive)
{
#ifndef NN_BUILD_CONFIG_OS_WIN
    NN_SDK_REQUIRES_GREATER(static_cast<int>(sizeSend), 0); NN_SDK_REQUIRES_GREATER(static_cast<int>(sizeReceive), 0);
#endif
    m_nBufferSizeReceive = static_cast<uint32_t>(sizeReceive);
    m_nBufferSizeSend    = static_cast<uint32_t>(sizeSend);
}

Result ConnectionBroker::CurlWait(const TimeSpan & wait)
{
    NN_SDK_REQUIRES_NOT_NULL(m_CurlMultiHandle);

    nn::socket::FdSet fdread, fdwrite, fdexcep;
    int maxfd = -1;
    long timeoutMs = static_cast<long>(wait.GetMilliSeconds());

    nn::socket::FdSetZero(&fdread);
    nn::socket::FdSetZero(&fdwrite);
    nn::socket::FdSetZero(&fdexcep);

    {
        NN_HTTP_CURL_SYNCHRONIZED;

        CURLMcode mcode = curl_multi_fdset(m_CurlMultiHandle, &fdread, &fdwrite, &fdexcep, &maxfd);
        if (mcode != CURLM_OK)
        {
            NN_HTTP_ERROR("curl_multi_fdset() faile.(%s)\n", curl_multi_strerror(mcode));
            return ConvertCurlMultiCodeToResult(mcode);
        }

        mcode = curl_multi_timeout(m_CurlMultiHandle, &timeoutMs);
        if (mcode != CURLM_OK)
        {
            NN_HTTP_ERROR("curl_multi_timeout() failed.(%s)\n", curl_multi_strerror(mcode));
            return ConvertCurlMultiCodeToResult(mcode);
        }
    }

    int64_t effectiveWaitMs = std::min<int64_t>(timeoutMs, wait.GetMilliSeconds());
    if (maxfd >= 0)
    {
        nn::socket::TimeVal tv ={
            static_cast<int>(effectiveWaitMs / 1000),
            static_cast<int>(effectiveWaitMs % 1000) * 1000
        };
        nn::socket::Select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &tv);
    }
    else if (effectiveWaitMs > 0)
    {
        os::SleepThread(TimeSpan::FromMilliSeconds(effectiveWaitMs));
    }
    return ResultSuccess();
}

Result ConnectionBroker::CurlAddEasyHandleToMulti(CURL* pCurlHandle)
{
    NN_HTTP_CURL_SYNCHRONIZED;

    NN_SDK_REQUIRES(m_CurlMultiHandle != nullptr, "nn::http::ConnectionBroker::Initialize() was not called.");
    NN_SDK_REQUIRES_NOT_NULL(pCurlHandle);

    curl_easy_setopt(pCurlHandle, CURLOPT_PROXYAUTOCONFIG, m_bEnableManualProxy ? 0 : 1);
    if (m_bEnableManualProxy)
    {
        curl_easy_setopt(pCurlHandle, CURLOPT_PROXYTYPE, CURLPROXY_HTTP);
        curl_easy_setopt(pCurlHandle, CURLOPT_HTTPPROXYTUNNEL, 0L);
        curl_easy_setopt(pCurlHandle, CURLOPT_PROXYPORT, m_ProxyPort);
        CURLcode code = curl_easy_setopt(pCurlHandle, CURLOPT_PROXY, m_ProxyHostname);
        if (code != CURLE_OK)
        {
            NN_HTTP_ERROR("curl_easy_setopt(CURLOPT_PROXY) failed.(%s)\n", curl_easy_strerror(code));
            return ConvertCurlCodeToResult(code);
        }
        if (std::strcmp(m_ProxyUsername, "") != 0)
        {
            char userpass[sizeof(m_ProxyUsername) - 1 + sizeof(m_ProxyPassword) - 1 + 2];
            util::SNPrintf(userpass, sizeof(userpass), "%s:%s", m_ProxyUsername, m_ProxyPassword);
            code = curl_easy_setopt(pCurlHandle, CURLOPT_PROXYUSERPWD, userpass);
            if (code != CURLE_OK)
            {
                NN_HTTP_ERROR("curl_easy_setopt(CURLOPT_PROXYUSERPWD) failed.(%s)\n", curl_easy_strerror(code));
                return ConvertCurlCodeToResult(code);
            }
            curl_easy_setopt(pCurlHandle, CURLOPT_PROXYAUTH, CURLAUTH_BASIC);
        }
        if (m_pNoProxyHosts)
        {
            code = curl_easy_setopt(pCurlHandle, CURLOPT_NOPROXY, m_pNoProxyHosts);
            if (code != CURLE_OK)
            {
                NN_HTTP_ERROR("curl_easy_setopt(CURLOPT_NOPROXY) failed.(%s)\n", curl_easy_strerror(code));
                return ConvertCurlCodeToResult(code);
            }
        }
    }

    CURLMcode mcode = curl_multi_add_handle(m_CurlMultiHandle, pCurlHandle);
    return ConvertCurlMultiCodeToResult(mcode);
}

void ConnectionBroker::CurlRemoveEasyHandleFromMulti(CURL * pCurlHandle)
{
    NN_HTTP_CURL_SYNCHRONIZED;

    NN_SDK_REQUIRES_NOT_NULL(m_CurlMultiHandle);
    NN_SDK_REQUIRES_NOT_NULL(pCurlHandle);

    CURLMcode mcode = curl_multi_remove_handle(m_CurlMultiHandle, pCurlHandle);
    NN_SDK_ASSERT_EQUAL(mcode, CURLM_OK); NN_UNUSED(mcode);
}

Result ConnectionBroker::CurlPauseEasyHandle(CURL * pCurlHandle, int action)
{
    NN_HTTP_CURL_SYNCHRONIZED;

    NN_SDK_REQUIRES_NOT_NULL(pCurlHandle);

    CURLcode code = curl_easy_pause(pCurlHandle, action);
    return ConvertCurlCodeToResult(code);
}

void ConnectionBroker::ApplyDefaultOptions(Request & req)
{
    req.SetSkipSslVerificationForDebug(m_bSkipSslVerificationForDebug);
    req.SetVerbose(m_bVerbose);
}

CURLcode ConnectionBroker::CurlSslCtxFunction(CURL * pCurlHandle, void * pSslContextVoid, void * pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pCurlHandle); NN_SDK_ASSERT_NOT_NULL(pSslContextVoid);
    // INFO: Request が別でも実際には Keep-Alive で接続が使いまわされたり、
    //       pipe-lining で同じ接続になったりするので、SSL の設定を Request 別にできない
    ssl::Context* pContext = static_cast<nn::ssl::Context*>(pSslContextVoid);
    ConnectionBroker* pThis = static_cast<ConnectionBroker*>(pUserData);

    Result result = pThis->OnSslContextNeeded(pCurlHandle, *pContext);
    if (result.IsFailure())
    {
        return CURLE_SSL_CONNECT_ERROR;
    }

    return CURLE_OK;
}

curl_socket_t ConnectionBroker::CurlOpenSocketFunction(void* pUserData, curlsocktype purpose, curl_sockaddr * address)
{
    ConnectionBroker* pThis = static_cast<ConnectionBroker*>(pUserData);

    int fd = socket::Socket(static_cast<nn::socket::Family>(address->family),
                            static_cast<nn::socket::Type>(address->socktype),
                            static_cast<nn::socket::Protocol>(address->protocol));
    if (fd >= 0)
    {
        int val, ret;
        val = static_cast<int>(pThis->m_nBufferSizeReceive);
        ret = socket::SetSockOpt(fd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &val, sizeof(val));
        if (ret < 0)
        {
            NN_HTTP_ERROR("Failed to resize receive buffer of a socket to %d.(%d)\n", val, socket::GetLastError());
        }
        val = static_cast<int>(pThis->m_nBufferSizeSend);
        ret = socket::SetSockOpt(fd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &val, sizeof(val));
        if (ret < 0)
        {
            NN_HTTP_ERROR("Failed to resize send buffer of a socket to %d.(%d)\n", val, socket::GetLastError());
        }
#ifndef NN_BUILD_CONFIG_OS_WIN
        // Setting linger option so that sockets do not consume memory after being closed
        linger soLinger = {true,  1};
        ret = socket::SetSockOpt(fd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Nn_Linger, &soLinger, sizeof(soLinger));
        if (ret < 0)
        {
            NN_HTTP_ERROR("Failed to enable SO_NN_LINGER.(%d)\n", val, socket::GetLastError());
        }
#endif
    }
    else
    {
        NN_HTTP_ERROR("Failed to create socket descriptor.(%d)\n", socket::GetLastError());
    }
    return fd;
}

int ConnectionBroker::CurlSocketOptionFunction(void * pUserData, curl_socket_t curlfd, curlsocktype purpose)
{
    ConnectionBroker* pThis = static_cast<ConnectionBroker*>(pUserData);

    int val, ret;
    val = static_cast<int>(pThis->m_nBufferSizeReceive);
    ret = socket::SetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &val, sizeof(val));
    if (ret < 0)
    {
        NN_HTTP_ERROR("Failed to resize receive buffer of a socket to %d.(%d)\n", val, socket::GetLastError());
    }
    val = static_cast<int>(pThis->m_nBufferSizeSend);
    ret = socket::SetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &val, sizeof(val));
    if (ret < 0)
    {
        NN_HTTP_ERROR("Failed to resize send buffer of a socket to %d.(%d)\n", val, socket::GetLastError());
    }
    return CURL_SOCKOPT_OK ;
}

int ConnectionBroker::CurlDebugFunction(CURL * pCurlHandle, curl_infotype type, char * pData, size_t size, void * pUserData)
{
    NN_UNUSED(pUserData);

    const char *text;

    switch (type)
    {
    case CURLINFO_TEXT:
        NN_HTTP_TRACE_ALWAYS("== Info: %s", pData);
        return 0;
    case CURLINFO_HEADER_OUT:
        text = "=> Send header";
        break;
    case CURLINFO_DATA_OUT:
        text = "=> Send data";
        break;
    case CURLINFO_SSL_DATA_OUT:
        text = "=> Send SSL data";
        break;
    case CURLINFO_HEADER_IN:
        text = "<= Recv header";
        break;
    case CURLINFO_DATA_IN:
        text = "<= Recv data";
        break;
    case CURLINFO_SSL_DATA_IN:
        text = "<= Recv SSL data";
        break;
    default:
        NN_HTTP_WARN("Unexpected curl_infotype: %d\n", type);
        return 0;
    }

    Dump(text, (unsigned char *)pData, size);
    return 0;
}

Result ConnectionBroker::OnSslContextNeeded(CURL * pCurlHandle, nn::ssl::Context & context)
{
    Result result = context.Create(nn::ssl::Context::SslVersion_Auto);
    if (result.IsFailure())
    {
        NN_HTTP_ERROR("Failed to create SSL context.(%d)\n", result.GetInnerValueForDebug());
        return result;
    }

    if (m_bEnableClientCert)
    {
        ssl::CertStoreId deviceCertId;
        result = context.RegisterInternalPki(
            &deviceCertId,
            ssl::Context::InternalPki_DeviceClientCertDefault);
        if (result.IsFailure())
        {
            NN_HTTP_ERROR("Failed to register the device unique client certificate.(0x%08x)\n", result.GetInnerValueForDebug());
        }
    }
    return ResultSuccess();
}

Result ConnectionBroker::CommunicateThreadProcedure()
{
    Result result;

    int countStillRunning = 0;
    result = CurlPerform(countStillRunning);
    if (result.IsFailure())
    {
        return result;
    }
    ProcessCurlMessages();

    const TimeSpan wait = TimeSpan::FromMilliSeconds(100);
    if (countStillRunning > 0)
    {
        return CurlWait(wait);
    }
    else
    {
        os::SleepThread(wait);
        //NN_SDK_LOG("idle\n");
    }
    return ResultSuccess();
}

void ConnectionBroker::ProcessCurlMessages()
{
    NN_SDK_REQUIRES_NOT_NULL(m_CurlMultiHandle);

    int countMessage = 0;
    CURLMsg* pMessage;
    do {
        {
            NN_HTTP_CURL_SYNCHRONIZED;
            pMessage = curl_multi_info_read(m_CurlMultiHandle, &countMessage);
        }
        if (pMessage)
        {
            HandleCurlMessage(*pMessage);
        }
    } while (pMessage);
}

void ConnectionBroker::HandleCurlMessage(CURLMsg & message)
{
    ResponseImpl* pResponse = ResponseImpl::GetInstanceFromEasyHandle(message.easy_handle);
    NN_SDK_ASSERT_NOT_NULL(pResponse);
    switch (message.msg)
    {
    case CURLMSG_DONE:
        pResponse->NotifyCompletedWithCode(message.data.result);
        break;

    default:
        NN_SDK_ASSERT(false, "unknown CURLMsg");
        break;
    }
}

Result ConnectionBroker::CurlPerform(int & countStillRunning)
{
    NN_HTTP_CURL_SYNCHRONIZED;

    NN_SDK_REQUIRES_NOT_NULL(m_CurlMultiHandle);
    CURLMcode mcode;
    do {
//        NN_SDK_LOG("curl_multi_perform\n");
        mcode = curl_multi_perform(m_CurlMultiHandle, &countStillRunning);
    } while (mcode == CURLM_CALL_MULTI_PERFORM);

    return ConvertCurlMultiCodeToResult(mcode);
}

Result ConnectionBroker::PsuedoWaitCondition(nn::os::Mutex& mutex, nn::os::ConditionVariable& /*cond*/, os::Event* pCancelEvent)
{
    Result result;

    // 条件変数を待つふりをして、本来別スレッドでやるべきだった処理を行う
    if (m_mutexPesudoThread.TryLock())
    {
        result = CommunicateThreadProcedure();
        m_eventPesudoThreadLeave.Signal();
        m_mutexPesudoThread.Unlock();
    }
    else
    {
        // 他のスレッドが疑似スレッド処理を終えたら自スレッドでもやるべき
        mutex.Unlock();
        if (pCancelEvent)
        {
            if (os::WaitAny(m_eventPesudoThreadLeave.GetBase(), pCancelEvent->GetBase()) == 0)
            {
                m_eventPesudoThreadLeave.Clear();
            }
            // pCancelEvent はクリアしない
        }
        else
        {
            m_eventPesudoThreadLeave.Wait();
        }
        mutex.Lock();
        return ResultSuccess();
    }
    return result;
}

}
} // ~namespace nn::http
