﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_SdkThreadLocalStorage.h>
#include <nn/htc.h>
#include <nn/htcs/htcs_Library.h>
#include <nn/htcs/htcs_LibraryPrivate.h>
#include <nn/htcs/htcs_Socket.h>
#include <nn/htcs/htcs_Result.h>
#include <nn/htcs/htcs_Config.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/util/util_StringUtil.h>
#include "htcs_Const.h"
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include "../tmagent/htcs/htcs_HtcsManager.h"
#else
#include "htcs_HtcsManagerByHipc.h"
#endif
#include <cstdlib>
#include <nn/nn_SdkLog.h>
#include "htcs_VirtualSocketCollection.h"
#include <nn/nn_Log.h>
#include <nn/os/os_TransferMemory.h>

#ifndef NN_LOG
#define TRACE_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#else
#define TRACE_LOG(...) NN_LOG(__VA_ARGS__)
#endif

#undef  TRACE_LOG
#define TRACE_LOG(...)

#define API_TRACE( channel, ... ) //NN_SDK_LOG( "[" ); NN_SDK_LOG( channel ); NN_SDK_LOG( "] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )

namespace nn { namespace htcs {

namespace {

    bool g_IsInitialized = false;

    bool g_EnableDisconnectionEmulation = true;

#if 0
    bool g_UsedDefaultAllocator = false;

    void* DefaultAllocate(size_t size) NN_NOEXCEPT
    {
        g_UsedDefaultAllocator = true;
        auto p = malloc(size);
        // auto p = ::operator new(size, std::nothrow);
        if( p == nullptr )
        {
            NN_SDK_LOG(
                "[fs] malloc() returned nullptr. "
                "Refer an API reference of nn::fs::SetAllocator(). \n"
                );
        }
        return p;
    }

    void DefaultDeallocate(void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        free(p);
        // ::operator delete(p);
    }
#endif

    AllocateFunction s_AllocateFunction = nullptr;
    DeallocateFunction s_DeallocateFunction = nullptr;
    void* g_Buffer = nullptr;
    size_t g_BufferSize = 0;

    // TORIAEZU: GetLastError に TLS を使用する
    nn::os::TlsSlot g_TlsSlot;

    nn::tma::IHtcsManager* g_Manager;
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    nn::tma::IHtcsManager* g_Monitor;
    nn::sf::HipcSimpleClientSessionManager g_SessionManager;
#endif

#if USE_VIRTUAL_SOCKETS
    detail::virtual_socket_collection* pSockets = NULL;
#else

    struct StaticMutex
    {
        os::MutexType    m_Mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex( &m_Mutex );
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex( &m_Mutex );
        }
    };
    StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    struct Descriptor
    {
        nn::sf::SharedPointer<nn::tma::ISocket> socket;
        uint16_t sequence;
    };

    static const int DescriptorArrayCountMax = 32;
    std::array<Descriptor, DescriptorArrayCountMax> g_DescriptorArray;
    uint16_t g_Sequence = 0;

    int32_t RegisterSocket( nn::sf::SharedPointer<nn::tma::ISocket> socket )
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);
        for (int i = 0; i < DescriptorArrayCountMax; i++)
        {
            if (g_DescriptorArray[i].socket == nullptr)
            {
                g_DescriptorArray[i].socket = socket;
                g_DescriptorArray[i].sequence = ++g_Sequence;
                int32_t ret = i | (g_DescriptorArray[i].sequence << 16);
                return ret;
            }
        }
        return -1;
    }

    nn::sf::SharedPointer<nn::tma::ISocket> GetSocket( int32_t descriptor )
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        int32_t index = descriptor & 0xFFFF;
        uint16_t sequence = descriptor >> 16;

        if (index >= DescriptorArrayCountMax)
        {
            return nullptr;
        }

        if (g_DescriptorArray[index].sequence == sequence)
        {
            return g_DescriptorArray[index].socket;
        }
        return nullptr;
    }

    void UnregisterSocket( int32_t descriptor )
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        int32_t index = descriptor & 0xFFFF;
        uint16_t sequence = descriptor >> 16;

        if (index >= DescriptorArrayCountMax)
        {
            return;
        }

        if (g_DescriptorArray[index].sequence == sequence)
        {
            g_DescriptorArray[index].socket = nullptr;
            g_DescriptorArray[index].sequence = 0;
        }
    }

#endif
    void SetLastError( uintptr_t errorCode )
    {
        nn::os::SetTlsValue( g_TlsSlot, errorCode );
    }

    void InitializeImpl(void* buffer, size_t bufferSize, int sessionCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(sessionCount > 0);
        NN_SDK_REQUIRES(sessionCount <= nn::htcs::SessionCountMax);

#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(sessionCount);
        nn::htc::Initialize();
        g_Manager = nn::tma::CreateHtcsManager().Detach();
#else
        g_Monitor = nn::htcs::CreateHtcsManagerByHipc().Detach();
        g_Manager = nn::htcs::CreateHtcsManagerByHipc(&g_SessionManager).Detach();
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SessionManager.SetSessionCount(sessionCount));

        // TORIAEZU: リソースリークの可能性があるが、暫定であるため、これを許容する。
        //   RegisterProcessId の呼び出しから MonitorManager の呼び出しまでの間に、
        //   クライアントプロセスが終了した場合、サーバ側のリソースがリークする。
        //   ただし、稀なケースであり、狙って起こすことも難しいと考えられるため、
        //   暫定対処の現状では、これを許容することにする。
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->RegisterProcessId(0));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Monitor->MonitorManager(0));
#endif
        nn::os::SdkAllocateTlsSlot(&g_TlsSlot, NULL);

#if USE_VIRTUAL_SOCKETS
        NN_SDK_ASSERT(buffer);
        char* pCur = static_cast<char*>(buffer);
        pSockets = new (pCur) detail::virtual_socket_collection();
        pCur += sizeof(detail::virtual_socket_collection);
        pSockets->Init(pCur, bufferSize - sizeof(detail::virtual_socket_collection));
#endif

        g_IsInitialized = true;
    }

    void InitializeImpl(AllocateFunction alloc, DeallocateFunction dealloc, int sessionCount, int socketCountMax) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(sessionCount > 0);
        NN_SDK_REQUIRES(sessionCount <= nn::htcs::SessionCountMax);

        s_AllocateFunction = alloc;
        s_DeallocateFunction = dealloc;

        g_BufferSize = sizeof(detail::virtual_socket_collection) + detail::virtual_socket_collection::GetWorkingMemorySize(socketCountMax);
        g_Buffer = s_AllocateFunction(g_BufferSize);
        InitializeImpl(g_Buffer, g_BufferSize, sessionCount);
    }
}

void Initialize(AllocateFunction alloc, DeallocateFunction dealloc) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    g_EnableDisconnectionEmulation = true;
    InitializeImpl(alloc, dealloc, nn::htcs::SessionCountMax, nn::htcs::SocketCountMax);
}

void Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    g_EnableDisconnectionEmulation = true;
    InitializeImpl(buffer, bufferSize, nn::htcs::SessionCountMax);
}

void Initialize(AllocateFunction alloc, DeallocateFunction dealloc, int sessionCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    g_EnableDisconnectionEmulation = true;
    InitializeImpl(alloc, dealloc, sessionCount, nn::htcs::SocketCountMax);
}

void InitializeForDisableDisconnectionEmulation(AllocateFunction alloc, DeallocateFunction dealloc, int sessionCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    g_EnableDisconnectionEmulation = false;
    InitializeImpl(alloc, dealloc, sessionCount, nn::htcs::SocketCountMax);
}

void InitializeForSystem(void* buffer, size_t bufferSize, int sessionCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    g_EnableDisconnectionEmulation = true;
    InitializeImpl(buffer, bufferSize, sessionCount);
}

void InitializeForDisableDisconnectionEmulation(void* buffer, size_t bufferSize, int sessionCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    g_EnableDisconnectionEmulation = false;

    InitializeImpl(buffer, bufferSize, sessionCount);
}

bool IsInitialized() NN_NOEXCEPT
{
    return g_IsInitialized;
}

void Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    g_IsInitialized = false;

#if USE_VIRTUAL_SOCKETS
    pSockets->~virtual_socket_collection();
    pSockets = nullptr;
    if (g_Buffer)
    {
        s_DeallocateFunction(g_Buffer, g_BufferSize);
        g_Buffer = nullptr;
        g_BufferSize = 0;
    }
#else
    Descriptor tmpDescriptor;
    tmpDescriptor.socket = nullptr;
    tmpDescriptor.sequence = 0;
    g_DescriptorArray.fill(tmpDescriptor);
#endif

    nn::os::FreeTlsSlot( g_TlsSlot );
    nn::sf::ReleaseSharedObject(g_Manager);
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::htc::Finalize();
#else
    nn::sf::ReleaseSharedObject(g_Monitor);
    g_SessionManager.Finalize();
#endif

}

size_t GetWorkingMemorySize(int socketCountMax) NN_NOEXCEPT
{
#if USE_VIRTUAL_SOCKETS
    NN_SDK_REQUIRES(socketCountMax <= nn::htcs::SocketCountMax);
    return sizeof(detail::virtual_socket_collection)
        + detail::virtual_socket_collection::GetWorkingMemorySize(socketCountMax);
#else
    return 0;
#endif
}

//==============================================================================

#if USE_VIRTUAL_SOCKETS

int Socket() NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );
    int errorCode = 0;
    int ret = pSockets->Socket( errorCode );
    if( ret < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return ret;
}

//==============================================================================

int Close(int descriptor) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    int intResult = pSockets->Close( descriptor, errorCode );
    if( intResult < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

//==============================================================================

int Accept(int descriptor, struct SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    nn::htcs::SockAddrHtcs tmpAddr;
    if( !address )
    {
        address = &tmpAddr;
    }

    int errorCode = 0;
    int ret = pSockets->Accept( descriptor, address, errorCode );

    if( ret < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return ret;
}

//==============================================================================

int Bind(int descriptor, const struct SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES(address->family == HTCS_AF_HTCS);

    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    int intResult = pSockets->Bind( descriptor, address, errorCode );
    if( intResult < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

//==============================================================================

int Connect(int descriptor, const struct SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES(address->family == HTCS_AF_HTCS);

    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    int intResult = pSockets->Connect( descriptor, address, errorCode );

    if( intResult < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

//==============================================================================

int Listen(int descriptor, int backlogCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    int intResult = pSockets->Listen( descriptor, backlogCount, errorCode );

    if( intResult < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

//==============================================================================

ssize_t Recv(int descriptor, void *buf, size_t len, int flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buf);

    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    nn::htcs::ssize_t receivedSize = pSockets->Recv( descriptor, buf, len, flags, errorCode );
    if( receivedSize < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }
    return receivedSize;
}

//==============================================================================

ssize_t Send(int descriptor, const void *buf, size_t len, int flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buf);

    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    nn::htcs::ssize_t sentSize = pSockets->Send( descriptor, buf, len, flags, errorCode );
    if( sentSize < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return sentSize;
}

//==============================================================================

int Shutdown(int descriptor, int how) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    int intResult = pSockets->Shutdown( descriptor, how, errorCode );

    if( intResult < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

//==============================================================================

int Fcntl(int descriptor, int cmd, int val) NN_NOEXCEPT
{
    NN_SDK_ASSERT( g_Manager );
    NN_SDK_ASSERT( pSockets );

    int errorCode = 0;
    int intResult = pSockets->Fcntl( descriptor, cmd, val, errorCode );

    if( intResult < 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

//============================================================================

namespace detail {

//============================================================================

nn::sf::SharedPointer<nn::tma::ISocket> socket( int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "socket", "socket" );
    nn::sf::SharedPointer<nn::tma::ISocket> socket = nullptr;
#if 0
    static int sNumberOfSkips = 20;
    if( sNumberOfSkips > 0 )
    {
        LastError = HTCS_ENETDOWN;
        sNumberOfSkips -= 1;
        return -1;
    }
#endif
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->CreateSocket(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>>(&socket), g_EnableDisconnectionEmulation));

    return socket;

}

//============================================================================

int close( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "close", "close socket %p", descriptor );
    int32_t Ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(descriptor->Close(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int32_t>(&Ret)));

    return Ret;
}

//============================================================================

int connect( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const nn::htcs::SockAddrHtcs* address, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "connect", "connect socket %p", descriptor );

    // ヌル終端されていない場合があるので、強制的にヌル終端
    SockAddrHtcs nullTerminatedAddress;
    nullTerminatedAddress.family = address->family;
    nn::util::Strlcpy(nullTerminatedAddress.peerName.name, address->peerName.name, sizeof(nullTerminatedAddress.peerName.name));
    nn::util::Strlcpy(nullTerminatedAddress.portName.name, address->portName.name, sizeof(nullTerminatedAddress.portName.name));

    int32_t Ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(descriptor->Connect(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int32_t>(&Ret), nullTerminatedAddress));

    return Ret;
}

//============================================================================

int bind( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const nn::htcs::SockAddrHtcs* address, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "bind", "bind socket %p", descriptor );

    // ヌル終端されていない場合があるので、強制的にヌル終端
    SockAddrHtcs nullTerminatedAddress;
    nullTerminatedAddress.family = address->family;
    nn::util::Strlcpy(nullTerminatedAddress.peerName.name, address->peerName.name, sizeof(nullTerminatedAddress.peerName.name));
    nn::util::Strlcpy(nullTerminatedAddress.portName.name, address->portName.name, sizeof(nullTerminatedAddress.portName.name));

    int32_t Ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(descriptor->Bind(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int32_t>(&Ret), nullTerminatedAddress));
    return Ret;
}

//============================================================================

int listen( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int backlogCount, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "listen", "listen socket %p", descriptor );
    int32_t Ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(descriptor->Listen(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int32_t>(&Ret), backlogCount));
    return Ret;
}

//============================================================================

nn::sf::SharedPointer<nn::tma::ISocket> accept( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, nn::htcs::SockAddrHtcs* address, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "accept", "accept socket %p", descriptor );
    nn::sf::SharedPointer<nn::tma::ISocket> Ret = nullptr;
    nn::sf::NativeHandle waitHandle;
    uint32_t taskId;

    //TRACE_LOG("[libnn_htcs] [accept] START %p\n", address );
    nn::Result res = descriptor->AcceptStart( nn::sf::Out<uint32_t>(&taskId), nn::sf::Out<nn::sf::NativeHandle>(&waitHandle) );
    if( res.IsFailure() )
    {
        LastError = HTCS_EINTR;

        //============================================================================
        // SIGLO-80591 (1 of 5), When the testHtcs_SpecialCase test is executed on EDEV, the tma process runs away.
        // - Prevent the CPU from getting tied up in a fast loop - preventing TMA's other threads from getting CPU cycles.
        //============================================================================
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 1 ) );
        TRACE_LOG("[libnn_htcs] [accept] Socket %p AcceptStart FAILED - 0x%x\n", address, res.GetInnerValueForDebug() );
    }
    else
    {
        nn::os::SystemEventType waitEvent;
        nn::os::AttachReadableHandleToSystemEvent( &waitEvent, waitHandle.GetOsHandle(), waitHandle.IsManaged(), nn::os::EventClearMode_ManualClear );
        waitHandle.Detach();

        //TRACE_LOG("[libnn_htcs] Waiting until signalled to call AcceptResults for task %d\n", taskId );
        nn::os::WaitSystemEvent( &waitEvent );

        descriptor->AcceptResults(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>>(&Ret), nn::sf::Out<nn::htcs::SockAddrHtcs>(address), taskId );

        nn::os::DestroySystemEvent( &waitEvent );

        //TRACE_LOG("[libnn_htcs] [accept] Socket %p COMPLETED, LastError %d, task %d\n", address, LastError, taskId );
    }

    return Ret;
}

//============================================================================

enum
{
    TransferAddressMask = ~(nn::os::MemoryPageSize - 1),
    //=========================================================================
    // Apologies for this magic number - This is the amount of data that can
    // fit in one of the packets that's sent from TMA to TMS.  This is the
    // most we can send without having to wait in a thread.
    //=========================================================================
    MaxTMSPacketDataSize = (56 * 1024),
    ReservedForAdditionalPacketData = (sizeof( int32_t ) * 8),  // Reserve some space for the system to use
    MaxBufferSizeWithoutMapping = (MaxTMSPacketDataSize - ReservedForAdditionalPacketData),
};

//============================================================================

nn::htcs::ssize_t recvLarge( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, void* buffer, size_t recvSize, int flags, int& LastError ) NN_NOEXCEPT
{
    // Set up our defaults
    int64_t Ret = -1;
    LastError = HTCS_EINTR;

    // Calculate all our interesting sizes and offsets
    int64_t srcAddress = (int64_t)buffer;
    int64_t srcEndAddress = (int64_t)srcAddress + recvSize;
    int64_t startTranferAddress = ( srcAddress & TransferAddressMask ) + nn::os::MemoryPageSize;
    int64_t endTransferAddress = ( srcEndAddress & TransferAddressMask );
    int32_t preTransferSize = (int32_t)( startTranferAddress - srcAddress );
    int64_t alignedTransferSize = endTransferAddress - startTranferAddress;
    int32_t postTransferSize = (int32_t)( srcEndAddress - endTransferAddress );

    {
        nn::os::TransferMemoryType transferMemory;
        nn::Result createRes = nn::os::CreateTransferMemory( &transferMemory, (void*)startTranferAddress, (size_t)alignedTransferSize, ::nn::os::MemoryPermission_None );
        if( createRes.IsFailure() )
        {
            //TRACE_LOG("[libnn_htcs] [recv] CreateTransferMemory FAILED 0x%x\n", createRes.GetInnerValueForDebug() );
        }
        else
        {
            nn::os::NativeHandle transferMemoryHandle = ::nn::os::DetachTransferMemory( &transferMemory );
            nn::os::DestroyTransferMemory( &transferMemory );

            //=========================================================================================
            // Start the recv
            //=========================================================================================
            uint32_t taskId = 0;
            nn::sf::NativeHandle waitHandle;
            nn::sf::NativeHandle transferHandle(transferMemoryHandle, true);
            nn::Result res = descriptor->RecvLargeStart( nn::sf::Out<uint32_t>(&taskId), nn::sf::Out<nn::sf::NativeHandle>(&waitHandle),
                preTransferSize, postTransferSize, alignedTransferSize, std::move(transferHandle), flags );

            transferHandle.Detach();

            // Success??
            if( res.IsFailure() )
            {
                LastError = HTCS_EINTR;
                //============================================================================
                // SIGLO-80591 (2 of 5), When the testHtcs_SpecialCase test is executed on EDEV, the tma process runs away.
                // - Prevent the CPU from getting tied up in a fast loop - preventing TMA's other threads from getting CPU cycles.
                //============================================================================
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 1 ) );
                TRACE_LOG("[libnn_htcs] [recv] RecvLargeStart FAILED - 0x%x\n", res.GetInnerValueForDebug() );
            }
            else
            {
                nn::os::SystemEventType waitEvent;
                nn::os::AttachReadableHandleToSystemEvent( &waitEvent, waitHandle.GetOsHandle(), waitHandle.IsManaged(), nn::os::EventClearMode_ManualClear );
                waitHandle.Detach();

                nn::os::WaitSystemEvent( &waitEvent );

                descriptor->RecvResults( nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int64_t>(&Ret), nn::sf::OutBuffer(static_cast<char*>(buffer), recvSize), taskId );
                nn::os::DestroySystemEvent( &waitEvent );
            }
        }
    }

    //TRACE_LOG("[libnn_htcs] [recv] COMPLETE, returning %d\n", Ret );
    return static_cast<nn::htcs::ssize_t>(Ret);
}

//============================================================================

nn::htcs::ssize_t recv( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, void* buffer, size_t bufferByteSize, int flags, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "recv", "recv socket %p", descriptor );
    int64_t Ret = -1;

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_ABORT_UNLESS_RESULT_SUCCESS( descriptor->Recv( nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int64_t>(&Ret),  nn::sf::OutBuffer(static_cast<char*>(buffer), bufferByteSize), flags ) );
#else

    //TRACE_LOG("[libnn_htcs] [recv] read %d bytes.\n", bufferByteSize );

    size_t recvSize = bufferByteSize;

    //==========================================================================
    // If this is a non-blocking call, we will never receive more than this
    //==========================================================================
    if( (flags & nn::htcs::HTCS_MSG_WAITALL) == 0 )
    {
        recvSize = ( bufferByteSize > MaxBufferSizeWithoutMapping ) ? MaxBufferSizeWithoutMapping : bufferByteSize;
    }


    if( recvSize > MaxBufferSizeWithoutMapping )
    {
        return recvLarge( descriptor, buffer, recvSize, flags, LastError );
    }
    else
    {
        nn::sf::NativeHandle waitHandle;
        uint32_t taskId = 0;

        nn::Result res = descriptor->RecvStart( nn::sf::Out<uint32_t>(&taskId), nn::sf::Out<nn::sf::NativeHandle>(&waitHandle), (int32_t)recvSize, flags );
        if( res.IsFailure() )
        {
            LastError = HTCS_EINTR;
            //============================================================================
            // SIGLO-80591 (3 of 5), When the testHtcs_SpecialCase test is executed on EDEV, the tma process runs away.
            // - Prevent the CPU from getting tied up in a fast loop - preventing TMA's other threads from getting CPU cycles.
            //============================================================================
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 1 ) );

            TRACE_LOG("[libnn_htcs] [recv] RecvStart FAILED - 0x%x\n", res.GetInnerValueForDebug() );
        }
        else
        {
            nn::os::SystemEventType waitEvent;
            nn::os::AttachReadableHandleToSystemEvent( &waitEvent, waitHandle.GetOsHandle(), waitHandle.IsManaged(), nn::os::EventClearMode_ManualClear );
            waitHandle.Detach();

            nn::os::WaitSystemEvent( &waitEvent );

            descriptor->RecvResults( nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int64_t>(&Ret), nn::sf::OutBuffer(static_cast<char*>(buffer), recvSize), taskId );
            nn::os::DestroySystemEvent( &waitEvent );
        }
    }

    //TRACE_LOG("[libnn_htcs] recv COMPLETE\n" );
#endif

    return static_cast<nn::htcs::ssize_t>(Ret);
}

//============================================================================

nn::htcs::ssize_t sendLarge( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const void* buffer, size_t bufferByteSize, int flags, int& LastError ) NN_NOEXCEPT
{
    // Set up our defaults
    int64_t Ret = -1;
    LastError = HTCS_EINTR;

    // Calculate all our interesting sizes and offsets
    int64_t srcAddress = (int64_t)buffer;
    int64_t srcEndAddress = (int64_t)srcAddress + bufferByteSize;
    int64_t startTranferAddress = ( srcAddress & TransferAddressMask ) + nn::os::MemoryPageSize;
    int64_t endTransferAddress = ( srcEndAddress & TransferAddressMask );
    int32_t preTransferSize = (int32_t)( startTranferAddress - srcAddress );
    int64_t alignedTransferSize = endTransferAddress - startTranferAddress;
    int32_t postTransferSize = (int32_t)( srcEndAddress - endTransferAddress );

    // Transfer the aligned portion of this data via the transfer memory functionality
    nn::os::TransferMemoryType transferMemory;
    nn::Result createRes = nn::os::CreateTransferMemory( &transferMemory, (void*)startTranferAddress, (size_t)alignedTransferSize, ::nn::os::MemoryPermission_None );
    if( createRes.IsFailure() )
    {
        TRACE_LOG("[libnn_htcs] [send] CreateTransferMemory FAILED 0x%x\n", createRes.GetInnerValueForDebug() );
    }
    else
    {
        nn::os::NativeHandle transferMemoryHandle = ::nn::os::DetachTransferMemory( &transferMemory );
        nn::os::DestroyTransferMemory( &transferMemory );

        //=========================================================================================
        // Start the send
        //=========================================================================================
        void* pUnalignStart = (void*)srcAddress;
        void* pUnalignEnd = (void*)endTransferAddress;
        uint32_t taskId = 0;
        nn::sf::NativeHandle waitHandle;
        nn::sf::NativeHandle transferHandle(transferMemoryHandle, true);
        nn::Result res = descriptor->SendLargeStart( nn::sf::Out<uint32_t>(&taskId), nn::sf::Out<nn::sf::NativeHandle>(&waitHandle),
            nn::sf::InBuffer(static_cast<const char*>(pUnalignStart), preTransferSize ), nn::sf::InBuffer(static_cast<const char*>(pUnalignEnd), postTransferSize ),
            std::move(transferHandle), alignedTransferSize, flags );

        transferHandle.Detach();

        // Success??
        if( res.IsFailure() )
        {
            LastError = HTCS_EINTR;
            //============================================================================
            // SIGLO-80591 (4 of 5), When the testHtcs_SpecialCase test is executed on EDEV, the tma process runs away.
            // - Prevent the CPU from getting tied up in a fast loop - preventing TMA's other threads from getting CPU cycles.
            //============================================================================
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 1 ) );

            TRACE_LOG("[libnn_htcs] [send] SendLargeStart FAILED - 0x%x\n", res.GetInnerValueForDebug() );
        }
        else
        {
            nn::os::SystemEventType waitEvent;
            nn::os::AttachReadableHandleToSystemEvent( &waitEvent, waitHandle.GetOsHandle(), waitHandle.IsManaged(), nn::os::EventClearMode_ManualClear );
            waitHandle.Detach();

            nn::os::WaitSystemEvent( &waitEvent );

            descriptor->SendResults( nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int64_t>(&Ret), taskId );
            nn::os::DestroySystemEvent( &waitEvent );
        }
    }

    //TRACE_LOG("[libnn_htcs] [send] COMPLETE, returning %d\n", Ret );
    return static_cast<nn::htcs::ssize_t>(Ret);
}

//============================================================================

nn::htcs::ssize_t send( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const void* buffer, size_t bufferByteSize, int flags, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "send", "send socket %p", descriptor );
    int64_t Ret = -1;

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_ABORT_UNLESS_RESULT_SUCCESS( descriptor->Send( nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int64_t>(&Ret),  nn::sf::InBuffer(static_cast<const char*>(buffer), bufferByteSize), flags ) );
#else
    if( bufferByteSize > MaxBufferSizeWithoutMapping )
    {
        return sendLarge( descriptor, buffer, bufferByteSize, flags, LastError );
    }

    nn::sf::NativeHandle waitHandle;
    uint32_t taskId = 0;
    nn::Result res = descriptor->SendStart( nn::sf::Out<uint32_t>(&taskId), nn::sf::Out<nn::sf::NativeHandle>(&waitHandle), nn::sf::InBuffer(static_cast<const char*>(buffer), bufferByteSize), flags );

    if( res.IsFailure() )
    {
        LastError = HTCS_EINTR;
        //============================================================================
        // SIGLO-80591 (5 of 5), When the testHtcs_SpecialCase test is executed on EDEV, the tma process runs away.
        // - Prevent the CPU from getting tied up in a fast loop - preventing TMA's other threads from getting CPU cycles.
        //============================================================================
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 1 ) );

        TRACE_LOG("[libnn_htcs] [send] SendStart FAILED - 0x%x\n", res.GetInnerValueForDebug() );
    }
    else
    {
        nn::os::SystemEventType waitEvent;
        nn::os::AttachReadableHandleToSystemEvent( &waitEvent, waitHandle.GetOsHandle(), waitHandle.IsManaged(), nn::os::EventClearMode_ManualClear );
        waitHandle.Detach();

        nn::os::WaitSystemEvent( &waitEvent );

        descriptor->SendResults( nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int64_t>(&Ret), taskId );
        nn::os::DestroySystemEvent( &waitEvent );
    }

    //TRACE_LOG("[libnn_htcs] [send] COMPLETE, returning %d\n", Ret );
#endif
    return static_cast<nn::htcs::ssize_t>(Ret);
}

//============================================================================

int shutdown( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int how, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "shutdown", "shutdown socket %p", descriptor );
    int32_t Ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(descriptor->Shutdown(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int32_t>(&Ret), how));
    return Ret;
}

//============================================================================

int fcntl( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int command, int value, int& LastError ) NN_NOEXCEPT
{
    API_TRACE( "fcntl", "fcntl socket %p", descriptor );
    int32_t Ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(descriptor->Fcntl(nn::sf::Out<int32_t>(&LastError), nn::sf::Out<int32_t>(&Ret), command, value));
    return Ret;
}

//============================================================================

//============================================================================
} //namespace detail
//============================================================================
#else

int Socket() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    nn::sf::SharedPointer<nn::tma::ISocket> socket;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->CreateSocket(&errorCode, &socket, g_EnableDisconnectionEmulation));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
        return -1;
    }

    int ret = RegisterSocket(socket);
    if( ret == -1 )
    {
        SetLastError( nn::htcs::HTCS_EMFILE );
    }
    return ret;
}

int Close(int descriptor) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int intResult;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Close(&errorCode, nn::sf::Out<int32_t>(&intResult)));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    UnregisterSocket(descriptor);

    return intResult;
}

int Accept(int descriptor, struct SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    nn::sf::SharedPointer<nn::tma::ISocket> outSocket;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    nn::htcs::SockAddrHtcs tmpAddr;
    if( !address )
    {
        address = &tmpAddr;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Accept(&errorCode, &outSocket, nn::sf::Out<nn::htcs::SockAddrHtcs>(address)));

    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
        return -1;
    }

    int ret = RegisterSocket(outSocket);
    if( ret == -1 )
    {
        SetLastError( nn::htcs::HTCS_EMFILE );
    }
    return ret;
}

int Bind(int descriptor, const struct SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES(address->family == HTCS_AF_HTCS);

    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int intResult;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    // ヌル終端されていない場合があるので、強制的にヌル終端
    SockAddrHtcs nullTerminatedAddress;
    nullTerminatedAddress.family = address->family;
    nn::util::Strlcpy(nullTerminatedAddress.peerName.name, address->peerName.name, sizeof(nullTerminatedAddress.peerName.name));
    nn::util::Strlcpy(nullTerminatedAddress.portName.name, address->portName.name, sizeof(nullTerminatedAddress.portName.name));

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Bind(&errorCode, nn::sf::Out<int32_t>(&intResult), nullTerminatedAddress));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

int Connect(int descriptor, const struct SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES(address->family == HTCS_AF_HTCS);

    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int intResult;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    // ヌル終端されていない場合があるので、強制的にヌル終端
    SockAddrHtcs nullTerminatedAddress;
    nullTerminatedAddress.family = address->family;
    nn::util::Strlcpy(nullTerminatedAddress.peerName.name, address->peerName.name, sizeof(nullTerminatedAddress.peerName.name));
    nn::util::Strlcpy(nullTerminatedAddress.portName.name, address->portName.name, sizeof(nullTerminatedAddress.portName.name));

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Connect(&errorCode, nn::sf::Out<int32_t>(&intResult), nullTerminatedAddress));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

int Listen(int descriptor, int backlogCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int intResult;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Listen(&errorCode, nn::sf::Out<int32_t>(&intResult), backlogCount));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

ssize_t Recv(int descriptor, void *buf, size_t len, int flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buf);

    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int64_t receivedSize;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Recv(&errorCode, nn::sf::Out<int64_t>(&receivedSize), nn::sf::OutBuffer(static_cast<char*>(buf), len), flags));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return static_cast<nn::htcs::ssize_t>(receivedSize);
}

ssize_t Send(int descriptor, const void *buf, size_t len, int flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buf);

    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int64_t sentSize;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Send(&errorCode, nn::sf::Out<int64_t>(&sentSize), nn::sf::InBuffer(static_cast<const char*>(buf), len), flags));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return static_cast<nn::htcs::ssize_t>(sentSize);
}

int Shutdown(int descriptor, int how) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int intResult;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Shutdown(&errorCode, nn::sf::Out<int32_t>(&intResult), how));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

int Fcntl(int descriptor, int cmd, int val) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int errorCode;
    int intResult;

    auto socket = GetSocket(descriptor);
    if( !socket )
    {
        SetLastError( nn::htcs::HTCS_EBADF);
        return -1;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(socket->Fcntl(&errorCode, nn::sf::Out<int32_t>(&intResult), cmd, val));
    if( errorCode != 0 )
    {
        SetLastError( static_cast<uintptr_t>(errorCode) );
    }

    return intResult;
}

#endif

//==============================================================================

int GetLastError() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);
    return static_cast<int>(nn::os::GetTlsValue( g_TlsSlot ));
}

//==============================================================================

const HtcsPeerName GetPeerNameAny() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    nn::htcs::HtcsPeerName name;
    g_Manager->GetPeerNameAny(nn::sf::Out<nn::htcs::HtcsPeerName>(&name));

    return name;
}

/**
 * @brief HtcsPeerName に規定のホスト PC を指定する場合に使用します。
 */
const HtcsPeerName GetDefaultHostName() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    nn::htcs::HtcsPeerName name;
    g_Manager->GetDefaultHostName(nn::sf::Out<nn::htcs::HtcsPeerName>(&name));

    return name;
}

//==============================================================================
}} //namespace nn { namespace htcs {
//==============================================================================

