﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>

#include "../htclow_Packet.h"
#include "../htclow_PacketFactory.h"
#include "htclow_ReceiveBuffer.h"
#include "htclow_SendBuffer.h"

namespace nn { namespace htclow { namespace server { namespace mux {

enum class ChannelState
{
    Opened,
    SynSent,
    Established,
    CloseWait,
    LastAck,
    FinWait1,
    FinWait2,
    Closing,
    Closed,
};

class ChannelResource
{
    NN_DISALLOW_COPY(ChannelResource);
    NN_DISALLOW_MOVE(ChannelResource);

public:
    explicit ChannelResource(bool retransmitEnabled) NN_NOEXCEPT
        : m_Version(MaxVersion)
        , m_State(ChannelState::Opened)
        , m_ClosedEvent(nn::os::EventClearMode_ManualClear, true)
        , m_EstablishedEvent(nn::os::EventClearMode_ManualClear, true)
        , m_SendBuffer(retransmitEnabled)
    {
    }

    int16_t GetVersion() NN_NOEXCEPT
    {
        return m_Version;
    }

    ChannelState GetState() NN_NOEXCEPT
    {
        return m_State;
    }

    nn::os::SystemEvent* GetClosedEvent() NN_NOEXCEPT
    {
        return &m_ClosedEvent;
    }

    nn::os::SystemEvent* GetEstablishedEvent() NN_NOEXCEPT
    {
        return &m_EstablishedEvent;
    }

    nn::os::SystemEvent* GetReceiveReadyEvent() NN_NOEXCEPT
    {
        return m_ReceiveBuffer.GetReceiveReadyEvent();
    }

    nn::os::SystemEvent* GetSendCompleteEvent() NN_NOEXCEPT
    {
        return m_SendBuffer.GetEmptyEvent();
    }

    SendPacketFactory* GetSendPacketFactory() NN_NOEXCEPT
    {
        return &m_SendPacketFactory;
    }

    SendBuffer* GetSendBuffer() NN_NOEXCEPT
    {
        return &m_SendBuffer;
    }

    ReceiveBuffer* GetReceiveBuffer() NN_NOEXCEPT
    {
        return &m_ReceiveBuffer;
    }

    Result CheckPacketVersion(BasePacket* packet) NN_NOEXCEPT;
    void SetState(ChannelState state) NN_NOEXCEPT;
    void SetVersion(int16_t version) NN_NOEXCEPT;

private:
    bool IsStateTransitionAllowed(ChannelState current, ChannelState next) NN_NOEXCEPT;

    // チャンネルで使用する通信プロトコルバージョン
    int16_t m_Version;

    ChannelState m_State;

    nn::os::SystemEvent m_ClosedEvent;
    nn::os::SystemEvent m_EstablishedEvent;

    SendPacketFactory m_SendPacketFactory;

    SendBuffer m_SendBuffer;
    ReceiveBuffer m_ReceiveBuffer;
};

}}}}
