﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/htclow/htclow_ResultPrivate.h>
#include <nn/htclow/detail/htclow_Log.h>

#include "htclow_Allocator.h"
#include "htclow_Print.h"
#include "htclow_Packet.h"
#include "htclow_PacketFactory.h"
#include "htclow_Worker.h"

namespace nn { namespace htclow { namespace server {

Worker::Worker(mux::Mux* pMux) NN_NOEXCEPT
    : m_CancelEvent(nn::os::EventClearMode_ManualClear)
    , m_pMux(pMux)
    , m_ThreadRunning(false)
{
    m_pReceiveThreadStack = AllocateThreadStack(ThreadStackSize);
    m_pSendThreadStack = AllocateThreadStack(ThreadStackSize);

    m_pMux->SetCancelEvent(&m_CancelEvent);
}

Worker::~Worker() NN_NOEXCEPT
{
    FreeThreadStack(m_pReceiveThreadStack);
    FreeThreadStack(m_pSendThreadStack);
}

void Worker::Initialize(driver::IDriver* pDriver) NN_NOEXCEPT
{
    m_pDriver = pDriver;
    m_CancelEvent.Clear();
}

Result Worker::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_ThreadRunning);

    NN_RESULT_DO(nn::os::CreateThread(&m_ReceiveThread, ReceiveThreadEntry, this, m_pReceiveThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(htc, ReceiveThread)));
    NN_RESULT_DO(nn::os::CreateThread(&m_SendThread, SendThreadEntry, this, m_pSendThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(htc, SendThread)));

    nn::os::SetThreadNamePointer(&m_ReceiveThread, NN_SYSTEM_THREAD_NAME(htc, ReceiveThread));
    nn::os::SetThreadNamePointer(&m_SendThread, NN_SYSTEM_THREAD_NAME(htc, SendThread));

    nn::os::StartThread(&m_ReceiveThread);
    nn::os::StartThread(&m_SendThread);

    m_ThreadRunning = true;

    NN_RESULT_SUCCESS;
}

void Worker::Cancel() NN_NOEXCEPT
{
    m_CancelEvent.Signal();
}

void Worker::Wait() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ThreadRunning);

    nn::os::WaitThread(&m_ReceiveThread);
    nn::os::WaitThread(&m_SendThread);

    nn::os::DestroyThread(&m_ReceiveThread);
    nn::os::DestroyThread(&m_SendThread);

    m_ThreadRunning = false;
}

void Worker::ReceiveThreadEntry(void* p) NN_NOEXCEPT
{
    reinterpret_cast<Worker*>(p)->ReceiveThread();
}

void Worker::SendThreadEntry(void* p) NN_NOEXCEPT
{
    reinterpret_cast<Worker*>(p)->SendThread();
}

void Worker::ReceiveThread() NN_NOEXCEPT
{
    const auto result = ProcessReceive();
    NN_UNUSED(result);
    NN_DETAIL_HTCLOW_INFO("Receive thread finished with result (module:%d, description:%d).\n", result.GetModule(), result.GetDescription());

    m_pDriver->Cancel();
    m_pMux->CancelAllChannel();
    m_CancelEvent.Signal();
}

void Worker::SendThread() NN_NOEXCEPT
{
    const auto result = ProcessSend();
    NN_UNUSED(result);
    NN_DETAIL_HTCLOW_INFO("Send thread finished with result (module:%d, description:%d).\n", result.GetModule(), result.GetDescription());

    m_pDriver->Cancel();
    m_pMux->CancelAllChannel();
    m_CancelEvent.Signal();
}

Result Worker::ProcessReceive() NN_NOEXCEPT
{
    for (;;)
    {
        // ヘッダ受信
        PacketHeader header;
        NN_RESULT_DO(m_pDriver->Receive(&header, sizeof(header)));

        // 通信続行不可能なヘッダのエラーを検査
        NN_RESULT_DO(CheckReceivedHeader(header));

        // パケットオブジェクト作成
        auto packet = m_ReceivePacketFactory.MakePacket(&header);

        if (!packet)
        {
            // アロケートに失敗した場合はパケットを破棄
            NN_RESULT_DO(ReceiveAndDiscard(header.bodySize));
            PrintIgnorePacket(header);
            continue;
        }

        packet->CopyHeader(&header);

        // ボディ受信
        if (header.bodySize != 0)
        {
            m_pDriver->Receive(packet->GetBody(), packet->GetBodySize());
        }

        PrintReceivePacket(header);

        // 受信したパケットを処理
        const auto result = m_pMux->ProcessReceivePacket(std::move(packet));

        // エラーが起きても、そのパケットは無視して次のパケットを処理する
        if (result.IsFailure())
        {
            PrintIgnorePacket(header);
        }
    }
}

Result Worker::ProcessSend() NN_NOEXCEPT
{
    for (;;)
    {
        SendPacket* pPacket;

        NN_RESULT_DO(m_pMux->QuerySendPacket(&pPacket));
        NN_RESULT_DO(m_pDriver->Send(pPacket->GetBuffer(), pPacket->GetBufferSize()));

        PrintSendPacket(*pPacket->GetHeader());

        // Ack パケットは、一度送信したら二度と使わないので削除
        if (IsAckPacket(pPacket->GetHeader()->packetType))
        {
            m_pMux->RemoveAckPacket(pPacket->GetHeader()->channel, pPacket->GetHeader()->sequenceId);
        }
    }
}

Result Worker::CheckReceivedHeader(const PacketHeader& header) NN_NOEXCEPT
{
    if (header.protocol != ProtocolId)
    {
        return ResultProtocolError();
    }

    if (header.version > MaxVersion)
    {
        return ResultProtocolError();
    }

    if (header.packetType == PacketType::Data)
    {
        if (header.bodySize < 0 || header.bodySize > MaxBodySize)
        {
            return ResultProtocolError();
        }
    }
    else
    {
        if (header.bodySize != 0)
        {
            return ResultProtocolError();
        }
    }

    NN_RESULT_SUCCESS;
}

Result Worker::ReceiveAndDiscard(int32_t size) NN_NOEXCEPT
{
    const int bufferSize = 256;
    char buffer[bufferSize];

    int32_t totalReceiveSize = 0;
    while (totalReceiveSize < size)
    {
        const auto receiveSize = std::min(size - totalReceiveSize, bufferSize);
        NN_RESULT_DO(m_pDriver->Receive(buffer, receiveSize));

        totalReceiveSize += receiveSize;
    }

    NN_SDK_ASSERT_EQUAL(size, totalReceiveSize);

    NN_RESULT_SUCCESS;
}

}}}
