﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/htclow/htclow_Result.h>

#include <nn/os.h>

#include "htclow_Impl.h"
#include "htclow_UserServiceObject.h"

namespace nn {
namespace htclow {
namespace server {

nn::Result UserServiceObject::Open(detail::ChannelInternalType channel) NN_NOEXCEPT
{
    return HtclowImpl::GetInstance()->Open(channel);
}

nn::Result UserServiceObject::Close(detail::ChannelInternalType channel) NN_NOEXCEPT
{
    return HtclowImpl::GetInstance()->Close(channel);
}

nn::Result UserServiceObject::Connect(detail::ChannelInternalType channel) NN_NOEXCEPT
{
    return HtclowImpl::GetInstance()->Connect(channel);
}

nn::Result UserServiceObject::Shutdown(detail::ChannelInternalType channel) NN_NOEXCEPT
{
    return HtclowImpl::GetInstance()->Shutdown(channel);
}

nn::Result UserServiceObject::Receive(nn::sf::Out<std::uint64_t> outSize, const nn::sf::OutBuffer& outBuffer, detail::ChannelInternalType channel) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(size_t) >= sizeof(int));
    if (outBuffer.GetSize() > static_cast<size_t>(std::numeric_limits<int>::max()))
    {
        return ResultOverflow();
    }

    const auto bufferSize = static_cast<int>(outBuffer.GetSize());

    int size;
    NN_RESULT_DO(HtclowImpl::GetInstance()->Receive(&size, outBuffer.GetPointerUnsafe(), bufferSize, channel));

    if (size < 0)
    {
        return ResultOverflow();
    }
    *outSize = static_cast<uint64_t>(size);

    NN_RESULT_SUCCESS;
}

nn::Result UserServiceObject::Send(nn::sf::Out<std::uint64_t> outSize, const nn::sf::InBuffer& buffer, detail::ChannelInternalType channel) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(size_t) >= sizeof(int));
    if (buffer.GetSize() > static_cast<size_t>(std::numeric_limits<int>::max()))
    {
        return ResultOverflow();
    }

    const auto bufferSize = static_cast<int>(buffer.GetSize());

    int size;
    NN_RESULT_DO(HtclowImpl::GetInstance()->Send(&size, buffer.GetPointerUnsafe(), bufferSize, channel));

    if (size < 0)
    {
        return ResultOverflow();
    }
    *outSize = static_cast<uint64_t>(size);

    NN_RESULT_SUCCESS;
}

nn::Result UserServiceObject::GetClosedEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::htclow::detail::ChannelInternalType channel) NN_NOEXCEPT
{
    nn::os::NativeHandle nativeHandle;
    const auto result = HtclowImpl::GetInstance()->GetClosedEventHandle(&nativeHandle, channel);

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    return result;
}

nn::Result UserServiceObject::GetEstablishedEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, nn::htclow::detail::ChannelInternalType channel) NN_NOEXCEPT
{
    nn::os::NativeHandle nativeHandle;
    const auto result = HtclowImpl::GetInstance()->GetEstablishedEventHandle(&nativeHandle, channel);

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    return result;
}

nn::Result UserServiceObject::GetReceiveReadyEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
{
    nn::os::NativeHandle nativeHandle;
    const auto result = HtclowImpl::GetInstance()->GetReceiveReadyEventHandle(&nativeHandle, channel);

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    return result;
}

nn::Result UserServiceObject::GetSendReadyEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
{
    nn::os::NativeHandle nativeHandle;
    const auto result = HtclowImpl::GetInstance()->GetSendReadyEventHandle(&nativeHandle, channel);

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    return result;
}

nn::Result UserServiceObject::GetSendCompleteEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
{
    nn::os::NativeHandle nativeHandle;
    const auto result = HtclowImpl::GetInstance()->GetSendCompleteEventHandle(&nativeHandle, channel);

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    return result;
}

nn::Result UserServiceObject::GetHostConnectEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    const auto nativeHandle = HtclowImpl::GetInstance()->GetConnectEvent()->GetReadableHandle();

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    NN_RESULT_SUCCESS;
}

nn::Result UserServiceObject::GetHostDisconnectEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    const auto nativeHandle = HtclowImpl::GetInstance()->GetDisconnectEvent()->GetReadableHandle();

    *outHandle = nn::sf::NativeHandle(nativeHandle, false);
    NN_RESULT_SUCCESS;
}

}}}
